<?php
/**
 * EDM E-Fatura Premium Admin Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling admin functionality
 */
class EDM_Efatura_Premium_Admin {
    /**
     * API instance
     *
     * @var EDM_Efatura_Premium_API
     */
    private $api;
    
    /**
     * Constructor
     *
     * @param EDM_Efatura_Premium_API $api API instance
     */
    public function __construct($api = null) {
		$this->api = $api;
		
		// Add AJAX handlers
		add_action('wp_ajax_edm_test_connection', array($this, 'ajax_test_connection'));
		add_action('wp_ajax_edm_get_invoices', array($this, 'ajax_get_invoices'));
		add_action('wp_ajax_edm_get_invoice_status', array($this, 'ajax_get_invoice_status'));
		add_action('wp_ajax_edm_send_invoice', array($this, 'ajax_send_invoice'));
		add_action('wp_ajax_edm_check_invoice_status', array($this, 'ajax_check_invoice_status'));
		add_action('wp_ajax_edm_generate_pdf', array($this, 'ajax_generate_pdf'));
		add_action('wp_ajax_edm_cancel_invoice', array($this, 'ajax_cancel_invoice'));
		
		add_action('wp_ajax_edm_check_user', array($this, 'ajax_check_user'));
		add_action('wp_ajax_edm_clean_logs', array($this, 'ajax_clean_logs'));
		add_action('wp_ajax_edm_delete_all_logs', array($this, 'ajax_delete_all_logs'));
		add_action('wp_ajax_edm_save_invoice_info', array($this, 'ajax_save_invoice_info'));
		add_action('wp_ajax_edm_send_invoice_email', array($this, 'ajax_send_invoice_email'));
	}
	
	/**
 * Get API instance if not already set
 * 
 * @return EDM_Efatura_Premium_API|null API instance or null
 */
private function get_api() {
    if ($this->api) {
        return $this->api;
    }
    
    // Get settings
    $settings = get_option('edm_efatura_premium_settings', array());
    
    // Check if API settings are available
    if (!empty($settings['api_url']) && !empty($settings['username']) && !empty($settings['password'])) {
        $edm_email = !empty($settings['edm_email']) ? $settings['edm_email'] : '';
        
        // Create API instance
        $this->api = new EDM_Efatura_Premium_API(
            $settings['api_url'],
            $settings['username'],
            EDM_Efatura_Premium_Security::decrypt($settings['password']),
            isset($settings['debug_mode']) && $settings['debug_mode'] === 'yes',
            $edm_email
        );
        
        return $this->api;
    }
    
    return null;
}

    /**
     * Register admin menu
     */
    public function add_admin_menu() {
    // Main menu
    add_menu_page(
        __('EDM E-Fatura Premium', 'edm-efatura-premium'),
        __('EDM E-Fatura', 'edm-efatura-premium'),
        'manage_options',
        'edm-efatura-premium',
        array($this, 'display_settings_page'),
        'dashicons-media-spreadsheet',
        30
    );
    
    // Submenus
    add_submenu_page(
        'edm-efatura-premium',
        __('Settings', 'edm-efatura-premium'),
        __('Settings', 'edm-efatura-premium'),
        'manage_options',
        'edm-efatura-premium',
        array($this, 'display_settings_page')
    );
    
    
    
    add_submenu_page(
        'edm-efatura-premium',
        __('Invoice List', 'edm-efatura-premium'),
        __('Invoice List', 'edm-efatura-premium'),
        'manage_options',
        'edm-efatura-premium-list',
        array($this, 'display_invoice_list_page')
    );
    
    add_submenu_page(
        'edm-efatura-premium',
        __('User Check', 'edm-efatura-premium'),
        __('User Check', 'edm-efatura-premium'),
        'manage_options',
        'edm-efatura-premium-users',
        array($this, 'display_user_check_page')
    );
    
    add_submenu_page(
        'edm-efatura-premium',
        __('Logs', 'edm-efatura-premium'),
        __('Logs', 'edm-efatura-premium'),
        'manage_options',
        'edm-efatura-premium-logs',
        array($this, 'display_logs_page')
    );
    
    // Updates page will be added by Updates class
}

    
	/**
     * AJAX handler for generating PDF
     */
    public function ajax_generate_pdf() {
    // Temel hata günlüğü
    $logger = new EDM_Efatura_Premium_Logger();
    $logger->debug('Generate PDF AJAX called');
    
    // Nonce kontrolü
    check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
    
    // Yetki kontrolü
    if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
        wp_send_json_error(__('You do not have permission to perform this action.', 'edm-efatura-premium'));
        return;
    }
    
    // UUID al
    $uuid = isset($_POST['uuid']) ? sanitize_text_field($_POST['uuid']) : '';
    
    if (empty($uuid)) {
        wp_send_json_error(__('Invalid UUID.', 'edm-efatura-premium'));
        return;
    }
    
    $logger->debug('Generate PDF for UUID: ' . $uuid);
    
    // Veritabanı örneği al
    $db = new EDM_Efatura_Premium_DB();
    
    // Faturayı al
    $invoice = $db->get_invoice_by_uuid($uuid);
    
    if (!$invoice) {
        wp_send_json_error(__('Invoice not found.', 'edm-efatura-premium'));
        return;
    }
    
    try {
        // Order handler sınıfını kullanarak PDF oluştur
        $order_handler = new EDM_Efatura_Premium_Order_Handler($this->api);
        $pdf_path = $order_handler->generate_invoice_pdf($invoice);
        
        if ($pdf_path) {
            // Faturayı PDF yoluyla güncelle
            $invoice['pdf_path'] = $pdf_path;
            $db->save_invoice($invoice);
            
            wp_send_json_success(__('PDF generated successfully.', 'edm-efatura-premium'));
        } else {
            wp_send_json_error(__('Failed to generate PDF.', 'edm-efatura-premium'));
        }
    } catch (Exception $e) {
        $logger->error('PDF Generation Error: ' . $e->getMessage());
        wp_send_json_error(__('Error: ', 'edm-efatura-premium') . $e->getMessage());
    }
}
	
/**
 * AJAX handler for checking invoice status
 */
public function ajax_check_invoice_status() {
    // Check nonce
    check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
    
    // Check if user has permission
    if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('view')) {
        wp_send_json_error(__('You do not have permission to perform this action.', 'edm-efatura-premium'));
        return;
    }
    
    // Get UUID
    $uuid = isset($_POST['uuid']) ? sanitize_text_field($_POST['uuid']) : '';
    
    if (empty($uuid)) {
        wp_send_json_error(__('Invalid UUID.', 'edm-efatura-premium'));
        return;
    }
    
    // Get API instance
    $api = $this->get_api();
    
    if (!$api) {
        wp_send_json_error(__('API not available. Please check your API settings.', 'edm-efatura-premium'));
        return;
    }
    
    try {
        // Get invoice status
        $status = $api->get_invoice_status($uuid);
        
        // Update database status
        $db = new EDM_Efatura_Premium_DB();
        $db->update_invoice_status($uuid, $status['status'], $status['status_description']);
        
        // Return status data
        wp_send_json_success(array(
            'status' => $status['status'],
            'description' => $status['status_description']
        ));
    } catch (Exception $e) {
        wp_send_json_error($e->getMessage());
    }
}
    /**
     * Register settings
     */
    public function register_settings() {
			register_setting('edm_efatura_premium_settings_group', 'edm_efatura_premium_settings');
		}
		
		/**
		 * Enqueue admin scripts and styles
		 *
		 * @param string $hook Current admin page
		 */
	public function enqueue_scripts($hook) {
		// Get current screen
		$screen = get_current_screen();
		
		// Check if we're on plugin pages or WooCommerce order pages
		$is_plugin_page = (strpos($hook, 'edm-efatura-premium') !== false);
		$is_order_page = ($screen && (
			$screen->id === 'shop_order' || 
			$screen->id === 'woocommerce_page_wc-orders' ||
			$screen->post_type === 'shop_order'
		));
		$is_orders_list = ($screen && (
			$screen->id === 'edit-shop_order' || 
			$screen->id === 'woocommerce_page_wc-orders'
		));
		$is_post_page = in_array($hook, array('post.php', 'post-new.php', 'edit.php'));
		
		// Only load on our pages or WooCommerce order pages
		if (!$is_plugin_page && !$is_order_page && !$is_orders_list && !$is_post_page) {
			return;
		}
		
		// Add jQuery UI for dialogs
		wp_enqueue_script('jquery-ui-dialog');
		wp_enqueue_style('wp-jquery-ui-dialog');
		
		// Add admin CSS
		wp_enqueue_style(
			'edm-efatura-premium-admin',
			EDM_EFATURA_PREMIUM_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			EDM_EFATURA_PREMIUM_VERSION
		);
		
		// Add admin JS
		wp_enqueue_script(
			'edm-efatura-premium-admin',
			EDM_EFATURA_PREMIUM_PLUGIN_URL . 'assets/js/admin.js',
			array('jquery', 'jquery-ui-dialog'),
			EDM_EFATURA_PREMIUM_VERSION,
			true
		);
		
		// Media uploader (for company logo)
		wp_enqueue_media();
		
		// Add localized script data
		wp_localize_script('edm-efatura-premium-admin', 'edmEfaturaParams', array(
			'ajaxUrl' => admin_url('admin-ajax.php'),
			'nonce' => wp_create_nonce('edm_efatura_premium_nonce'),
			'messages' => array(
				'loading' => __('Loading...', 'edm-efatura-premium'),
				'generating' => __('Generating...', 'edm-efatura-premium'),
				'confirmCancel' => __('Are you sure you want to cancel this invoice?', 'edm-efatura-premium'),
				'confirmGenerate' => __('Are you sure you want to generate an invoice for this order?', 'edm-efatura-premium'),
				'confirmClean' => __('Are you sure you want to clean old logs?', 'edm-efatura-premium'),
				'confirmSendEmail' => __('Are you sure you want to send invoice email to the customer?', 'edm-efatura-premium'),
				'error' => __('Error', 'edm-efatura-premium'),
				'success' => __('Success', 'edm-efatura-premium'),
				'statusTitle' => __('Invoice Status', 'edm-efatura-premium'),
				'showContext' => __('Show Context', 'edm-efatura-premium'),
				'hideContext' => __('Hide Context', 'edm-efatura-premium'),
				'invoiceSettings' => __('Invoice Settings', 'edm-efatura-premium'),
				'chooseInvoiceSettings' => __('Please choose invoice settings', 'edm-efatura-premium'),
				'invoiceProfile' => __('Invoice Profile', 'edm-efatura-premium'),
				'eArchiveInvoice' => __('E-Archive Invoice', 'edm-efatura-premium'),
				'basicInvoice' => __('Basic Invoice', 'edm-efatura-premium'),
				'invoiceTypeCode' => __('Invoice Type Code', 'edm-efatura-premium'),
				'sales' => __('Sales', 'edm-efatura-premium'),
				'return' => __('Return', 'edm-efatura-premium'),
				'exemption' => __('Exemption', 'edm-efatura-premium'),
				'exceptionCode' => __('Exception Code', 'edm-efatura-premium'),
				'selectExceptionCode' => __('Select Exception Code', 'edm-efatura-premium'),
				'exportGoods' => __('Export of Goods 11/1-a', 'edm-efatura-premium'),
				'exportServices' => __('Export of Services 11/1-a', 'edm-efatura-premium'),
				'roamingServices' => __('Roaming Services 11/1-a', 'edm-efatura-premium'),
				'transportVehicles' => __('Sea, Air and Rail Transport Vehicles 13/a', 'edm-efatura-premium'),
				'portAirportServices' => __('Port and Airport Services for Transport Vehicles 13/b', 'edm-efatura-premium'),
				'disabledEquipment' => __('Equipment for Disabled Persons 17/4-s', 'edm-efatura-premium'),
				'generate' => __('Generate', 'edm-efatura-premium'),
				'generateInvoice' => __('Generate Invoice', 'edm-efatura-premium'),
				'generatePdf' => __('Generate PDF', 'edm-efatura-premium'),
				'checkStatus' => __('Check Status', 'edm-efatura-premium'),
				'sendEmail' => __('Send Invoice Email', 'edm-efatura-premium'),
				'sending' => __('Sending...', 'edm-efatura-premium'),
				'dismiss' => __('Dismiss this notice.', 'edm-efatura-premium'),
				'cancel' => __('Cancel', 'edm-efatura-premium')
			)
		));
	}
    
    /**
     * Display settings page
     */
    public function display_settings_page() {
        // Include settings page template
        require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/partials/settings-page.php';
    }
    
    
    
    /**
     * Display invoice list page
     */
    public function display_invoice_list_page() {
        // Include invoice list page template
        require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/partials/invoice-list-page.php';
    }
    
    /**
     * Display user check page
     */
    public function display_user_check_page() {
        // Include user check page template
        require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/partials/user-check-page.php';
    }
    
    /**
     * Display logs page
     */
    public function display_logs_page() {
        // Include logs page template
        require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/partials/logs-page.php';
    }
    
    /**
     * Test connection AJAX handler
     */
    public function ajax_test_connection() {
		// Verify nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check permissions
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
			wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get API instance
		$api = $this->get_api();
		
		// Check if API is available
		if (!$api) {
			wp_send_json_error(__('API settings are missing or invalid. Please check your settings.', 'edm-efatura-premium'));
			return;
		}
		
		try {
			// Try to login
			$session_id = $api->login();
			
			if ($session_id) {
				$response = array(
					'success' => true,
					'message' => __('Successfully connected to EDM e-invoice system.', 'edm-efatura-premium'),
					'session_id' => $session_id
				);
				
				// Close session
				$logout_result = $api->logout($session_id);
				if ($logout_result) {
					$response['message'] .= ' ' . __('Session closed successfully.', 'edm-efatura-premium');
				}
				
				wp_send_json_success($response);
			} else {
				wp_send_json_error(__('Connection failed. Please check your settings.', 'edm-efatura-premium'));
			}
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());
		}
	}

    
    /**
     * Get invoices AJAX handler
     */
    public function ajax_get_invoices() {
		// Verify nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check permissions
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
			wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get parameters
		$start_date = isset($_POST['start_date']) ? sanitize_text_field($_POST['start_date']) : '';
		$end_date = isset($_POST['end_date']) ? sanitize_text_field($_POST['end_date']) : '';
		$direction = isset($_POST['direction']) ? sanitize_text_field($_POST['direction']) : 'IN';
		$limit = isset($_POST['limit']) ? intval($_POST['limit']) : 50;
		
		if (empty($start_date) || empty($end_date)) {
			wp_send_json_error(__('Start and end dates are required.', 'edm-efatura-premium'));
			return;
		}
		
		// Get API instance
		$api = $this->get_api();
		
		// Check if API is available
		if (!$api) {
			wp_send_json_error(__('API settings are missing or invalid. Please check your settings.', 'edm-efatura-premium'));
			return;
		}
		
		try {
			// Get invoices
			$invoices = $api->get_invoices($start_date, $end_date, $direction, $limit);
			
			// Send response
			wp_send_json_success([
				'invoices' => $invoices,
				'count' => count($invoices)
			]);
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());
			return;
		}
	}
    
    /**
     * Get invoice status AJAX handler
     */
    public function ajax_get_invoice_status() {
		// Verify nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check permissions
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
			wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get UUID parameter
		$uuid = isset($_POST['uuid']) ? sanitize_text_field($_POST['uuid']) : '';
		
		if (empty($uuid)) {
			wp_send_json_error(__('Invoice UUID is required.', 'edm-efatura-premium'));
			return;
		}
		
		// Get API instance
		$api = $this->get_api();
		
		// Check if API is available
		if (!$api) {
			wp_send_json_error(__('API settings are missing or invalid. Please check your settings.', 'edm-efatura-premium'));
			return;
		}
		
		try {
			// Get invoice status
			$status = $api->get_invoice_status($uuid);
			
			// Send response
			wp_send_json_success($status);
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());
		}
	}
    
    /**
     * Send invoice AJAX handler
     */
    public function ajax_send_invoice() {
		// Verify nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check permissions
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
			wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get parameters
		$receiver_vkn = isset($_POST['receiver_vkn']) ? sanitize_text_field($_POST['receiver_vkn']) : '';
		$receiver_alias = isset($_POST['receiver_alias']) ? sanitize_text_field($_POST['receiver_alias']) : '';
		$invoice_content = isset($_POST['invoice_content']) ? wp_kses_post(wp_unslash($_POST['invoice_content'])) : '';
		
		if (empty($receiver_vkn) || empty($receiver_alias) || empty($invoice_content)) {
			wp_send_json_error(__('All fields are required.', 'edm-efatura-premium'));
			return;
		}
		
		// Get API instance
		$api = $this->get_api();
		
		// Check if API is available
		if (!$api) {
			wp_send_json_error(__('API settings are missing or invalid. Please check your settings.', 'edm-efatura-premium'));
			return;
		}
		
		try {
			// Send invoice
			$result = $api->send_invoice($receiver_vkn, $receiver_alias, $invoice_content);
			
			// Send response
			if ($result['success']) {
				wp_send_json_success($result);
			} else {
				wp_send_json_error($result['message']);
			}
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());
		}
	}

    
    /**
     * Cancel invoice AJAX handler
     */
    public function ajax_cancel_invoice() {
    // Check nonce
    check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
    
    // Check if user has permission
    if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
        wp_send_json_error(__('You do not have permission to perform this action.', 'edm-efatura-premium'));
        return;
    }
    
    // Get UUID
    $uuid = isset($_POST['uuid']) ? sanitize_text_field($_POST['uuid']) : '';
    
    if (!$uuid) {
        wp_send_json_error(__('Invalid UUID.', 'edm-efatura-premium'));
        return;
    }
    
    // Get API instance
    $api = $this->get_api();
    
    if (!$api) {
        wp_send_json_error(__('API not available. Please check your API settings.', 'edm-efatura-premium'));
        return;
    }
    
    try {
        // Get the invoice before cancellation to retrieve the PDF path
        $db = new EDM_Efatura_Premium_DB();
        $invoice = $db->get_invoice_by_uuid($uuid);
        
        // Cancel invoice
        $result = $api->cancel_invoice($uuid);
        
        if ($result['success']) {
            // Delete the PDF file if it exists
            if ($invoice && !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path'])) {
                @unlink($invoice['pdf_path']);
                
                // Update the invoice record to remove the PDF path
                $invoice['pdf_path'] = '';
                $db->save_invoice($invoice);
            }
            
            
            // Update invoice status in database
            $db->update_invoice_status($uuid, 'CANCELLED', $result['message']);
            
            wp_send_json_success($result['message']);
        } else {
            wp_send_json_error($result['message']);
        }
    } catch (Exception $e) {
        wp_send_json_error($e->getMessage());
    }
}
    
    /**
     * Check user AJAX handler
     */
    public function ajax_check_user() {
		// Verify nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check permissions
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
			wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get VKN parameter
		$vkn = isset($_POST['vkn']) ? sanitize_text_field($_POST['vkn']) : '';
		
		if (empty($vkn)) {
			wp_send_json_error(__('VKN/TCKN is required.', 'edm-efatura-premium'));
			return;
		}
		
		// Get API instance
		$api = $this->get_api();
		
		// Check if API is available
		if (!$api) {
			wp_send_json_error(__('API settings are missing or invalid. Please check your settings.', 'edm-efatura-premium'));
			return;
		}
		
		try {
			// Check user
			$user = $api->check_user($vkn);
			
			// Debug log
			if (defined('WP_DEBUG') && WP_DEBUG) {
				$logger = new EDM_Efatura_Premium_Logger();
				$logger->debug('CheckUser response', array('user' => $user));
			}
			
			// Send response
			if (!$user || empty($user['identifier'])) {
				wp_send_json_error(__('User not found in e-invoice system or API returned invalid response.', 'edm-efatura-premium'));
				return;
			}
			
			wp_send_json_success($user);
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());
			return;
		}
	}
    
    /**
     * Clean logs AJAX handler
     */
    public function ajax_clean_logs() {
		// Verify nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check permissions
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
			wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get days parameter
		$days = isset($_POST['days']) ? intval($_POST['days']) : 30;
		
		if ($days < 1) {
			$days = 30;
		}
		
		try {
			// Clean logs
			$logger = new EDM_Efatura_Premium_Logger();
			$deleted = $logger->clean_logs($days);
			
			// Send response
			wp_send_json_success(array(
				'message' => sprintf(
					/* translators: %d: number of deleted logs */
					_n(
						'%d log entry deleted successfully.',
						'%d log entries deleted successfully.',
						$deleted,
						'edm-efatura-premium'
					),
					$deleted
				)
			));
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());
		}
	}
	
	/**
	 * Delete all logs AJAX handler
	 */
	public function ajax_delete_all_logs() {
		// Verify nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check permissions
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('admin')) {
			wp_send_json_error(__('You do not have sufficient permissions to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		try {
			// Delete all logs
			$logger = new EDM_Efatura_Premium_Logger();
			$deleted = $logger->delete_all_logs();
			
			// Send response
			wp_send_json_success(array(
				'message' => sprintf(
					/* translators: %d: number of deleted logs */
					_n(
						'%d log entry deleted successfully.',
						'%d log entries deleted successfully.',
						$deleted,
						'edm-efatura-premium'
					),
					$deleted
				)
			));
		} catch (Exception $e) {
			wp_send_json_error($e->getMessage());
		}
	}
	
	/**
	 * AJAX handler for saving invoice info
	 */
	public function ajax_save_invoice_info() {
		// Check nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check if user has permission
		if (!current_user_can('edit_shop_orders')) {
			wp_send_json_error(__('You do not have permission to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get order ID
		$order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
		
		if (!$order_id) {
			wp_send_json_error(__('Invalid order ID.', 'edm-efatura-premium'));
			return;
		}
		
		// Get order
		$order = wc_get_order($order_id);
		
		if (!$order) {
			wp_send_json_error(__('Order not found.', 'edm-efatura-premium'));
			return;
		}
		
		// Check if HPOS is enabled
		$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
		
		// Save invoice type
		if (isset($_POST['invoice_type'])) {
			$invoice_type = sanitize_text_field($_POST['invoice_type']);
			
			if ($hpos_enabled) {
				$order->update_meta_data('edm_invoice_type', $invoice_type);
			} else {
				update_post_meta($order_id, 'edm_invoice_type', $invoice_type);
			}
			
			// Save TC number for individuals
			if ($invoice_type === 'individual' && isset($_POST['tc_number'])) {
				$tc_number = sanitize_text_field($_POST['tc_number']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_tc_number', $tc_number);
				} else {
					update_post_meta($order_id, 'edm_tc_number', $tc_number);
				}
			}
			
			// Save corporate fields
			if ($invoice_type === 'corporate') {
				if (isset($_POST['company_name'])) {
					$company_name = sanitize_text_field($_POST['company_name']);
					
					if ($hpos_enabled) {
						$order->update_meta_data('edm_company_name', $company_name);
					} else {
						update_post_meta($order_id, 'edm_company_name', $company_name);
					}
				}
				
				if (isset($_POST['tax_number'])) {
					$tax_number = sanitize_text_field($_POST['tax_number']);
					
					if ($hpos_enabled) {
						$order->update_meta_data('edm_tax_number', $tax_number);
					} else {
						update_post_meta($order_id, 'edm_tax_number', $tax_number);
					}
				}
				
				if (isset($_POST['tax_office'])) {
					$tax_office = sanitize_text_field($_POST['tax_office']);
					
					if ($hpos_enabled) {
						$order->update_meta_data('edm_tax_office', $tax_office);
					} else {
						update_post_meta($order_id, 'edm_tax_office', $tax_office);
					}
				}
			}
			
			// Save invoice profile
			if (isset($_POST['invoice_profile'])) {
				$invoice_profile = sanitize_text_field($_POST['invoice_profile']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_invoice_profile', $invoice_profile);
				} else {
					update_post_meta($order_id, 'edm_invoice_profile', $invoice_profile);
				}
			}
			
			// Save invoice type code
			if (isset($_POST['invoice_type_code'])) {
				$invoice_type_code = sanitize_text_field($_POST['invoice_type_code']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_invoice_type_code', $invoice_type_code);
				} else {
					update_post_meta($order_id, 'edm_invoice_type_code', $invoice_type_code);
				}
			}
			
			// Save exception code
			if (isset($_POST['exception_code'])) {
				$exception_code = sanitize_text_field($_POST['exception_code']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_exception_code', $exception_code);
				} else {
					update_post_meta($order_id, 'edm_exception_code', $exception_code);
				}
			}
			
			// Save order
			if ($hpos_enabled) {
				$order->save();
			}
			
			wp_send_json_success(__('Invoice information saved successfully.', 'edm-efatura-premium'));
		} else {
			wp_send_json_error(__('No data to save.', 'edm-efatura-premium'));
		}
	}
	
	/**
	 * AJAX handler for sending invoice email
	 */
	public function ajax_send_invoice_email() {
		// Check nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check if user has permission
		if (!EDM_Efatura_Premium_Security::can_user_perform_invoice_operation('create')) {
			wp_send_json_error(__('You do not have permission to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get parameters
		$order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
		$uuid = isset($_POST['uuid']) ? sanitize_text_field($_POST['uuid']) : '';
		
		if (!$order_id) {
			wp_send_json_error(__('Invalid order ID.', 'edm-efatura-premium'));
			return;
		}
		
		if (empty($uuid)) {
			wp_send_json_error(__('Invalid UUID.', 'edm-efatura-premium'));
			return;
		}
		
		// Get order
		$order = wc_get_order($order_id);
		
		if (!$order) {
			wp_send_json_error(__('Order not found.', 'edm-efatura-premium'));
			return;
		}
		
		// Get invoice from database
		$db = new EDM_Efatura_Premium_DB();
		$invoice = $db->get_invoice_by_uuid($uuid);
		
		if (!$invoice) {
			wp_send_json_error(__('Invoice not found.', 'edm-efatura-premium'));
			return;
		}
		
		try {
			// Get API instance
			$api = $this->get_api();
			if (!$api) {
				wp_send_json_error(__('API not initialized.', 'edm-efatura-premium'));
				return;
			}
			
			// Use the existing order handler's direct email function
			$order_handler = new EDM_Efatura_Premium_Order_Handler($api);
			$result = $order_handler->send_invoice_email_directly($order, $invoice);
			
			if ($result) {
				// Add admin note for manual send (visible in order notes, no email)
				$order->add_order_note(
					sprintf(
						__('✅ Invoice PDF manually sent to customer by admin (%s). Invoice ID: %s', 'edm-efatura-premium'),
						$order->get_billing_email(),
						!empty($invoice['invoice_id']) ? $invoice['invoice_id'] : 'N/A'
					),
					0 // Private note - visible in order notes but no email
				);
				
				wp_send_json_success(__('Invoice email sent successfully.', 'edm-efatura-premium'));
			} else {
				wp_send_json_error(__('Failed to send invoice email.', 'edm-efatura-premium'));
			}
		} catch (Exception $e) {
			wp_send_json_error(__('Error: ', 'edm-efatura-premium') . $e->getMessage());
		}
	}
}