<?php
/**
 * EDM E-Fatura Premium Settings Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling plugin settings
 */
class EDM_Efatura_Premium_Settings {
    /**
     * Settings sections
     *
     * @var array
     */
    private $sections;
    
    /**
     * Settings fields
     *
     * @var array
     */
    private $fields;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->sections = array(
            'api' => __('API Settings', 'edm-efatura-premium'),
            'company' => __('Company Information', 'edm-efatura-premium'),
            'invoice' => __('Invoice Settings', 'edm-efatura-premium'),
            'woocommerce' => __('WooCommerce Integration', 'edm-efatura-premium'),
            'logs' => __('Logs', 'edm-efatura-premium')
        );
        
        $this->fields = $this->get_settings_fields();
        
        add_action('admin_init', array($this, 'register_settings'));
    }
    
    /**
	 * Register settings sections and fields
	 */
	public function register_settings() {
		register_setting(
			'edm_efatura_premium_settings_group',
			'edm_efatura_premium_settings',
			array($this, 'validate_settings')
		);
		
		// Register sections
		foreach ($this->sections as $id => $title) {
			add_settings_section(
				'edm_efatura_premium_' . $id . '_section',
				$title,
				array($this, 'section_callback'),
				'edm_efatura_premium_settings'
			);
		}
		
		// Register fields
		foreach ($this->fields as $id => $field) {
			add_settings_field(
				$id,
				$field['label'],
				array($this, 'field_callback'),
				'edm_efatura_premium_settings',
				'edm_efatura_premium_' . $field['section'] . '_section',
				array(
					'id' => $id,
					'field' => $field
				)
			);
		}
	}
    
    /**
     * Section callback
     *
     * @param array $args Section arguments
     */
    public function section_callback($args) {
        $section_id = str_replace('edm_efatura_premium_', '', str_replace('_section', '', $args['id']));
        
        switch ($section_id) {
            case 'api':
                echo '<p>' . __('Configure your EDM e-invoice API credentials.', 'edm-efatura-premium') . '</p>';
                break;
                
            case 'company':
                echo '<p>' . __('Enter your company information to be used in invoices.', 'edm-efatura-premium') . '</p>';
                break;
                
            case 'invoice':
                echo '<p>' . __('Configure general invoice settings.', 'edm-efatura-premium') . '</p>';
                break;
                
            case 'woocommerce':
                echo '<p>' . __('Configure WooCommerce integration settings.', 'edm-efatura-premium') . '</p>';
                break;
                
            case 'pdf':
                echo '<p>' . __('Configure PDF invoice settings.', 'edm-efatura-premium') . '</p>';
                break;
                
            case 'logs':
                echo '<p>' . __('Configure logging settings.', 'edm-efatura-premium') . '</p>';
                break;
        }
    }
    
    /**
     * Field callback
     *
     * @param array $args Field arguments
     */
    public function field_callback($args) {
        $id = $args['id'];
        $field = $args['field'];
        $settings = get_option('edm_efatura_premium_settings', array());
        $value = isset($settings[$id]) ? $settings[$id] : '';
        
        $placeholder = isset($field['placeholder']) ? $field['placeholder'] : '';
        $description = isset($field['description']) ? $field['description'] : '';
        $options = isset($field['options']) ? $field['options'] : array();
        
        switch ($field['type']) {
            case 'text':
                printf(
                    '<input type="text" id="%1$s" name="edm_efatura_premium_settings[%1$s]" value="%2$s" placeholder="%3$s" class="regular-text" />',
                    esc_attr($id),
                    esc_attr($value),
                    esc_attr($placeholder)
                );
                break;
                
            case 'password':
                // Decrypt password for display (but keep it encrypted in database)
                $decrypted_value = !empty($value) ? EDM_Efatura_Premium_Security::decrypt($value) : '';
                printf(
                    '<input type="password" id="%1$s" name="edm_efatura_premium_settings[%1$s]" value="%2$s" placeholder="%3$s" class="regular-text" />',
                    esc_attr($id),
                    esc_attr($decrypted_value),
                    esc_attr($placeholder)
                );
                break;
                
            case 'textarea':
                printf(
                    '<textarea id="%1$s" name="edm_efatura_premium_settings[%1$s]" placeholder="%3$s" rows="5" cols="50">%2$s</textarea>',
                    esc_attr($id),
                    esc_textarea($value),
                    esc_attr($placeholder)
                );
                break;
                
            case 'checkbox':
                printf(
                    '<input type="checkbox" id="%1$s" name="edm_efatura_premium_settings[%1$s]" value="yes" %2$s />',
                    esc_attr($id),
                    checked($value, 'yes', false)
                );
                break;
                
            case 'radio':
                foreach ($options as $option_value => $option_label) {
                    printf(
                        '<label><input type="radio" id="%1$s_%2$s" name="edm_efatura_premium_settings[%1$s]" value="%2$s" %3$s /> %4$s</label><br />',
                        esc_attr($id),
                        esc_attr($option_value),
                        checked($value, $option_value, false),
                        esc_html($option_label)
                    );
                }
                break;
                
            case 'select':
                printf('<select id="%1$s" name="edm_efatura_premium_settings[%1$s]">', esc_attr($id));
                
                foreach ($options as $option_value => $option_label) {
                    printf(
                        '<option value="%1$s" %2$s>%3$s</option>',
                        esc_attr($option_value),
                        selected($value, $option_value, false),
                        esc_html($option_label)
                    );
                }
                
                echo '</select>';
                break;
                
            case 'image':
                $image_url = '';
                $image_id = $value;
                
                if ($image_id) {
                    $image_url = wp_get_attachment_url($image_id);
                }
                
                echo '<div class="edm-image-upload">';
                printf(
                    '<input type="hidden" id="%1$s" name="edm_efatura_premium_settings[%1$s]" value="%2$s" />',
                    esc_attr($id),
                    esc_attr($value)
                );
                
                echo '<div class="edm-image-preview">';
                if ($image_url) {
                    printf('<img src="%s" alt="" />', esc_url($image_url));
                }
                echo '</div>';
                
                printf(
                    '<input type="button" class="button edm-upload-image" data-field="%1$s" value="%2$s" />',
                    esc_attr($id),
                    esc_attr__('Upload Image', 'edm-efatura-premium')
                );
                
                printf(
                    '<input type="button" class="button edm-remove-image" data-field="%1$s" value="%2$s" %3$s />',
                    esc_attr($id),
                    esc_attr__('Remove Image', 'edm-efatura-premium'),
                    empty($image_url) ? 'style="display:none;"' : ''
                );
                
                echo '</div>';
                break;
        }
        
        if ($description) {
            printf('<p class="description">%s</p>', $description);
        }
    }
    
    /**
     * Validate settings
     *
     * @param array $input Settings input
     * @return array Validated settings
     */
    public function validate_settings($input) {
		$output = array();
		$required_fields = array(
			'company_prefix' => __('Company Code', 'edm-efatura-premium'),
			'company_vkn' => __('Tax Number (VKN/TCKN)', 'edm-efatura-premium'),
			'company_name' => __('Company Name', 'edm-efatura-premium'),
			'company_tax_office' => __('Tax Office', 'edm-efatura-premium'),
			'company_street' => __('Street', 'edm-efatura-premium'),
			'company_city' => __('City', 'edm-efatura-premium'),
			'company_postal' => __('Postal Code', 'edm-efatura-premium'),
			'company_phone' => __('Phone', 'edm-efatura-premium'),
			'company_email' => __('Email', 'edm-efatura-premium'),
		);
		
		// Add validation for invoice type specific required fields
		if (isset($input['invoice_type']) && $input['invoice_type'] === 'ISTISNA') {
			$required_fields['invoice_exception_code'] = __('Exception Code', 'edm-efatura-premium');
		}
		
		$errors = array();
		
		// Check required fields
		foreach ($required_fields as $field => $label) {
			if (empty($input[$field])) {
				$errors[] = sprintf(__('%s is required.', 'edm-efatura-premium'), $label);
				add_settings_error(
					'edm_efatura_premium_settings',
					'required_' . $field,
					sprintf(__('%s is required.', 'edm-efatura-premium'), $label),
					'error'
				);
			}
		}
		
		// If there are errors, return the old options
		if (!empty($errors)) {
			add_settings_error(
				'edm_efatura_premium_settings',
				'settings_updated',
				__('Settings were not saved. Please fix the errors below.', 'edm-efatura-premium'),
				'error'
			);
			
			return get_option('edm_efatura_premium_settings');
		}
		
		// Validate each field
		foreach ($this->fields as $id => $field) {
			if (isset($input[$id])) {
				switch ($field['type']) {
					case 'text':
					case 'textarea':
					case 'select':
					case 'radio':
					case 'image':
						$output[$id] = sanitize_text_field($input[$id]);
						break;
						
					case 'password':
						$output[$id] = EDM_Efatura_Premium_Security::encrypt(sanitize_text_field($input[$id]));
						break;
						
					case 'checkbox':
						$output[$id] = ($input[$id] === 'yes') ? 'yes' : 'no';
						break;
				}
			} else {
				// For checkboxes that are unchecked
				if ($field['type'] === 'checkbox') {
					$output[$id] = 'no';
				}
			}
		}
		
		// Special validations
		
		// Company prefix validation (3 letters max)
		if (!empty($output['company_prefix'])) {
			$output['company_prefix'] = substr(strtoupper($output['company_prefix']), 0, 3);
		}
		
		// VKN/TCKN validation
		if (!empty($output['company_vkn'])) {
			// Only allow digits
			$output['company_vkn'] = preg_replace('/[^0-9]/', '', $output['company_vkn']);
			
			// Check if valid length
			if (strlen($output['company_vkn']) != 10 && strlen($output['company_vkn']) != 11) {
				add_settings_error(
					'edm_efatura_premium_settings',
					'vkn_length',
					__('Tax Number must be 10 digits (VKN) or 11 digits (TCKN).', 'edm-efatura-premium'),
					'error'
				);
			}
		}
		
		if (isset($input['invoice_type']) && $input['invoice_type'] === 'SATIS') {
			if (empty($input['default_tax_rate'])) {
				$errors[] = sprintf(__('Default Tax Rate is required when invoice type is Sales.', 'edm-efatura-premium'));
				add_settings_error(
					'edm_efatura_premium_settings',
					'required_default_tax_rate',
					sprintf(__('Default Tax Rate is required when invoice type is Sales.', 'edm-efatura-premium')),
					'error'
				);
			} elseif (!is_numeric($input['default_tax_rate']) || floatval($input['default_tax_rate']) <= 0) {
				$errors[] = sprintf(__('Default Tax Rate must be a number greater than 0.', 'edm-efatura-premium'));
				add_settings_error(
					'edm_efatura_premium_settings',
					'invalid_default_tax_rate',
					sprintf(__('Default Tax Rate must be a number greater than 0.', 'edm-efatura-premium')),
					'error'
				);
			}
		}
		
		return $output;
	}
    
    /**
     * Get settings fields
     *
     * @return array Settings fields
     */
    private function get_settings_fields() {
        return array(
            // API Settings
			'api_url' => array(
				'label' => __('EDM API URL', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'api',
				'placeholder' => 'https://portal2.edmbilisim.com.tr/EFaturaEDM/EFaturaEDM.svc',
				'description' => __('EDM e-invoice API address. Test environment: https://test.edmbilisim.com.tr/EFaturaEDM21ea/EFaturaEDM.svc', 'edm-efatura-premium')
			),
			'username' => array(
				'label' => __('Username', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'api',
				'placeholder' => __('Your EDM username', 'edm-efatura-premium'),
				'description' => __('Your EDM e-invoice system username.', 'edm-efatura-premium')
			),
			'password' => array(
				'label' => __('Password', 'edm-efatura-premium'),
				'type' => 'password',
				'section' => 'api',
				'placeholder' => __('Your EDM password', 'edm-efatura-premium'),
				'description' => __('Your EDM e-invoice system password.', 'edm-efatura-premium')
			),
			// API Settings - Secret Key yerine EDM Email ekle
			'edm_email' => array(
				'label' => __('EDM Email', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'api',
				'placeholder' => __('example@edmbilisim.com', 'edm-efatura-premium'),
				'description' => __('Your EDM email address (e.g. username@edmbilisim.com) that matches your GIB label.', 'edm-efatura-premium')
			),
			'debug_mode' => array(
				'label' => __('Debug Mode', 'edm-efatura-premium'),
				'type' => 'checkbox',
				'section' => 'api',
				'description' => __('Enable debug mode to log detailed API requests and responses.', 'edm-efatura-premium')
			),
            
            // Company Information
            'company_name' => array(
                'label' => __('Company Name', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Your company name', 'edm-efatura-premium'),
                'description' => __('Your company name to be used in invoices.', 'edm-efatura-premium')
            ),
            'company_vkn' => array(
                'label' => __('Tax Number (VKN)', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('10 digits tax number', 'edm-efatura-premium'),
                'description' => __('Your company tax number.', 'edm-efatura-premium')
            ),
            'company_tax_office' => array(
                'label' => __('Tax Office', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Your tax office', 'edm-efatura-premium'),
                'description' => __('Your company tax office.', 'edm-efatura-premium')
            ),
            'company_street' => array(
                'label' => __('Street', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Street address', 'edm-efatura-premium'),
                'description' => __('Your company street address.', 'edm-efatura-premium')
            ),
            'company_building' => array(
                'label' => __('Building Number', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Building number', 'edm-efatura-premium'),
                'description' => __('Your company building number.', 'edm-efatura-premium')
            ),
            'company_district' => array(
                'label' => __('District', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('District', 'edm-efatura-premium'),
                'description' => __('Your company district.', 'edm-efatura-premium')
            ),
            'company_city' => array(
                'label' => __('City', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('City', 'edm-efatura-premium'),
                'description' => __('Your company city.', 'edm-efatura-premium')
            ),
			'company_region' => array(
				'label' => __('Region', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'company',
				'placeholder' => __('Marmara', 'edm-efatura-premium'),
				'description' => __('Your company region (e.g., Marmara, İç Anadolu, etc.).', 'edm-efatura-premium')
			),

			'company_subdistrict' => array(
				'label' => __('Sub-district', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'company',
				'placeholder' => __('Merkez', 'edm-efatura-premium'),
				'description' => __('Your company sub-district.', 'edm-efatura-premium')
			),
			
			'company_country_code' => array(
				'label' => __('Country Code', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'company',
				'placeholder' => 'TR',
				'description' => __('Your company country code (e.g., TR for Turkey).', 'edm-efatura-premium')
			),

			'company_country' => array(
				'label' => __('Country', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'company',
				'placeholder' => 'Türkiye',
				'description' => __('Your company country.', 'edm-efatura-premium')
			),
            'company_postal' => array(
                'label' => __('Postal Code', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Postal code', 'edm-efatura-premium'),
                'description' => __('Your company postal code.', 'edm-efatura-premium')
            ),
            'company_phone' => array(
                'label' => __('Phone', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Phone number', 'edm-efatura-premium'),
                'description' => __('Your company phone number.', 'edm-efatura-premium')
            ),
            'company_fax' => array(
                'label' => __('Fax', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Fax number', 'edm-efatura-premium'),
                'description' => __('Your company fax number.', 'edm-efatura-premium')
            ),
            'company_email' => array(
                'label' => __('Email', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Email address', 'edm-efatura-premium'),
                'description' => __('Your company email address.', 'edm-efatura-premium')
            ),
            'company_website' => array(
                'label' => __('Website', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'company',
                'placeholder' => __('Website URL', 'edm-efatura-premium'),
                'description' => __('Your company website URL.', 'edm-efatura-premium')
            ),
			// Ödeme Aracısı
			'payment_intermediary' => array(
				'label' => __('Payment Intermediary', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'invoice',
				'placeholder' => 'sanal',
				'description' => __('Payment intermediary name to be used in invoices. Default: sanal', 'edm-efatura-premium')
			),

			// Kargo Şirket Adı
			'shipping_company_name' => array(
				'label' => __('Shipping Company Name', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'invoice',
				'placeholder' => 'Kargo',
				'description' => __('Shipping company name to be used in invoices. Default: Kargo', 'edm-efatura-premium')
			),

			// Ticaret Sicil Numarası
			'company_trade_registry' => array(
				'label' => __('Trade Registry Number', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'company',
				'placeholder' => __('Trade registry number', 'edm-efatura-premium'),
				'description' => __('Your company trade registry number.', 'edm-efatura-premium')
			),

			// Mersis No
			'company_mersis' => array(
				'label' => __('MERSIS Number', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'company',
				'placeholder' => __('MERSIS number', 'edm-efatura-premium'),
				'description' => __('Your company MERSIS number.', 'edm-efatura-premium')
			),
            'company_logo' => array(
                'label' => __('Company Logo', 'edm-efatura-premium'),
                'type' => 'image',
                'section' => 'company',
                'description' => __('Upload your company logo to be used in invoices.', 'edm-efatura-premium')
            ),
            
            // Invoice Settings
            'invoice_type' => array(
				'label' => __('Default Invoice Type', 'edm-efatura-premium'),
				'type' => 'select',
				'section' => 'invoice',
				'options' => array(
					'SATIS' => __('Sales Invoice', 'edm-efatura-premium'),
					'IADE' => __('Return Invoice', 'edm-efatura-premium'),
					'ISTISNA' => __('Exemption Invoice', 'edm-efatura-premium')
					
				),
				'description' => __('Default invoice type to be used.', 'edm-efatura-premium')
			),
			// İstisna kodları listesini güncelleme
			'invoice_exception_code' => array(
				'label' => __('İstisna Kodu', 'edm-efatura-premium'),
				'type' => 'select',
				'section' => 'invoice',
				'options' => EDM_EFatura_Premium_Exception_Codes::get_exception_codes_for_dropdown(),
				'description' => __('İstisna faturalar için istisna kodunu seçin.', 'edm-efatura-premium')
			),
			'default_tax_rate' => array(
				'label' => __('Default Tax Rate (%)', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'invoice',
				'placeholder' => '20',
				'description' => __('Default tax rate (%) to be used for Sales invoices. Must be greater than 0.', 'edm-efatura-premium')
			),
			// WooCommerce prices include tax or not
			'woocommerce_prices_include_tax' => array(
				'label' => __('WooCommerce Prices', 'edm-efatura-premium'),
				'type' => 'select',
				'section' => 'invoice',
				'options' => array(
					'yes' => __('Include Tax', 'edm-efatura-premium'),
					'no' => __('Exclude Tax', 'edm-efatura-premium')
				),
				'description' => __('Select whether your WooCommerce product prices include tax or not. This affects how invoice prices are calculated.', 'edm-efatura-premium')
			),
			// Birim kodları güncelleme
			'default_unit_code' => array(
				'label' => __('Default Unit Code', 'edm-efatura-premium'),
				'type' => 'select',
				'section' => 'invoice',
				'options' => array(
					'C62' => 'C62 - ' . __('Unit/Piece', 'edm-efatura-premium'),
					'KGM' => 'KGM - ' . __('Kilogram', 'edm-efatura-premium'),
					'LTR' => 'LTR - ' . __('Liter', 'edm-efatura-premium'),
					'MTR' => 'MTR - ' . __('Meter', 'edm-efatura-premium'),
					'MTK' => 'MTK - ' . __('Square Meter', 'edm-efatura-premium'),
					'MTQ' => 'MTQ - ' . __('Cubic Meter', 'edm-efatura-premium'),
					'GRM' => 'GRM - ' . __('Gram', 'edm-efatura-premium'),
					'KWH' => 'KWH - ' . __('Kilowatt Hour', 'edm-efatura-premium'),
					'B32' => 'B32 - ' . __('KG-Square Meter', 'edm-efatura-premium'),
					'CCT' => 'CCT - ' . __('Carrying Capacity per Ton', 'edm-efatura-premium'),
					'PR' => 'PR - ' . __('Pair', 'edm-efatura-premium'),
					'D30' => 'D30 - ' . __('Gross Calorie Value', 'edm-efatura-premium'),
					'D40' => 'D40 - ' . __('Thousand Liters', 'edm-efatura-premium'),
					'GFI' => 'GFI - ' . __('Gram of Fissile Isotope', 'edm-efatura-premium'),
					'GT' => 'GT - ' . __('Gross Ton', 'edm-efatura-premium'),
					'CEN' => 'CEN - ' . __('Hundred Units', 'edm-efatura-premium'),
					'KPO' => 'KPO - ' . __('Kilogram Potassium Oxide', 'edm-efatura-premium'),
					'MND' => 'MND - ' . __('Dried Net Weight Kilogram', 'edm-efatura-premium'),
					'3I' => '3I - ' . __('Kilogram-Piece', 'edm-efatura-premium'),
					'KFO' => 'KFO - ' . __('Kilogram Diphosphorus Pentoxide', 'edm-efatura-premium'),
					'KHY' => 'KHY - ' . __('Kilogram Hydrogen Peroxide', 'edm-efatura-premium'),
					'KMA' => 'KMA - ' . __('Kilogram Methyl Amines', 'edm-efatura-premium'),
					'KNI' => 'KNI - ' . __('Kilogram of Nitrogen', 'edm-efatura-premium'),
					'KPH' => 'KPH - ' . __('Kilogram Potassium Hydroxide', 'edm-efatura-premium'),
					'KSD' => 'KSD - ' . __('Kilogram 90% Dry Product', 'edm-efatura-premium'),
					'KSH' => 'KSH - ' . __('Kilogram Sodium Hydroxide', 'edm-efatura-premium'),
					'KUR' => 'KUR - ' . __('Kilogram Uranium', 'edm-efatura-premium'),
					'D32' => 'D32 - ' . __('Terawatt Hour', 'edm-efatura-premium'),
					'GWH' => 'GWH - ' . __('Gigawatt Hour', 'edm-efatura-premium'),
					'MWH' => 'MWH - ' . __('Megawatt Hour', 'edm-efatura-premium'),
					'KWT' => 'KWT - ' . __('Kilowatt', 'edm-efatura-premium'),
					'LPA' => 'LPA - ' . __('Liter Pure Alcohol', 'edm-efatura-premium'),
					'DMK' => 'DMK - ' . __('Square Decimeter', 'edm-efatura-premium'),
					'NCL' => 'NCL - ' . __('Number of Cells', 'edm-efatura-premium'),
					'CTM' => 'CTM - ' . __('Carat', 'edm-efatura-premium'),
					'SM3' => 'SM3 - ' . __('Standard Cubic Meter', 'edm-efatura-premium'),
					'R9' => 'R9 - ' . __('Thousand Cubic Meters', 'edm-efatura-premium'),
					'SET' => 'SET - ' . __('Set', 'edm-efatura-premium'),
					'T3' => 'T3 - ' . __('Thousand Units', 'edm-efatura-premium')
				),
				'description' => __('Default unit code to be used for products without specified units.', 'edm-efatura-premium')
			),
			
            // Profil ID güncelleme
			'invoice_profile' => array(
				'label' => __('Invoice Profile', 'edm-efatura-premium'),
				'type' => 'select',
				'section' => 'invoice',
				'options' => array(
					'TEMELFATURA' => __('Basic Invoice', 'edm-efatura-premium'),
					
					'EARSIVFATURA' => __('E-Archive Invoice', 'edm-efatura-premium')
					
				),
				'description' => __('Invoice profile to be used.', 'edm-efatura-premium')
			),
			// Şirket öneki (prefix) ekleme
			'company_prefix' => array(
				'label' => __('Company Code', 'edm-efatura-premium'),
				'type' => 'text',
				'section' => 'company',
				'placeholder' => 'MKZ',
				'description' => __('Company code prefix (3 letters max) used for invoice numbering.', 'edm-efatura-premium')
			),
            'default_tc' => array(
                'label' => __('Default TC Number', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'invoice',
                'placeholder' => '11111111111',
                'description' => __('Default TC number to be used for individual customers who do not provide their own TC number.', 'edm-efatura-premium')
            ),
            
            // WooCommerce Integration
            'woocommerce_enabled' => array(
                'label' => __('Enable WooCommerce Integration', 'edm-efatura-premium'),
                'type' => 'checkbox',
                'section' => 'woocommerce',
                'description' => __('Enable integration with WooCommerce.', 'edm-efatura-premium')
            ),
            'auto_generate_invoice' => array(
                'label' => __('Auto-Generate Invoice', 'edm-efatura-premium'),
                'type' => 'checkbox',
                'section' => 'woocommerce',
                'description' => __('Automatically generate invoice when order is completed.', 'edm-efatura-premium')
            ),
            
            'attach_pdf_to_email' => array(
                'label' => __('Attach Invoice to Order Email', 'edm-efatura-premium'),
                'type' => 'checkbox',
                'section' => 'woocommerce',
                'description' => __('Attach PDF invoice to completed order email.', 'edm-efatura-premium')
            ),
            'order_status_for_invoice' => array(
                'label' => __('Order Status for Invoice', 'edm-efatura-premium'),
                'type' => 'select',
                'section' => 'woocommerce',
                'options' => array(
                    'processing' => __('Processing', 'edm-efatura-premium'),
                    'completed' => __('Completed', 'edm-efatura-premium'),
                    'on-hold' => __('On Hold', 'edm-efatura-premium'),
                    'paid' => __('Paid', 'edm-efatura-premium')
                ),
                'description' => __('Order status when invoice should be generated.', 'edm-efatura-premium')
            ),
			'invoice_delay_days' => array(
				'label' => __('Invoice Generation Delay (minutes)', 'edm-efatura-premium'), // Changed from "days" to "minutes"
				'type' => 'text',
				'section' => 'woocommerce',
				'placeholder' => '0',
				'description' => __('Number of minutes to wait after order completion before generating the invoice. Set to 0 for immediate generation.', 'edm-efatura-premium') // Changed description accordingly
			),
            
            
            
            // Logs Settings
            'log_level' => array(
                'label' => __('Log Level', 'edm-efatura-premium'),
                'type' => 'select',
                'section' => 'logs',
                'options' => array(
                    'emergency' => __('Emergency', 'edm-efatura-premium'),
                    'alert' => __('Alert', 'edm-efatura-premium'),
                    'critical' => __('Critical', 'edm-efatura-premium'),
                    'error' => __('Error', 'edm-efatura-premium'),
                    'warning' => __('Warning', 'edm-efatura-premium'),
                    'notice' => __('Notice', 'edm-efatura-premium'),
                    'info' => __('Info', 'edm-efatura-premium'),
                    'debug' => __('Debug', 'edm-efatura-premium')
                ),
                'description' => __('Log level.', 'edm-efatura-premium')
            ),
            'log_retention' => array(
                'label' => __('Log Retention (days)', 'edm-efatura-premium'),
                'type' => 'text',
                'section' => 'logs',
                'placeholder' => '30',
                'description' => __('Number of days to keep logs.', 'edm-efatura-premium')
            )
        );
    }
    
    
    
    
    

}