<?php
/**
 * EDM E-Fatura Premium Updates Admin Page
 * 
 * Handles plugin update management and display
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class EDM_Efatura_Premium_Updates {
    
    /**
     * Update checker instance
     */
    private $update_checker;
    
    /**
     * Logger instance
     */
    private $logger;
    
    /**
     * Constructor
     */
    public function __construct($update_checker) {
        $this->update_checker = $update_checker;
        $this->logger = new EDM_Efatura_Premium_Logger();
        
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'), 20); // Run after main admin menu
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_edm_check_updates', array($this, 'ajax_check_updates'));
        add_action('wp_ajax_edm_dismiss_update', array($this, 'ajax_dismiss_update'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Add submenu under EDM E-Fatura Premium
        add_submenu_page(
            'edm-efatura-premium', // Parent slug
            __('Plugin Updates', 'edm-efatura-premium'),
            __('Updates', 'edm-efatura-premium'),
            'manage_options',
            'edm-efatura-premium-updates',
            array($this, 'updates_page')
        );
    }
    
    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_scripts($hook) {
        // Only load on our updates page
        if ($hook !== 'edm-e-fatura_page_edm-efatura-premium-updates') {
            return;
        }
        
        wp_enqueue_script('jquery');
        
        // Add inline styles
        $css = '
        .edm-updates-wrap {
            max-width: 1200px;
            margin: 20px 0;
        }
        .edm-update-card {
            background: #fff;
            border: 1px solid #c3c4c7;
            border-radius: 4px;
            padding: 20px;
            margin-bottom: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,.04);
        }
        .edm-update-header {
            display: flex;
            align-items: center;
            margin-bottom: 15px;
        }
        .edm-update-icon {
            font-size: 32px;
            margin-right: 15px;
        }
        .edm-update-icon.available { color: #d63384; }
        .edm-update-icon.up-to-date { color: #198754; }
        .edm-update-icon.checking { color: #0d6efd; }
        .edm-update-title {
            font-size: 18px;
            font-weight: 600;
            margin: 0;
        }
        .edm-update-subtitle {
            color: #646970;
            margin: 5px 0 0 0;
        }
        .edm-update-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
        }
        .edm-changelog {
            background: #f9f9f9;
            border: 1px solid #e1e1e1;
            border-radius: 4px;
            padding: 15px;
            margin-top: 15px;
        }
        .edm-version-comparison {
            display: flex;
            gap: 20px;
            margin: 10px 0;
        }
        .edm-version-box {
            padding: 10px 15px;
            border-radius: 4px;
            text-align: center;
            min-width: 120px;
        }
        .edm-version-current {
            background: #fff3cd;
            border: 1px solid #ffecb5;
            color: #664d03;
        }
        .edm-version-latest {
            background: #d1e7dd;
            border: 1px solid #badbcc;
            color: #0a3622;
        }
        .edm-priority-badge {
            display: inline-block;
            padding: 4px 8px;
            border-radius: 3px;
            font-size: 11px;
            font-weight: bold;
            text-transform: uppercase;
            margin-left: 10px;
        }
        .edm-priority-security {
            background: #dc3545;
            color: white;
        }
        .edm-priority-critical {
            background: #ffc107;
            color: black;
        }
        .edm-last-check {
            color: #646970;
            font-size: 12px;
            margin-top: 10px;
        }
        @keyframes edm-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .edm-button-spinner {
            animation: edm-spin 1s linear infinite;
        }
        .edm-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        .edm-stat-box {
            background: #fff;
            border: 1px solid #c3c4c7;
            border-radius: 4px;
            padding: 15px;
            text-align: center;
        }
        .edm-stat-number {
            font-size: 24px;
            font-weight: bold;
            color: #2271b1;
        }
        .edm-stat-label {
            color: #646970;
            font-size: 12px;
            text-transform: uppercase;
            margin-top: 5px;
        }
        ';
        
        wp_add_inline_style('wp-admin', $css);
        
        // Add JavaScript functionality
        $js = '
        jQuery(document).ready(function($) {
            // Check for updates button
            $("#edm-check-updates").click(function(e) {
                e.preventDefault();
                
                var $btn = $(this);
                var originalText = $btn.text();
                
                // Show loading state
                $btn.html("<span class=\"dashicons dashicons-update edm-button-spinner\"></span> ' . esc_js(__('Checking...', 'edm-efatura-premium')) . '").prop("disabled", true);
                
                $.post(ajaxurl, {
                    action: "edm_check_updates",
                    nonce: "' . wp_create_nonce('edm_check_updates') . '"
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        var errorMsg = response.data && response.data.message ? response.data.message : "Unknown error";
                        alert("' . esc_js(__('Update check failed:', 'edm-efatura-premium')) . ' " + errorMsg);
                        $btn.html("<span class=\"dashicons dashicons-update\"></span> " + originalText).prop("disabled", false);
                    }
                }).fail(function() {
                    alert("' . esc_js(__('Connection error. Please try again.', 'edm-efatura-premium')) . '");
                    $btn.html("<span class=\"dashicons dashicons-update\"></span> " + originalText).prop("disabled", false);
                });
            });
            
            // Dismiss update button
            $("#edm-dismiss-update").click(function(e) {
                e.preventDefault();
                
                if (!confirm("' . esc_js(__('Are you sure you want to dismiss this update notification for one week?', 'edm-efatura-premium')) . '")) {
                    return;
                }
                
                var $btn = $(this);
                
                $.post(ajaxurl, {
                    action: "edm_dismiss_update",
                    nonce: "' . wp_create_nonce('edm_dismiss_update') . '"
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        var errorMsg = response.data && response.data.message ? response.data.message : "Unknown error";
                        alert("' . esc_js(__('Failed to dismiss update notification:', 'edm-efatura-premium')) . ' " + errorMsg);
                    }
                }).fail(function() {
                    alert("' . esc_js(__('Connection error. Please try again.', 'edm-efatura-premium')) . '");
                });
            });
            
            // Auto-refresh every 5 minutes
            setTimeout(function() {
                $("#edm-check-updates").trigger("click");
            }, 300000); // 5 minutes
        });
        ';
        
        wp_add_inline_script('jquery', $js);
    }
    
    /**
     * Display updates page
     */
    public function updates_page() {
        // Get current update status
        $status = $this->update_checker->get_update_status();
        $last_check = get_transient('edm_update_data_timestamp');
        $dismissed_until = get_option('edm_update_dismissed');
        
        include EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/partials/updates-page.php';
    }
    
    /**
     * AJAX handler for checking updates
     */
    public function ajax_check_updates() {
        if (!wp_verify_nonce($_POST['nonce'], 'edm_check_updates') || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Force check for updates
        $result = $this->update_checker->force_check();
        
        if ($result === false) {
            wp_send_json_error(array('message' => 'Update check failed. Please try again later.'));
        }
        
        // Store check timestamp
        set_transient('edm_update_data_timestamp', time(), WEEK_IN_SECONDS);
        
        wp_send_json_success($result);
    }
    
    /**
     * AJAX handler for dismissing updates
     */
    public function ajax_dismiss_update() {
        if (!wp_verify_nonce($_POST['nonce'], 'edm_dismiss_update') || !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Security check failed'));
        }
        
        // Dismiss for 1 week
        update_option('edm_update_dismissed', time() + WEEK_IN_SECONDS);
        
        $this->logger->info('Update notification dismissed by user from updates page');
        
        wp_send_json_success();
    }
    
    /**
     * Get update statistics
     */
    public function get_update_stats() {
        $stats = array(
            'current_version' => EDM_EFATURA_PREMIUM_VERSION,
            'last_check' => get_transient('edm_update_data_timestamp'),
            'check_interval' => '5 minutes', // Since we changed it to 5 minutes
            'api_endpoint' => 'wcplugin.com'
        );
        
        return $stats;
    }
}