<?php
/**
 * Logs page template
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Get logger instance
$logger = new EDM_Efatura_Premium_Logger();

// Set default filter values
$page = isset($_GET['paged']) ? intval($_GET['paged']) : 1;
$per_page = isset($_GET['per_page']) ? intval($_GET['per_page']) : 50;
$level = isset($_GET['level']) ? sanitize_text_field($_GET['level']) : '';

// Get logs with pagination
$result = $logger->get_logs($page, $per_page, $level);
$logs = $result['logs'];
$total = $result['total'];
$total_pages = $result['pages'];

// Get settings
$settings = get_option('edm_efatura_premium_settings', array());
$log_retention = isset($settings['log_retention']) ? intval($settings['log_retention']) : 30;
?>

<div class="wrap">
    <h1><?php echo esc_html__('Logs', 'edm-efatura-premium'); ?></h1>
    
    <div class="edm-log-filters">
        <form method="get">
            <input type="hidden" name="page" value="edm-efatura-premium-logs" />
            
            <div class="form-row">
                <label for="level"><?php esc_html_e('Log Level', 'edm-efatura-premium'); ?></label>
                <select id="level" name="level">
                    <option value=""><?php esc_html_e('All Levels', 'edm-efatura-premium'); ?></option>
                    <option value="EMERGENCY" <?php selected($level, 'EMERGENCY'); ?>><?php esc_html_e('Emergency', 'edm-efatura-premium'); ?></option>
                    <option value="ALERT" <?php selected($level, 'ALERT'); ?>><?php esc_html_e('Alert', 'edm-efatura-premium'); ?></option>
                    <option value="CRITICAL" <?php selected($level, 'CRITICAL'); ?>><?php esc_html_e('Critical', 'edm-efatura-premium'); ?></option>
                    <option value="ERROR" <?php selected($level, 'ERROR'); ?>><?php esc_html_e('Error', 'edm-efatura-premium'); ?></option>
                    <option value="WARNING" <?php selected($level, 'WARNING'); ?>><?php esc_html_e('Warning', 'edm-efatura-premium'); ?></option>
                    <option value="NOTICE" <?php selected($level, 'NOTICE'); ?>><?php esc_html_e('Notice', 'edm-efatura-premium'); ?></option>
                    <option value="INFO" <?php selected($level, 'INFO'); ?>><?php esc_html_e('Info', 'edm-efatura-premium'); ?></option>
                    <option value="DEBUG" <?php selected($level, 'DEBUG'); ?>><?php esc_html_e('Debug', 'edm-efatura-premium'); ?></option>
                </select>
            </div>
            
            <div class="form-row">
                <label for="per_page"><?php esc_html_e('Per Page', 'edm-efatura-premium'); ?></label>
                <select id="per_page" name="per_page">
                    <option value="20" <?php selected($per_page, 20); ?>>20</option>
                    <option value="50" <?php selected($per_page, 50); ?>>50</option>
                    <option value="100" <?php selected($per_page, 100); ?>>100</option>
                    <option value="200" <?php selected($per_page, 200); ?>>200</option>
                </select>
            </div>
            
            <div class="form-row">
                <button type="submit" class="button button-primary"><?php esc_html_e('Filter', 'edm-efatura-premium'); ?></button>
                <a href="<?php echo esc_url(admin_url('admin.php?page=edm-efatura-premium-logs')); ?>" class="button"><?php esc_html_e('Reset', 'edm-efatura-premium'); ?></a>
            </div>
        </form>
    </div>
    
    <div class="edm-log-maintenance">
        <h2><?php esc_html_e('Log Maintenance', 'edm-efatura-premium'); ?></h2>
        <p>
            <?php 
            printf(
                /* translators: %d: number of days */
                esc_html__('Logs are automatically kept for %d days. You can manually clean older logs below:', 'edm-efatura-premium'),
                $log_retention
            ); 
            ?>
        </p>
        
        <div class="form-row">
            <label for="days"><?php esc_html_e('Keep logs newer than (days)', 'edm-efatura-premium'); ?></label>
            <input type="number" id="days" name="days" value="<?php echo esc_attr($log_retention); ?>" min="1" max="365" />
            <button type="button" id="edm-clean-logs" class="button"><?php esc_html_e('Clean Old Logs', 'edm-efatura-premium'); ?></button>
            <button type="button" id="edm-delete-all-logs" class="button button-secondary" style="margin-left: 10px;"><?php esc_html_e('Delete All Logs', 'edm-efatura-premium'); ?></button>
        </div>
    </div>
    
    <div class="edm-result-box" style="display: none;"></div>
    
    <h2><?php esc_html_e('Log Entries', 'edm-efatura-premium'); ?></h2>
    
    <?php if (empty($logs)) : ?>
        <p><?php esc_html_e('No logs found matching your criteria.', 'edm-efatura-premium'); ?></p>
    <?php else : ?>
        <div class="tablenav top">
            <div class="tablenav-pages">
                <span class="displaying-num">
                    <?php 
                    printf(
                        /* translators: %s: number of log entries */
                        _n('%s log entry', '%s log entries', $total, 'edm-efatura-premium'),
                        number_format_i18n($total)
                    ); 
                    ?>
                </span>
                
                <?php if ($total_pages > 1) : ?>
                    <span class="pagination-links">
                        <?php
                        echo paginate_links(array(
                            'base' => add_query_arg('paged', '%#%'),
                            'format' => '',
                            'prev_text' => '&laquo;',
                            'next_text' => '&raquo;',
                            'total' => $total_pages,
                            'current' => $page
                        ));
                        ?>
                    </span>
                <?php endif; ?>
            </div>
        </div>
        
        <table class="widefat striped edm-log-table">
            <thead>
                <tr>
                    <th><?php esc_html_e('Time', 'edm-efatura-premium'); ?></th>
                    <th><?php esc_html_e('Level', 'edm-efatura-premium'); ?></th>
                    <th><?php esc_html_e('Message', 'edm-efatura-premium'); ?></th>
                    <th><?php esc_html_e('Context', 'edm-efatura-premium'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($logs as $log) : 
                    $level_class = 'edm-log-level-' . strtolower($log['level']);
                    $has_context = !empty($log['context']);
                ?>
                    <tr>
                        <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($log['created_at']))); ?></td>
                        <td><span class="edm-log-level <?php echo esc_attr($level_class); ?>"><?php echo esc_html($log['level']); ?></span></td>
                        <td><?php echo esc_html($log['message']); ?></td>
                        <td>
							<?php if ($has_context) : ?>
								<button type="button" class="button button-small edm-log-toggle" data-log-id="<?php echo esc_attr($log['id']); ?>">
									<?php esc_html_e('Show Context', 'edm-efatura-premium'); ?>
								</button>
								<div id="log-context-<?php echo esc_attr($log['id']); ?>" class="edm-log-context" style="display: none;">
									<pre><?php echo wp_json_encode($log['context'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE); ?></pre>
								</div>
							<?php else : ?>
								<?php esc_html_e('No context', 'edm-efatura-premium'); ?>
							<?php endif; ?>
						</td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        
        <?php if ($total_pages > 1) : ?>
            <div class="tablenav bottom">
                <div class="tablenav-pages">
                    <span class="displaying-num">
                        <?php 
                        printf(
                            /* translators: %s: number of log entries */
                            _n('%s log entry', '%s log entries', $total, 'edm-efatura-premium'),
                            number_format_i18n($total)
                        ); 
                        ?>
                    </span>
                    
                    <span class="pagination-links">
                        <?php
                        echo paginate_links(array(
                            'base' => add_query_arg('paged', '%#%'),
                            'format' => '',
                            'prev_text' => '&laquo;',
                            'next_text' => '&raquo;',
                            'total' => $total_pages,
                            'current' => $page
                        ));
                        ?>
                    </span>
                </div>
            </div>
        <?php endif; ?>
    <?php endif; ?>
</div>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Delete all logs button
    $('#edm-delete-all-logs').on('click', function(e) {
        e.preventDefault();
        
        if (!confirm('<?php echo esc_js(__('Are you sure you want to delete ALL logs? This action cannot be undone!', 'edm-efatura-premium')); ?>')) {
            return;
        }
        
        var button = $(this);
        button.prop('disabled', true).text('<?php echo esc_js(__('Deleting...', 'edm-efatura-premium')); ?>');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'edm_delete_all_logs',
                nonce: '<?php echo wp_create_nonce('edm_efatura_premium_nonce'); ?>'
            },
            success: function(response) {
                button.prop('disabled', false).text('<?php echo esc_js(__('Delete All Logs', 'edm-efatura-premium')); ?>');
                
                if (response.success) {
                    alert(response.data.message);
                    location.reload(); // Refresh page to show empty logs
                } else {
                    alert('<?php echo esc_js(__('Error:', 'edm-efatura-premium')); ?> ' + response.data);
                }
            },
            error: function(xhr, status, error) {
                button.prop('disabled', false).text('<?php echo esc_js(__('Delete All Logs', 'edm-efatura-premium')); ?>');
                alert('<?php echo esc_js(__('Error:', 'edm-efatura-premium')); ?> ' + error);
            }
        });
    });
});
</script>