<?php
/**
 * Settings page template
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Get current tab
$current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'api';

// Create settings page instance
$settings = new EDM_Efatura_Premium_Settings();

// Get all settings
$options = get_option('edm_efatura_premium_settings', array());

// Get counter information safely
$counter_info = '';
$enable_counter_check = true; // Enable counter check safely

if ($enable_counter_check && !empty($options['api_url']) && !empty($options['username']) && !empty($options['password'])) {
    try {
        // Check if all required classes exist
        if (!class_exists('EDM_Efatura_Premium_API') || !class_exists('EDM_Efatura_Premium_Security')) {
            $counter_info = '<span style="color: #dc3545; font-size: 12px;">API sınıfları yüklenmedi</span>';
        } else {
            $edm_email = !empty($options['edm_email']) ? $options['edm_email'] : '';
            
            // Decrypt password safely
            $decrypted_password = '';
            try {
                $decrypted_password = EDM_Efatura_Premium_Security::decrypt($options['password']);
            } catch (Exception $decrypt_e) {
                $counter_info = '<span style="color: #dc3545; font-size: 12px;">Şifre çözülemedi</span>';
            }
            
            if (!empty($decrypted_password)) {
                $api = new EDM_Efatura_Premium_API(
                    $options['api_url'],
                    $options['username'],
                    $decrypted_password,
                    $edm_email
                );
                
                // Check counter with timeout protection
                $counter_result = $api->check_counter();
                if (is_array($counter_result) && isset($counter_result['success']) && $counter_result['success']) {
                    $counter_left = isset($counter_result['counter_left']) ? intval($counter_result['counter_left']) : 0;
                    $counter_info = '<span style="color: #28a745; font-weight: bold;">Kalan Kontör: ' . number_format($counter_left) . '</span>';
                } else {
                    $error_msg = isset($counter_result['message']) ? $counter_result['message'] : 'Bilinmeyen hata';
                    $counter_info = '<span style="color: #dc3545; font-size: 12px;">' . esc_html($error_msg) . '</span>';
                    
                    // Only show debug info to admin users and in debug mode
                    $settings = get_option('edm_efatura_premium_settings', array());
                    $debug_mode = isset($settings['debug_mode']) && $settings['debug_mode'] === 'yes';
                    
                    if ($debug_mode && current_user_can('manage_options') && isset($counter_result['debug'])) {
                        $counter_info .= '<br><details style="margin-top: 5px;"><summary style="color: #666; font-size: 11px; cursor: pointer;">Teknik Detaylar</summary>';
                        $counter_info .= '<small style="color: #666; font-family: monospace; word-break: break-all;">' . esc_html($counter_result['debug']) . '</small>';
                        $counter_info .= '</details>';
                    }
                }
            }
        }
    } catch (Exception $e) {
        $counter_info = '<span style="color: #dc3545; font-size: 12px;">Kontör sorgusu sırasında hata oluştu</span>';
    } catch (Error $e) {
        $counter_info = '<span style="color: #dc3545; font-size: 12px;">Kontör sorgusu sırasında hata oluştu</span>';
    }
}
?>

<div class="wrap edm-settings-wrap">
    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px;">
        <h1 style="margin: 0;"><?php echo esc_html__('EDM E-Fatura Premium Settings', 'edm-efatura-premium'); ?></h1>
        <?php if (!empty($counter_info)) : ?>
            <div style="font-size: 16px; padding: 8px 15px; background: #f8f9fa; border: 1px solid #dee2e6; border-radius: 4px;">
                <?php echo $counter_info; ?>
            </div>
        <?php endif; ?>
    </div>
    
    <?php if (isset($_GET['settings-updated']) && $_GET['settings-updated']) : ?>
        <div class="notice notice-success is-dismissible">
            <p><?php esc_html_e('Settings saved successfully.', 'edm-efatura-premium'); ?></p>
        </div>
    <?php endif; ?>
    
    <?php 
    // Show error messages if any
    settings_errors('edm_efatura_premium_settings'); 
    ?>
    
    <h2 class="nav-tab-wrapper">
        <?php
        // Create tabs
        $tabs = array(
            'api' => __('API Settings', 'edm-efatura-premium'),
            'company' => __('Company Information', 'edm-efatura-premium'),
            'invoice' => __('Invoice Settings', 'edm-efatura-premium'),
            'woocommerce' => __('WooCommerce Integration', 'edm-efatura-premium'),
            'logs' => __('Logs', 'edm-efatura-premium')
        );
        
        foreach ($tabs as $tab_id => $tab_title) {
            $active_class = ($current_tab === $tab_id) ? 'nav-tab-active' : '';
            echo '<a href="?page=edm-efatura-premium&tab=' . esc_attr($tab_id) . '" class="nav-tab ' . esc_attr($active_class) . '">' . esc_html($tab_title) . '</a>';
        }
        ?>
    </h2>
    
    <form method="post" action="options.php">
        <?php
        // All fields single form
        settings_fields('edm_efatura_premium_settings_group');
        
        // Add all existing settings as hidden fields
        // This helps preserve settings for other tabs when switching
        foreach ($options as $option_key => $option_value) {
            if (strpos($option_key, $current_tab) === false) {
                if (is_array($option_value)) {
                    // Store array values as JSON
                    $json_value = wp_json_encode($option_value);
                    echo '<input type="hidden" name="edm_efatura_premium_settings[' . esc_attr($option_key) . ']" value="' . esc_attr($json_value) . '" data-json="true" />';
                } else {
                    echo '<input type="hidden" name="edm_efatura_premium_settings[' . esc_attr($option_key) . ']" value="' . esc_attr($option_value) . '" />';
                }
            }
        }
        
        // Display current section
        echo '<div class="edm-settings-section-container">';
        
        switch ($current_tab) {
            case 'api':
                echo '<h3>' . esc_html__('API Settings', 'edm-efatura-premium') . '</h3>';
                echo '<p>' . esc_html__('Configure your EDM e-invoice API credentials.', 'edm-efatura-premium') . '</p>';
                
                // API URL
                echo '<table class="form-table">';
                echo '<tr>';
                echo '<th scope="row"><label for="api_url">' . esc_html__('EDM API URL', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="api_url" name="edm_efatura_premium_settings[api_url]" value="' . esc_attr(isset($options['api_url']) ? $options['api_url'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('EDM e-invoice API address. Test environment: https://test.edmbilisim.com.tr/EFaturaEDM21ea/EFaturaEDM.svc', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Username
                echo '<tr>';
                echo '<th scope="row"><label for="username">' . esc_html__('Username', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="username" name="edm_efatura_premium_settings[username]" value="' . esc_attr(isset($options['username']) ? $options['username'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your EDM e-invoice system username.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Password
                echo '<tr>';
                echo '<th scope="row"><label for="password">' . esc_html__('Password', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="password" id="password" name="edm_efatura_premium_settings[password]" value="' . esc_attr(isset($options['password']) ? $options['password'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your EDM e-invoice system password.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                echo '<tr>';
                echo '<th scope="row"><label for="edm_email">' . esc_html__('EDM Email', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="edm_email" name="edm_efatura_premium_settings[edm_email]" value="' . esc_attr(isset($options['edm_email']) ? $options['edm_email'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your EDM email address (e.g. username@edmbilisim.com) that matches your GIB label.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Debug Mode
                echo '<tr>';
                echo '<th scope="row"><label for="debug_mode">' . esc_html__('Debug Mode', 'edm-efatura-premium') . '</label></th>';
                echo '<td><input type="checkbox" id="debug_mode" name="edm_efatura_premium_settings[debug_mode]" value="yes" ' . checked(isset($options['debug_mode']) && $options['debug_mode'] === 'yes', true, false) . ' />';
                echo '<p class="description">' . esc_html__('Enable debug mode to log detailed API requests and responses.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                echo '</table>';
                
                // Test Connection button
                echo '<div class="edm-settings-action-buttons">';
                echo '<button type="button" id="edm-test-connection" class="button">' . esc_html__('Test Connection', 'edm-efatura-premium') . '</button>';
                echo '<div id="edm-test-connection-result" style="margin-top: 10px; padding: 15px; background-color: #f8f8f8; border-left: 4px solid #ccc; display: none;"></div>';
                echo '</div>';
                break;
                
            case 'company':
                echo '<h3>' . esc_html__('Company Information', 'edm-efatura-premium') . '</h3>';
                echo '<p>' . esc_html__('Enter your company information to be used in invoices.', 'edm-efatura-premium') . '</p>';
                
                echo '<table class="form-table">';
                
                // Company Prefix (for invoice numbering)
                echo '<tr>';
                echo '<th scope="row"><label for="company_prefix">' . esc_html__('Company Code', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_prefix" name="edm_efatura_premium_settings[company_prefix]" value="' . esc_attr(isset($options['company_prefix']) ? $options['company_prefix'] : '') . '" class="regular-text" maxlength="3" placeholder="MKZ" />';
                echo '<p class="description">' . esc_html__('Company code prefix (3 letters max) used for invoice numbering.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Name
                echo '<tr>';
                echo '<th scope="row"><label for="company_name">' . esc_html__('Company Name', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_name" name="edm_efatura_premium_settings[company_name]" value="' . esc_attr(isset($options['company_name']) ? $options['company_name'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company name to be used in invoices.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company VKN
                echo '<tr>';
                echo '<th scope="row"><label for="company_vkn">' . esc_html__('Tax Number (VKN/TCKN)', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_vkn" name="edm_efatura_premium_settings[company_vkn]" value="' . esc_attr(isset($options['company_vkn']) ? $options['company_vkn'] : '') . '" class="regular-text" pattern="[0-9]{10,11}" maxlength="11" />';
                echo '<p class="description">' . esc_html__('Your company tax identification number. Use VKN (10 digits) for companies, TCKN (11 digits) for sole proprietorships.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Tax Office
                echo '<tr>';
                echo '<th scope="row"><label for="company_tax_office">' . esc_html__('Tax Office', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_tax_office" name="edm_efatura_premium_settings[company_tax_office]" value="' . esc_attr(isset($options['company_tax_office']) ? $options['company_tax_office'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company tax office.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Street
                echo '<tr>';
                echo '<th scope="row"><label for="company_street">' . esc_html__('Street', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_street" name="edm_efatura_premium_settings[company_street]" value="' . esc_attr(isset($options['company_street']) ? $options['company_street'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company street address.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Building Number
                echo '<tr>';
                echo '<th scope="row"><label for="company_building">' . esc_html__('Building Number', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_building" name="edm_efatura_premium_settings[company_building]" value="' . esc_attr(isset($options['company_building']) ? $options['company_building'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company building number.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company District
                echo '<tr>';
                echo '<th scope="row"><label for="company_district">' . esc_html__('District', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_district" name="edm_efatura_premium_settings[company_district]" value="' . esc_attr(isset($options['company_district']) ? $options['company_district'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company district.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company City
                echo '<tr>';
                echo '<th scope="row"><label for="company_city">' . esc_html__('City', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_city" name="edm_efatura_premium_settings[company_city]" value="' . esc_attr(isset($options['company_city']) ? $options['company_city'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company city.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Postal Code
                echo '<tr>';
                echo '<th scope="row"><label for="company_postal">' . esc_html__('Postal Code', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_postal" name="edm_efatura_premium_settings[company_postal]" value="' . esc_attr(isset($options['company_postal']) ? $options['company_postal'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company postal code.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Region
                echo '<tr>';
				echo '<th scope="row"><label for="company_region">' . esc_html__('Region', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_region" name="edm_efatura_premium_settings[company_region]" value="' . esc_attr(isset($options['company_region']) ? $options['company_region'] : 'Marmara') . '" class="regular-text" placeholder="Marmara" />';
                echo '<p class="description">' . esc_html__('Your company region (e.g., Marmara, Aegean, etc.).', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Sub-district
                echo '<tr>';
				echo '<th scope="row"><label for="company_subdistrict">' . esc_html__('Sub-district', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_subdistrict" name="edm_efatura_premium_settings[company_subdistrict]" value="' . esc_attr(isset($options['company_subdistrict']) ? $options['company_subdistrict'] : 'Merkez') . '" class="regular-text" placeholder="Merkez" />';
                echo '<p class="description">' . esc_html__('Your company sub-district.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Country Code
                echo '<tr>';
				echo '<th scope="row"><label for="company_country_code">' . esc_html__('Country Code', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_country_code" name="edm_efatura_premium_settings[company_country_code]" value="' . esc_attr(isset($options['company_country_code']) ? $options['company_country_code'] : 'TR') . '" class="regular-text" placeholder="TR" maxlength="2" />';
                echo '<p class="description">' . esc_html__('Your company country code (e.g., TR for Turkey).', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Country
                echo '<tr>';
				echo '<th scope="row"><label for="company_country">' . esc_html__('Country', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_country" name="edm_efatura_premium_settings[company_country]" value="' . esc_attr(isset($options['company_country']) ? $options['company_country'] : 'Türkiye') . '" class="regular-text" placeholder="Türkiye" />';
                echo '<p class="description">' . esc_html__('Your company country.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Phone
                echo '<tr>';
                echo '<th scope="row"><label for="company_phone">' . esc_html__('Phone', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_phone" name="edm_efatura_premium_settings[company_phone]" value="' . esc_attr(isset($options['company_phone']) ? $options['company_phone'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company phone number.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Fax
                echo '<tr>';
                echo '<th scope="row"><label for="company_fax">' . esc_html__('Fax', 'edm-efatura-premium') . '</label></th>';
                echo '<td><input type="text" id="company_fax" name="edm_efatura_premium_settings[company_fax]" value="' . esc_attr(isset($options['company_fax']) ? $options['company_fax'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company fax number.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Email
                echo '<tr>';
                echo '<th scope="row"><label for="company_email">' . esc_html__('Email', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><input type="text" id="company_email" name="edm_efatura_premium_settings[company_email]" value="' . esc_attr(isset($options['company_email']) ? $options['company_email'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company email address.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Company Website
                echo '<tr>';
                echo '<th scope="row"><label for="company_website">' . esc_html__('Website', 'edm-efatura-premium') . '</label></th>';
                echo '<td><input type="text" id="company_website" name="edm_efatura_premium_settings[company_website]" value="' . esc_attr(isset($options['company_website']) ? $options['company_website'] : '') . '" class="regular-text" />';
                echo '<p class="description">' . esc_html__('Your company website URL.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
				// Ticaret Sicil Numarası
				echo '<tr>';
				echo '<th scope="row"><label for="company_trade_registry">' . esc_html__('Trade Registry Number', 'edm-efatura-premium') . '</label></th>';
				echo '<td><input type="text" id="company_trade_registry" name="edm_efatura_premium_settings[company_trade_registry]" value="' . esc_attr(isset($options['company_trade_registry']) ? $options['company_trade_registry'] : '') . '" class="regular-text" placeholder="' . esc_attr__('Trade registry number', 'edm-efatura-premium') . '" />';
				echo '<p class="description">' . esc_html__('Your company trade registry number.', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';

				// Mersis No
				echo '<tr>';
				echo '<th scope="row"><label for="company_mersis">' . esc_html__('MERSIS Number', 'edm-efatura-premium') . '</label></th>';
				echo '<td><input type="text" id="company_mersis" name="edm_efatura_premium_settings[company_mersis]" value="' . esc_attr(isset($options['company_mersis']) ? $options['company_mersis'] : '') . '" class="regular-text" placeholder="' . esc_attr__('MERSIS number', 'edm-efatura-premium') . '" />';
				echo '<p class="description">' . esc_html__('Your company MERSIS number.', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';
                // Ödeme Aracısı alanı
				echo '<tr>';
				echo '<th scope="row"><label for="payment_intermediary">' . esc_html__('Payment Intermediary', 'edm-efatura-premium') . '</label></th>';
				echo '<td><input type="text" id="payment_intermediary" name="edm_efatura_premium_settings[payment_intermediary]" value="' . esc_attr(isset($options['payment_intermediary']) ? $options['payment_intermediary'] : 'sanal') . '" class="regular-text" placeholder="sanal" />';
				echo '<p class="description">' . esc_html__('Payment intermediary name to be used in invoices. Default: sanal', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';

				// Kargo Şirket Adı alanı
				echo '<tr>';
				echo '<th scope="row"><label for="shipping_company_name">' . esc_html__('Shipping Company Name', 'edm-efatura-premium') . '</label></th>';
				echo '<td><input type="text" id="shipping_company_name" name="edm_efatura_premium_settings[shipping_company_name]" value="' . esc_attr(isset($options['shipping_company_name']) ? $options['shipping_company_name'] : 'Kargo') . '" class="regular-text" placeholder="Kargo" />';
				echo '<p class="description">' . esc_html__('Shipping company name to be used in invoices. Default: Kargo', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';
                // Company Logo
                echo '<tr>';
                echo '<th scope="row"><label for="company_logo">' . esc_html__('Company Logo', 'edm-efatura-premium') . '</label></th>';
                echo '<td><div class="edm-image-upload">';
                
                $logo_id = isset($options['company_logo']) ? $options['company_logo'] : '';
                $logo_url = $logo_id ? wp_get_attachment_url($logo_id) : '';
                
                echo '<input type="hidden" id="company_logo" name="edm_efatura_premium_settings[company_logo]" value="' . esc_attr($logo_id) . '" />';
                
                echo '<div class="edm-image-preview">';
                if ($logo_url) {
                    echo '<img src="' . esc_url($logo_url) . '" alt="" />';
                }
                echo '</div>';
                
                echo '<button type="button" class="button edm-upload-image" data-field="company_logo">' . esc_html__('Upload Logo', 'edm-efatura-premium') . '</button>';
                echo '<button type="button" class="button edm-remove-image" data-field="company_logo" style="' . ($logo_url ? '' : 'display:none;') . '">' . esc_html__('Remove Logo', 'edm-efatura-premium') . '</button>';
                
                echo '</div>';
                echo '<p class="description">' . esc_html__('Upload your company logo to be used in invoices.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                echo '</table>';
                break;
                
            case 'invoice':
                echo '<h3>' . esc_html__('Invoice Settings', 'edm-efatura-premium') . '</h3>';
                echo '<p>' . esc_html__('Configure general invoice settings.', 'edm-efatura-premium') . '</p>';
                
                echo '<table class="form-table">';
                
                // Invoice Type
                echo '<tr>';
                echo '<th scope="row"><label for="invoice_type">' . esc_html__('Default Invoice Type', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
                echo '<td><select id="invoice_type" name="edm_efatura_premium_settings[invoice_type]">';
                $invoice_types = array(
                    'SATIS' => __('Sales Invoice', 'edm-efatura-premium'),
                    
                    'ISTISNA' => __('Exemption Invoice', 'edm-efatura-premium'),
                    
                );
                foreach ($invoice_types as $key => $label) {
                    echo '<option value="' . esc_attr($key) . '" ' . selected(isset($options['invoice_type']) ? $options['invoice_type'] : 'SATIS', $key, false) . '>' . esc_html($label) . '</option>';
                }
                echo '</select>';
                echo '<p class="description">' . esc_html__('Default invoice type to be used.', 'edm-efatura-premium') . '</p></td>';
                echo '</tr>';
                
                // Exception Code (only visible when invoice type is ISTISNA)
                echo '<tr id="invoice_exception_code_row" style="' . (isset($options['invoice_type']) && $options['invoice_type'] === 'ISTISNA' ? '' : 'display:none;') . '">';
                echo '<th scope="row"><label for="invoice_exception_code">' . esc_html__('Exception Code', 'edm-efatura-premium') . ' <span class="required" style="' . (isset($options['invoice_type']) && $options['invoice_type'] === 'ISTISNA' ? '' : 'display:none;') . '">*</span></label></th>';
                echo '<td><select id="invoice_exception_code" name="edm_efatura_premium_settings[invoice_exception_code]">';
                echo '<option value="">' . esc_html__('None', 'edm-efatura-premium') . '</option>';
                
                // Merkezi istisna kodları sınıfını kullan
                $full_exception_codes = EDM_EFatura_Premium_Exception_Codes::get_formatted_exception_codes_for_forms();
                $full_codes = EDM_EFatura_Premium_Exception_Codes::get_full_exemption_codes();
                $partial_codes = EDM_EFatura_Premium_Exception_Codes::get_partial_exemption_codes();
                
                // Tam istisna kodları
                echo '<optgroup label="' . esc_attr__('Full Exemption Codes', 'edm-efatura-premium') . '">';
                foreach ($full_codes as $key => $description) {
                    $formatted_label = $key . ' - ' . $description;
                    echo '<option value="' . esc_attr($key) . '" ' . selected(isset($options['invoice_exception_code']) ? $options['invoice_exception_code'] : '', $key, false) . '>' . esc_html($formatted_label) . '</option>';
                }
                echo '</optgroup>';
               
                // Kısmi istisna kodları
                echo '<optgroup label="' . esc_attr__('Partial Exemption Codes', 'edm-efatura-premium') . '">';
                foreach ($partial_codes as $key => $description) {
                    $formatted_label = $key . ' - ' . $description;
                    echo '<option value="' . esc_attr($key) . '" ' . selected(isset($options['invoice_exception_code']) ? $options['invoice_exception_code'] : '', $key, false) . '>' . esc_html($formatted_label) . '</option>';
                }
                echo '</optgroup>';
               
               echo '</select>';
               echo '<p class="description">' . esc_html__('Select exception code for exemption invoices.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               // Default Tax Rate (for SATIS type)
				echo '<tr id="default_tax_rate_row" style="' . (isset($options['invoice_type']) && $options['invoice_type'] === 'SATIS' ? '' : 'display:none;') . '">';
				echo '<th scope="row"><label for="default_tax_rate">' . esc_html__('Default Tax Rate (%)', 'edm-efatura-premium') . ' <span class="required" style="' . (isset($options['invoice_type']) && $options['invoice_type'] === 'SATIS' ? '' : 'display:none;') . '">*</span></label></th>';
				echo '<td><input type="number" id="default_tax_rate" name="edm_efatura_premium_settings[default_tax_rate]" value="' . esc_attr(isset($options['default_tax_rate']) ? $options['default_tax_rate'] : '18') . '" class="small-text" min="0" step="0.01" />';
				echo '<p class="description">' . esc_html__('Default tax rate (%) to be used for Sales invoices. Must be greater than 0.', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';
				// WooCommerce Prices Include Tax
				echo '<tr id="woocommerce_prices_include_tax_row">';
				echo '<th scope="row"><label for="woocommerce_prices_include_tax">' . esc_html__('WooCommerce Prices', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
				echo '<td><select id="woocommerce_prices_include_tax" name="edm_efatura_premium_settings[woocommerce_prices_include_tax]">';
				echo '<option value="yes" ' . selected(isset($options['woocommerce_prices_include_tax']) ? $options['woocommerce_prices_include_tax'] : 'yes', 'yes', false) . '>' . esc_html__('Include Tax', 'edm-efatura-premium') . '</option>';
				echo '<option value="no" ' . selected(isset($options['woocommerce_prices_include_tax']) ? $options['woocommerce_prices_include_tax'] : 'yes', 'no', false) . '>' . esc_html__('Exclude Tax', 'edm-efatura-premium') . '</option>';
				echo '</select>';
				echo '<p class="description">' . esc_html__('Select whether your WooCommerce product prices include tax or not. This affects how invoice prices are calculated.', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';
              // Invoice Profile
				echo '<tr>';
				echo '<th scope="row"><label for="invoice_profile">' . esc_html__('Invoice Profile', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
				echo '<td><select id="invoice_profile" name="edm_efatura_premium_settings[invoice_profile]">';

				$profiles = array(
					'EARSIVFATURA' => __('E-Archive Invoice', 'edm-efatura-premium')
				);

				foreach ($profiles as $key => $label) {
					echo '<option value="' . esc_attr($key) . '" ' . selected(isset($options['invoice_profile']) ? $options['invoice_profile'] : 'EARSIVFATURA', $key, false) . '>' . esc_html($label) . '</option>';
				}

				echo '</select>';
				echo '<p class="description">' . esc_html__('E-Archive Invoice profile is used for individual customers. For corporate customers, TEMELFATURA profile will be used automatically when creating corporate invoices.', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';
               
               // Default Unit Code
               echo '<tr>';
               echo '<th scope="row"><label for="default_unit_code">' . esc_html__('Default Unit Code', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
               echo '<td><select id="default_unit_code" name="edm_efatura_premium_settings[default_unit_code]">';
               
               $unit_codes = array(
                   'C62' => 'C62 - ' . __('Unit/Piece', 'edm-efatura-premium'),
                   'KGM' => 'KGM - ' . __('Kilogram', 'edm-efatura-premium'),
                   'LTR' => 'LTR - ' . __('Liter', 'edm-efatura-premium'),
                   'MTR' => 'MTR - ' . __('Meter', 'edm-efatura-premium'),
                   'MTK' => 'MTK - ' . __('Square Meter', 'edm-efatura-premium'),
                   'MTQ' => 'MTQ - ' . __('Cubic Meter', 'edm-efatura-premium'),
                   'GRM' => 'GRM - ' . __('Gram', 'edm-efatura-premium'),
                   'KWH' => 'KWH - ' . __('Kilowatt Hour', 'edm-efatura-premium'),
                   'B32' => 'B32 - ' . __('KG-Square Meter', 'edm-efatura-premium'),
                   'CCT' => 'CCT - ' . __('Carrying Capacity per Ton', 'edm-efatura-premium'),
                   'PR' => 'PR - ' . __('Pair', 'edm-efatura-premium'),
                   'D30' => 'D30 - ' . __('Gross Calorie Value', 'edm-efatura-premium'),
                   'D40' => 'D40 - ' . __('Thousand Liters', 'edm-efatura-premium'),
                   'GFI' => 'GFI - ' . __('Gram of Fissile Isotope', 'edm-efatura-premium'),
                   'GT' => 'GT - ' . __('Gross Ton', 'edm-efatura-premium'),
                   'CEN' => 'CEN - ' . __('Hundred Units', 'edm-efatura-premium'),
                   'KPO' => 'KPO - ' . __('Kilogram Potassium Oxide', 'edm-efatura-premium'),
                   'MND' => 'MND - ' . __('Dried Net Weight Kilogram', 'edm-efatura-premium'),
                   '3I' => '3I - ' . __('Kilogram-Piece', 'edm-efatura-premium'),
                   'KFO' => 'KFO - ' . __('Kilogram Diphosphorus Pentoxide', 'edm-efatura-premium'),
                   'KHY' => 'KHY - ' . __('Kilogram Hydrogen Peroxide', 'edm-efatura-premium'),
                   'KMA' => 'KMA - ' . __('Kilogram Methyl Amines', 'edm-efatura-premium'),
                   'KNI' => 'KNI - ' . __('Kilogram of Nitrogen', 'edm-efatura-premium'),
                   'KPH' => 'KPH - ' . __('Kilogram Potassium Hydroxide', 'edm-efatura-premium'),
                   'KSD' => 'KSD - ' . __('Kilogram 90% Dry Product', 'edm-efatura-premium'),
                   'KSH' => 'KSH - ' . __('Kilogram Sodium Hydroxide', 'edm-efatura-premium'),
                   'KUR' => 'KUR - ' . __('Kilogram Uranium', 'edm-efatura-premium'),
                   'D32' => 'D32 - ' . __('Terawatt Hour', 'edm-efatura-premium'),
                   'GWH' => 'GWH - ' . __('Gigawatt Hour', 'edm-efatura-premium'),
                   'MWH' => 'MWH - ' . __('Megawatt Hour', 'edm-efatura-premium'),
                   'KWT' => 'KWT - ' . __('Kilowatt', 'edm-efatura-premium'),
                   'LPA' => 'LPA - ' . __('Liter Pure Alcohol', 'edm-efatura-premium'),
                   'DMK' => 'DMK - ' . __('Square Decimeter', 'edm-efatura-premium'),
                   'NCL' => 'NCL - ' . __('Number of Cells', 'edm-efatura-premium'),
                   'CTM' => 'CTM - ' . __('Carat', 'edm-efatura-premium'),
                   'SM3' => 'SM3 - ' . __('Standard Cubic Meter', 'edm-efatura-premium'),
                   'R9' => 'R9 - ' . __('Thousand Cubic Meters', 'edm-efatura-premium'),
                   'SET' => 'SET - ' . __('Set', 'edm-efatura-premium'),
                   'T3' => 'T3 - ' . __('Thousand Units', 'edm-efatura-premium')
               );
               
               foreach ($unit_codes as $key => $label) {
                   echo '<option value="' . esc_attr($key) . '" ' . selected(isset($options['default_unit_code']) ? $options['default_unit_code'] : 'C62', $key, false) . '>' . esc_html($label) . '</option>';
               }
               
               echo '</select>';
               echo '<p class="description">' . esc_html__('Default unit code to be used for products without specified units.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               
               
               // Default TC Number
               echo '<tr>';
               echo '<th scope="row"><label for="default_tc">' . esc_html__('Default TC Number', 'edm-efatura-premium') . ' <span class="required">*</span></label></th>';
               echo '<td><input type="text" id="default_tc" name="edm_efatura_premium_settings[default_tc]" value="' . esc_attr(isset($options['default_tc']) ? $options['default_tc'] : '11111111111') . '" class="regular-text" pattern="[0-9]{11}" maxlength="11" />';
               echo '<p class="description">' . esc_html__('Default TC number to be used for individual customers who do not provide their own TC number.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               echo '</table>';
               break;
               
           case 'woocommerce':
               echo '<h3>' . esc_html__('WooCommerce Integration', 'edm-efatura-premium') . '</h3>';
               echo '<p>' . esc_html__('Configure WooCommerce integration settings.', 'edm-efatura-premium') . '</p>';
               
               echo '<table class="form-table">';
               
               // Enable WooCommerce Integration
               echo '<tr>';
               echo '<th scope="row"><label for="woocommerce_enabled">' . esc_html__('Enable WooCommerce Integration', 'edm-efatura-premium') . '</label></th>';
               echo '<td><input type="checkbox" id="woocommerce_enabled" name="edm_efatura_premium_settings[woocommerce_enabled]" value="yes" ' . checked(isset($options['woocommerce_enabled']) && $options['woocommerce_enabled'] === 'yes', true, false) . ' />';
               echo '<p class="description">' . esc_html__('Enable integration with WooCommerce.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
			  
               
               // Conditional display for WooCommerce specific settings
               $woocommerce_display = isset($options['woocommerce_enabled']) && $options['woocommerce_enabled'] === 'yes' ? '' : 'style="display:none;"';
               
			   // Customer TC Number Meta Key
				echo '<tr ' . $woocommerce_display . '>';
				echo '<th scope="row"><label for="customer_tc_meta_key">' . esc_html__('Customer TC Number Meta Key', 'edm-efatura-premium') . '</label></th>';
				echo '<td><input type="text" id="customer_tc_meta_key" name="edm_efatura_premium_settings[customer_tc_meta_key]" value="' . esc_attr(isset($options['customer_tc_meta_key']) ? $options['customer_tc_meta_key'] : '') . '" class="regular-text" />';
				echo '<p class="description">' . esc_html__('If you store customer TC numbers in a custom meta key, enter it here. Leave empty to use default TC number.', 'edm-efatura-premium') . '</p></td>';
				echo '</tr>';
			   
			   
               // Auto-Generate Invoice
               echo '<tr ' . $woocommerce_display . '>';
               echo '<th scope="row"><label for="auto_generate_invoice">' . esc_html__('Auto-Generate Invoice', 'edm-efatura-premium') . '</label></th>';
               echo '<td><input type="checkbox" id="auto_generate_invoice" name="edm_efatura_premium_settings[auto_generate_invoice]" value="yes" ' . checked(isset($options['auto_generate_invoice']) && $options['auto_generate_invoice'] === 'yes', true, false) . ' />';
               echo '<p class="description">' . esc_html__('Automatically generate invoice when order is completed.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               
               
               // Attach Invoice to Order Email
               echo '<tr ' . $woocommerce_display . '>';
               echo '<th scope="row"><label for="attach_pdf_to_email">' . esc_html__('Attach Invoice to Order Email', 'edm-efatura-premium') . '</label></th>';
               echo '<td><input type="checkbox" id="attach_pdf_to_email" name="edm_efatura_premium_settings[attach_pdf_to_email]" value="yes" ' . checked(isset($options['attach_pdf_to_email']) && $options['attach_pdf_to_email'] === 'yes', true, false) . ' />';
               echo '<p class="description">' . esc_html__('Attach PDF invoice to completed order email.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';

               
               
               // Order Status for Invoice
               echo '<tr ' . $woocommerce_display . '>';
               echo '<th scope="row"><label for="order_status_for_invoice">' . esc_html__('Order Status for Invoice', 'edm-efatura-premium') . '</label></th>';
               echo '<td><select id="order_status_for_invoice" name="edm_efatura_premium_settings[order_status_for_invoice]">';
               
               $order_statuses = array(
                   'processing' => __('Processing', 'edm-efatura-premium'),
                   'completed' => __('Completed', 'edm-efatura-premium'),
                   'on-hold' => __('On Hold', 'edm-efatura-premium'),
                   'paid' => __('Paid', 'edm-efatura-premium')
               );
               
               foreach ($order_statuses as $key => $label) {
                   echo '<option value="' . esc_attr($key) . '" ' . selected(isset($options['order_status_for_invoice']) ? $options['order_status_for_invoice'] : 'completed', $key, false) . '>' . esc_html($label) . '</option>';
               }
               
               echo '</select>';
               echo '<p class="description">' . esc_html__('Order status when invoice should be generated.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               // Invoice Generation Delay (days)
               echo '<tr ' . $woocommerce_display . '>';
               echo '<th scope="row"><label for="invoice_delay_days">' . esc_html__('Invoice Generation Delay (days)', 'edm-efatura-premium') . '</label></th>';
               echo '<td><input type="number" id="invoice_delay_days" name="edm_efatura_premium_settings[invoice_delay_days]" value="' . esc_attr(isset($options['invoice_delay_days']) ? $options['invoice_delay_days'] : '0') . '" min="0" max="30" class="small-text" />';
               echo '<p class="description">' . esc_html__('Number of days to wait after order completion before generating the invoice. Set to 0 for immediate generation.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               echo '</table>';
               break;
               
           
               
           case 'logs':
               echo '<h3>' . esc_html__('Logs', 'edm-efatura-premium') . '</h3>';
               echo '<p>' . esc_html__('Configure logging settings.', 'edm-efatura-premium') . '</p>';
               
               echo '<table class="form-table">';
               
               // Log Level
               echo '<tr>';
               echo '<th scope="row"><label for="log_level">' . esc_html__('Log Level', 'edm-efatura-premium') . '</label></th>';
               echo '<td><select id="log_level" name="edm_efatura_premium_settings[log_level]">';
               
               $log_levels = array(
                   'emergency' => __('Emergency', 'edm-efatura-premium'),
                   'alert' => __('Alert', 'edm-efatura-premium'),
                   'critical' => __('Critical', 'edm-efatura-premium'),
                   'error' => __('Error', 'edm-efatura-premium'),
                   'warning' => __('Warning', 'edm-efatura-premium'),
                   'notice' => __('Notice', 'edm-efatura-premium'),
                   'info' => __('Info', 'edm-efatura-premium'),
                   'debug' => __('Debug', 'edm-efatura-premium')
               );
               
               foreach ($log_levels as $key => $label) {
                   echo '<option value="' . esc_attr($key) . '" ' . selected(isset($options['log_level']) ? $options['log_level'] : 'info', $key, false) . '>' . esc_html($label) . '</option>';
               }
               
               echo '</select>';
               echo '<p class="description">' . esc_html__('Log level.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               // Log Retention
               echo '<tr>';
               echo '<th scope="row"><label for="log_retention">' . esc_html__('Log Retention (days)', 'edm-efatura-premium') . '</label></th>';
               echo '<td><input type="number" id="log_retention" name="edm_efatura_premium_settings[log_retention]" value="' . esc_attr(isset($options['log_retention']) ? $options['log_retention'] : '30') . '" min="1" max="365" class="small-text" />';
               echo '<p class="description">' . esc_html__('Number of days to keep logs.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               // Clean logs button
               echo '<tr>';
               echo '<th scope="row">' . esc_html__('Clean Logs', 'edm-efatura-premium') . '</th>';
               echo '<td><button type="button" id="edm-clean-logs" class="button">' . esc_html__('Clean Old Logs', 'edm-efatura-premium') . '</button>';
               echo '<p class="description">' . esc_html__('Manually clean logs older than the retention period.', 'edm-efatura-premium') . '</p></td>';
               echo '</tr>';
               
               echo '</table>';
               break;
       }
       
       echo '</div>';
       
       submit_button();
       ?>
   </form>
</div>

<script>
jQuery(document).ready(function($) {
   // Invoice Type değiştiğinde istisna kodları alanını göster/gizle
   $('#invoice_type').on('change', function() {
		var type = $(this).val();
		
		if (type === 'ISTISNA') {
			$('#invoice_exception_code_row').show();
			$('.edm-settings-wrap label[for="invoice_exception_code"] .required').show();
			$('#default_tax_rate_row').hide();
			$('.edm-settings-wrap label[for="default_tax_rate"] .required').hide();
		} else if (type === 'SATIS') {
			$('#invoice_exception_code_row').hide();
			$('.edm-settings-wrap label[for="invoice_exception_code"] .required').hide();
			$('#default_tax_rate_row').show();
			$('.edm-settings-wrap label[for="default_tax_rate"] .required').show();
		} else {
			$('#invoice_exception_code_row').hide();
			$('.edm-settings-wrap label[for="invoice_exception_code"] .required').hide();
			$('#default_tax_rate_row').hide();
			$('.edm-settings-wrap label[for="default_tax_rate"] .required').hide();
		}
	}).trigger('change');
   
   // WooCommerce açık/kapalı durumuna göre ilgili alanları göster/gizle
   $('#woocommerce_enabled').on('change', function() {
       if ($(this).is(':checked')) {
			$('#auto_generate_invoice, #attach_pdf_to_email, #order_status_for_invoice, #invoice_delay_days').closest('tr').show();
		} else {
			$('#auto_generate_invoice, #attach_pdf_to_email, #order_status_for_invoice, #invoice_delay_days').closest('tr').hide();
		}
   }).trigger('change');
   
   // Test connection
   $('#edm-test-connection').on('click', function(e) {
       e.preventDefault();
       
       $('#edm-test-connection-result').html('<p><span class="spinner is-active"></span> <?php echo esc_js(__('Testing connection...', 'edm-efatura-premium')); ?></p>').show();
       
       $.ajax({
           url: ajaxurl,
           type: 'POST',
           data: {
               action: 'edm_test_connection',
               nonce: edmEfaturaParams.nonce
           },
           success: function(response) {
               if (response.success) {
                   $('#edm-test-connection-result').html('<p style="color: #46b450;"><strong><?php echo esc_js(__('Success:', 'edm-efatura-premium')); ?></strong> ' + response.data.message + '</p>');
               } else {
                   $('#edm-test-connection-result').html('<p style="color: #dc3232;"><strong><?php echo esc_js(__('Error:', 'edm-efatura-premium')); ?></strong> ' + response.data + '</p>');
               }
           },
           error: function(xhr, status, error) {
               $('#edm-test-connection-result').html('<p style="color: #dc3232;"><strong><?php echo esc_js(__('Error:', 'edm-efatura-premium')); ?></strong> ' + error + '</p>');
           }
       });
   });
   
   // Image uploader
   $('.edm-upload-image').on('click', function(e) {
       e.preventDefault();
       
       var button = $(this);
       var fieldId = button.data('field');
       
       var frame = wp.media({
           title: '<?php echo esc_js(__('Select or Upload Image', 'edm-efatura-premium')); ?>',
           button: {
               text: '<?php echo esc_js(__('Use this image', 'edm-efatura-premium')); ?>'
           },
           multiple: false
       });
       
       frame.on('select', function() {
           var attachment = frame.state().get('selection').first().toJSON();
           $('#' + fieldId).val(attachment.id);
           $('#' + fieldId).closest('.edm-image-upload').find('.edm-image-preview').html('<img src="' + attachment.url + '" alt="" />');
           button.closest('.edm-image-upload').find('.edm-remove-image').show();
       });
       
       frame.open();
   });
   
   // Remove image
   $('.edm-remove-image').on('click', function(e) {
       e.preventDefault();
       
       var button = $(this);
       var fieldId = button.data('field');
       
       $('#' + fieldId).val('');
       $('#' + fieldId).closest('.edm-image-upload').find('.edm-image-preview').html('');
       button.hide();
   });
   
   // Clean logs button
   $('#edm-clean-logs').on('click', function(e) {
       e.preventDefault();
       
       if (!confirm('<?php echo esc_js(__('Are you sure you want to clean old logs?', 'edm-efatura-premium')); ?>')) {
           return;
       }
       
       var days = $('#log_retention').val();
       
       $(this).prop('disabled', true).html('<span class="spinner is-active" style="float:left;margin:0 4px 0 0;"></span> <?php echo esc_js(__('Cleaning...', 'edm-efatura-premium')); ?>');
       
       $.ajax({
           url: ajaxurl,
           type: 'POST',
           data: {
               action: 'edm_clean_logs',
               nonce: edmEfaturaParams.nonce,
               days: days
           },
           success: function(response) {
               $('#edm-clean-logs').prop('disabled', false).text('<?php echo esc_js(__('Clean Old Logs', 'edm-efatura-premium')); ?>');
               
               if (response.success) {
                   alert(response.data.message);
               } else {
                   alert('<?php echo esc_js(__('Error:', 'edm-efatura-premium')); ?> ' + response.data);
               }
           },
           error: function(xhr, status, error) {
               $('#edm-clean-logs').prop('disabled', false).text('<?php echo esc_js(__('Clean Old Logs', 'edm-efatura-premium')); ?>');
               alert('<?php echo esc_js(__('Error:', 'edm-efatura-premium')); ?> ' + error);
           }
       });
   });
   
   // Form validation for required fields
   $('form').on('submit', function(e) {
       var hasErrors = false;
       var currentTab = '<?php echo esc_js($current_tab); ?>';
       
       // Check specific required fields based on current tab
       if (currentTab === 'api') {
           // Check API fields
           var fieldsToCheck = ['api_url', 'username', 'password', 'edm_email'];
           
           $.each(fieldsToCheck, function(index, field) {
               if ($('#' + field).val() === '') {
                   $('#' + field).addClass('form-invalid');
                   hasErrors = true;
               } else {
                   $('#' + field).removeClass('form-invalid');
               }
           });
       } 
       else if (currentTab === 'company') {
           // Check Company fields
           var fieldsToCheck = ['company_prefix', 'company_name', 'company_vkn', 'company_tax_office', 
               'company_street', 'company_building', 'company_district', 'company_city', 
               'company_postal', 'company_phone', 'company_email'];
           
           $.each(fieldsToCheck, function(index, field) {
               if ($('#' + field).val() === '') {
                   $('#' + field).addClass('form-invalid');
                   hasErrors = true;
               } else {
                   $('#' + field).removeClass('form-invalid');
               }
           });
           
           // Special validation for VKN/TCKN
           var vkn = $('#company_vkn').val();
           if (vkn !== '' && (vkn.length !== 10 && vkn.length !== 11)) {
               $('#company_vkn').addClass('form-invalid');
               
               if ($('#company_vkn').next('.form-invalid-message').length === 0) {
                   $('#company_vkn').after('<span class="form-invalid-message"><?php echo esc_js(__('Tax Number must be 10 digits (VKN) or 11 digits (TCKN).', 'edm-efatura-premium')); ?></span>');
               }
               
               hasErrors = true;
           }
       } 
       else if (currentTab === 'invoice') {
           // Check Invoice fields
           var fieldsToCheck = ['default_tc'];
           
           $.each(fieldsToCheck, function(index, field) {
               if ($('#' + field).val() === '') {
                   $('#' + field).addClass('form-invalid');
                   hasErrors = true;
               } else {
                   $('#' + field).removeClass('form-invalid');
               }
           });
           
           // Special validation for ISTISNA type
           if ($('#invoice_type').val() === 'ISTISNA' && $('#invoice_exception_code').val() === '') {
               $('#invoice_exception_code').addClass('form-invalid');
               
               if ($('#invoice_exception_code').next('.form-invalid-message').length === 0) {
                   $('#invoice_exception_code').after('<span class="form-invalid-message"><?php echo esc_js(__('Exception Code is required for Exemption Invoices.', 'edm-efatura-premium')); ?></span>');
               }
               
               hasErrors = true;
           }
       }
       
       if (hasErrors) {
           e.preventDefault();
           
           // Show error message
           var errorMessage = $('<div class="notice notice-error"><p><?php echo esc_js(__('Please correct the errors before saving settings.', 'edm-efatura-premium')); ?></p></div>');
           $('.wrap > h1').after(errorMessage);
           
           // Scroll to first error
           $('html, body').animate({
               scrollTop: $('.form-invalid').first().offset().top - 100
           }, 500);
       }
   });
   
   // Clear validation errors when value changes
   $('.edm-settings-section-container input, .edm-settings-section-container select').on('change keyup', function() {
       $(this).removeClass('form-invalid');
       $(this).next('.form-invalid-message').remove();
   });
});
</script>