<?php
/**
 * Plugin Name: EDM E-Fatura Premium
 * Plugin URI: https://wcplugin.com/eklenti/woocommerce-edm-e-fatura-premium-eklentisi/
 * Description: EDM E-Fatura Entegrasyon Premium Eklentisi
 * Version: 2.5.3
 * Author: WCPlugin.com Ekibi
 * Author URI: https://wcplugin.com
 * Text Domain: edm-efatura-premium
 * Domain Path: /languages
 * Requires at least: 5.6
 * Requires PHP: 7.3
 * WC requires at least: 5.0
 * WC tested up to: 8.0
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('EDM_EFATURA_PREMIUM_VERSION', '2.5.3');
define('EDM_EFATURA_PREMIUM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('EDM_EFATURA_PREMIUM_PLUGIN_URL', plugin_dir_url(__FILE__));
define('EDM_EFATURA_PREMIUM_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('EDM_EFATURA_PREMIUM_APPLICATION_NAME', 'EDM-Woocommerce-WCPlugincom');

/**
 * Load plugin textdomain for translations
 */
function edm_efatura_premium_load_textdomain() {
    load_plugin_textdomain('edm-efatura-premium', false, dirname(plugin_basename(__FILE__)) . '/languages');
}
add_action('plugins_loaded', 'edm_efatura_premium_load_textdomain');

/**
 * Main plugin class
 */
class EDM_Efatura_Premium {
    /**
     * Singleton instance
     *
     * @var EDM_Efatura_Premium
     */
    private static $instance = null;
    
    /**
     * The plugin loader
     *
     * @var EDM_Efatura_Premium_Loader
     */
    private $loader;
    
    /**
     * API instance
     *
     * @var EDM_Efatura_Premium_API
     */
    private $api;
    
    /**
     * Admin instance
     *
     * @var EDM_Efatura_Premium_Admin
     */
    private $admin;
    
    /**
     * WooCommerce integration instance
     *
     * @var EDM_Efatura_Premium_WooCommerce
     */
    private $woocommerce;
    
    /**
     * Usage tracker instance
     *
     * @var EDM_Efatura_Premium_Usage_Tracker
     */
    private $usage_tracker;
    
    /**
     * Get the singleton instance
     *
     * @return EDM_Efatura_Premium
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Private constructor to prevent creating a new instance
     */
    private function __construct() {
		$this->define_constants();
		$this->load_dependencies();
		$this->set_locale();
		$this->init_hooks();
		
		// Initialize API if settings are available
		$this->init_api();
		
		// Fix scheduled events
		add_action('admin_init', array($this, 'fix_scheduled_events'), 20);
		
		// Clean duplicate usage tracking records (one-time)
		add_action('admin_init', array($this, 'clean_usage_duplicates'), 21);
		
		// Ensure cron checker is initialized
		add_action('init', array($this, 'ensure_cron_checker_initialized'), 30);
		
		// Direkt hook kaydı
		add_action('edm_direct_generate_invoice', array($this, 'process_direct_invoice_generation'), 10, 2);
		add_action('edm_check_scheduled_invoices', array($this, 'process_scheduled_invoices_check'));
		
	}
	
	/**
	 * Process direct invoice generation
	 *
	 * @param int $order_id Order ID
	 * @param string $key Schedule key
	 * @return bool Success status
	 */
	public function process_direct_invoice_generation($order_id, $key) {
		if ($this->woocommerce) {
			return $this->woocommerce->direct_generate_invoice_for_order($order_id, $key);
		}
		return false;
	}

	/**
	 * Process scheduled invoices check
	 */
	public function process_scheduled_invoices_check() {
		global $edm_cron_checker_instance;
		if (isset($edm_cron_checker_instance)) {
			$edm_cron_checker_instance->check_scheduled_invoices();
		}
	}
	
	/**
	 * Clean duplicate usage tracking records (one-time operation)
	 */
	public function clean_usage_duplicates() {
		// Check if cleanup has already been done
		$cleanup_done = get_option('edm_usage_duplicates_cleaned', false);
		
		if (!$cleanup_done && $this->usage_tracker) {
			$deleted_count = $this->usage_tracker->clean_duplicate_records();
			
			if ($deleted_count !== false) {
				// Mark cleanup as completed
				update_option('edm_usage_duplicates_cleaned', true);
				
				if ($deleted_count > 0) {
					$logger = new EDM_Efatura_Premium_Logger();
					$logger->info("Usage duplicates cleanup completed: {$deleted_count} duplicate records removed");
				}
			}
		}
	}
	
	/**
	 * Ensure cron checker is initialized
	 */
	public function ensure_cron_checker_initialized() {
		global $edm_cron_checker_instance;
		if (!isset($edm_cron_checker_instance)) {
			$edm_cron_checker_instance = new EDM_Efatura_Premium_Cron_Checker();
		}
	}
	
	/**
	 * Initialize API instance
	 */
	private function init_api() {
		$settings = get_option('edm_efatura_premium_settings', array());
		if (!empty($settings['api_url']) && !empty($settings['username']) && !empty($settings['password'])) {
			try {
				$edm_email = !empty($settings['edm_email']) ? $settings['edm_email'] : '';
				
				$this->api = new EDM_Efatura_Premium_API(
					$settings['api_url'],
					$settings['username'],
					$settings['password'],
					isset($settings['debug_mode']) && $settings['debug_mode'] === 'yes',
					$edm_email
				);
				
				// Opsiyonel: API'nin çalıştığını doğrula
				if (isset($settings['verify_api_connection']) && $settings['verify_api_connection'] === 'yes') {
					$session_id = $this->api->login();
					if ($session_id) {
						$this->api->logout($session_id);
					}
				}
			} catch (Exception $e) {
				// API oluşturulurken hata olursa günlüğe kaydet
				$logger = new EDM_Efatura_Premium_Logger();
				$logger->error('Failed to initialize API: ' . $e->getMessage());
				$this->api = null;
			}
		}
	}
	
	/**
	 * Get WooCommerce integration instance
	 *
	 * @return EDM_Efatura_Premium_WooCommerce|null
	 */
	public function get_woocommerce() {
		return $this->woocommerce;
	}
	
	
	/**
 * Make sure scheduled events are working correctly
 */
public function fix_scheduled_events() {
    // Sadece admin arayüzünde çalışsın ve direk işlem yapmasın
    if (is_admin() && !defined('DOING_AJAX')) {
        global $wpdb;
        
        // WooCommerce aktif değilse çık
        if (!$this->is_woocommerce_active()) {
            return;
        }
        
        // HPOS uyumlu sorgu hazırla
        $hpos_enabled = false;
        if (class_exists('\Automattic\WooCommerce\Utilities\OrderUtil')) {
            try {
                $hpos_enabled = \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
            } catch (Exception $e) {
                // HPOS not available, use legacy method
                $hpos_enabled = false;
            }
        }
        
        // Zamanlanmış faturaları al ama işleme
        $order_ids = array();
        
        if ($hpos_enabled) {
            // HPOS için sorgu - tablo varlığını kontrol et
            $table_name = $wpdb->prefix . 'wc_orders_meta';
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name;
            
            if ($table_exists) {
                $order_ids = $wpdb->get_col(
                    "SELECT order_id FROM {$table_name} 
                    WHERE meta_key = 'edm_invoice_scheduled' 
                    AND meta_value != ''"
                );
            }
        } else {
            // Klasik postmeta için sorgu
            $order_ids = $wpdb->get_col(
                "SELECT post_id FROM {$wpdb->postmeta} 
                WHERE meta_key = 'edm_invoice_scheduled' 
                AND meta_value != ''"
            );
        }
        
        if (empty($order_ids)) {
            return;
        }
        
        $logger = new EDM_Efatura_Premium_Logger();
        
        
        // Scheduled invoices listesini güncelle, ama zamanı gelse bile işleme
        $scheduled_invoices = get_option('edm_scheduled_invoices', array());
        $updated = false;
        
        foreach ($order_ids as $order_id) {
            $order = wc_get_order($order_id);
            if (!$order) {
                continue;
            }
            
            // Check if invoice already exists - if so, skip and clean up scheduled data
            $db = new EDM_Efatura_Premium_DB();
            $existing_invoice = $db->get_invoice_by_order_id($order_id);
            if ($existing_invoice) {
                // Invoice exists, clean up scheduled data
                if ($hpos_enabled) {
                    $order->delete_meta_data('edm_invoice_scheduled');
                    $order->delete_meta_data('edm_invoice_schedule_key');
                    $order->save();
                } else {
                    delete_post_meta($order_id, 'edm_invoice_scheduled');
                    delete_post_meta($order_id, 'edm_invoice_schedule_key');
                }
                
                $logger->info('Cleaned up scheduled data for order with existing invoice', array(
                    'order_id' => $order_id,
                    'invoice_id' => $existing_invoice['invoice_id']
                ));
                continue;
            }
            
            // Zamanlanmış süreyi al
            $scheduled_time = $hpos_enabled ? 
                $order->get_meta('edm_invoice_scheduled', true) : 
                get_post_meta($order_id, 'edm_invoice_scheduled', true);
            
            if (empty($scheduled_time)) {
                continue;
            }
            
            // Schedule key'i al veya oluştur
            $schedule_key = $hpos_enabled ? 
                $order->get_meta('edm_invoice_schedule_key', true) : 
                get_post_meta($order_id, 'edm_invoice_schedule_key', true);
                
            if (empty($schedule_key)) {
                // Yeni key oluştur
                $schedule_key = 'edm_schedule_' . md5($order_id . time());
                
                if ($hpos_enabled) {
                    $order->update_meta_data('edm_invoice_schedule_key', $schedule_key);
                    $order->save();
                } else {
                    update_post_meta($order_id, 'edm_invoice_schedule_key', $schedule_key);
                }
            }
            
            // Scheduled invoices listesine ekle
            $scheduled_invoices[$order_id] = array(
                'timestamp' => $scheduled_time,
                'key' => $schedule_key
            );
            
            $updated = true;
        }
        
        // Sadece listeyi güncelle, işlem yapma
        if ($updated) {
            update_option('edm_scheduled_invoices', $scheduled_invoices);
            
        }
    }
}
    
    /**
     * Define plugin constants
     */
    private function define_constants() {
        // Database constants
        define('EDM_EFATURA_PREMIUM_DB_VERSION', '1.0');
        define('EDM_EFATURA_PREMIUM_TABLE_INVOICES', 'edm_invoices');
        define('EDM_EFATURA_PREMIUM_TABLE_INVOICE_ITEMS', 'edm_invoice_items');
        define('EDM_EFATURA_PREMIUM_TABLE_LOG', 'edm_log');
    }
    
    /**
     * Load required dependencies
     */
    private function load_dependencies() {
		// Include core files
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-loader.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-i18n.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-api.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-logger.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-security.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-cache.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-performance.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-db.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-cron-checker.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-exception-codes.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-usage-tracker.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/class-edm-efatura-premium-update-checker.php';
		
		// Include admin files
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/class-edm-efatura-premium-admin.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/class-edm-efatura-premium-settings.php';
		require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'admin/class-edm-efatura-premium-updates.php';
		
		// WooCommerce integration
		if ($this->is_woocommerce_active()) {
			require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/integrations/woocommerce/class-edm-efatura-premium-woocommerce.php';
			require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/integrations/woocommerce/class-edm-efatura-premium-order-handler.php';
			require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/integrations/woocommerce/class-edm-efatura-premium-email.php';
		}
		
		// Initialize loader
		$this->loader = new EDM_Efatura_Premium_Loader();
	}
    
    /**
     * Set the locale for internationalization
     */
    private function set_locale() {
        $i18n = new EDM_Efatura_Premium_i18n();
        $this->loader->add_action('plugins_loaded', $i18n, 'load_plugin_textdomain');
    }
    
    /**
     * Register all hooks for the plugin
     */
    private function init_hooks() {
		// Register activation and deactivation hooks
		register_activation_hook(__FILE__, array($this, 'activate'));
		register_deactivation_hook(__FILE__, array($this, 'deactivate'));
		
		// Initialize admin
		if (is_admin()) {
			// Check WooCommerce dependency
			$this->loader->add_action('admin_notices', $this, 'check_woocommerce_dependency');
			
			$this->admin = new EDM_Efatura_Premium_Admin($this->get_api());
			$this->loader->add_action('admin_menu', $this->admin, 'add_admin_menu');
			$this->loader->add_action('admin_init', $this->admin, 'register_settings');
			$this->loader->add_action('admin_enqueue_scripts', $this->admin, 'enqueue_scripts');
		}
		
		// Initialize usage tracker
		$this->usage_tracker = new EDM_Efatura_Premium_Usage_Tracker();
		
		// Initialize update checker
		$this->update_checker = new EDM_Efatura_Premium_Update_Checker();
		
		// Initialize updates admin page
		if (is_admin()) {
			$this->updates_admin = new EDM_Efatura_Premium_Updates($this->update_checker);
		}
		
		// Initialize WooCommerce integration if available
		if ($this->is_woocommerce_active()) {
			$this->woocommerce = new EDM_Efatura_Premium_WooCommerce($this->get_api());
			
			// Email handler
			$this->loader->add_action('woocommerce_init', $this, 'init_email_handler');
			
			// Register WooCommerce hooks
			if ($this->woocommerce) {
				$this->loader->add_action('woocommerce_checkout_update_order_meta', $this->woocommerce, 'save_invoice_fields');
				$this->loader->add_action('woocommerce_order_status_completed', $this->woocommerce, 'maybe_generate_invoice');
				
				// HPOS compatibility
				$this->loader->add_action('before_woocommerce_init', $this, 'declare_hpos_compatibility');
				
				
			}
		}
		
		// Execute all registered hooks
		$this->loader->run();
	}
	
	
	
	
	
	
	/**
	 * Initialize email handler
	 */
	public function init_email_handler() {
		// Initialize EDM e-mail handler
		$this->email = new EDM_Efatura_Premium_Email();
		
		// Force WooCommerce to load all email classes
		$mailer = WC()->mailer();
		if (method_exists($mailer, 'init')) {
			$mailer->init();
		}
		
		// Log available email classes for debugging
		$logger = new EDM_Efatura_Premium_Logger();
		$emails = $mailer->get_emails();
		
	}
	
	
	
    /**
     * Declare HPOS compatibility for WooCommerce
     */
    public function declare_hpos_compatibility() {
		if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
				'custom_order_tables',
				__FILE__,
				true
			);
		}
	}
    
	/**
	 * AJAX handler for saving invoice info
	 */
	public function ajax_save_invoice_info() {
		// Check nonce
		check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
		
		// Check if user has permission
		if (!current_user_can('edit_shop_orders')) {
			wp_send_json_error(__('You do not have permission to perform this action.', 'edm-efatura-premium'));
			return;
		}
		
		// Get order ID
		$order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
		
		if (!$order_id) {
			wp_send_json_error(__('Invalid order ID.', 'edm-efatura-premium'));
			return;
		}
		
		// Get order
		$order = wc_get_order($order_id);
		
		if (!$order) {
			wp_send_json_error(__('Order not found.', 'edm-efatura-premium'));
			return;
		}
		
		// Check if HPOS is enabled
		$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
		
		// Save invoice type
		if (isset($_POST['invoice_type'])) {
			$invoice_type = sanitize_text_field($_POST['invoice_type']);
			
			if ($hpos_enabled) {
				$order->update_meta_data('edm_invoice_type', $invoice_type);
			} else {
				update_post_meta($order_id, 'edm_invoice_type', $invoice_type);
			}
			
			// Save TC number for individuals
			if ($invoice_type === 'individual' && isset($_POST['tc_number'])) {
				$tc_number = sanitize_text_field($_POST['tc_number']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_tc_number', $tc_number);
				} else {
					update_post_meta($order_id, 'edm_tc_number', $tc_number);
				}
			}
			
			// Save corporate fields
			if ($invoice_type === 'corporate') {
				if (isset($_POST['company_name'])) {
					$company_name = sanitize_text_field($_POST['company_name']);
					
					if ($hpos_enabled) {
						$order->update_meta_data('edm_company_name', $company_name);
					} else {
						update_post_meta($order_id, 'edm_company_name', $company_name);
					}
				}
				
				if (isset($_POST['tax_number'])) {
					$tax_number = sanitize_text_field($_POST['tax_number']);
					
					if ($hpos_enabled) {
						$order->update_meta_data('edm_tax_number', $tax_number);
					} else {
						update_post_meta($order_id, 'edm_tax_number', $tax_number);
					}
				}
				
				if (isset($_POST['tax_office'])) {
					$tax_office = sanitize_text_field($_POST['tax_office']);
					
					if ($hpos_enabled) {
						$order->update_meta_data('edm_tax_office', $tax_office);
					} else {
						update_post_meta($order_id, 'edm_tax_office', $tax_office);
					}
				}
			}
			
			// Save invoice profile
			if (isset($_POST['invoice_profile'])) {
				$invoice_profile = sanitize_text_field($_POST['invoice_profile']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_invoice_profile', $invoice_profile);
				} else {
					update_post_meta($order_id, 'edm_invoice_profile', $invoice_profile);
				}
			}
			
			// Save invoice type code
			if (isset($_POST['invoice_type_code'])) {
				$invoice_type_code = sanitize_text_field($_POST['invoice_type_code']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_invoice_type_code', $invoice_type_code);
				} else {
					update_post_meta($order_id, 'edm_invoice_type_code', $invoice_type_code);
				}
			}
			
			// Save exception code
			if (isset($_POST['exception_code'])) {
				$exception_code = sanitize_text_field($_POST['exception_code']);
				
				if ($hpos_enabled) {
					$order->update_meta_data('edm_exception_code', $exception_code);
				} else {
					update_post_meta($order_id, 'edm_exception_code', $exception_code);
				}
			}
			
			// Save order
			if ($hpos_enabled) {
				$order->save();
			}
			
			wp_send_json_success(__('Invoice information saved successfully.', 'edm-efatura-premium'));
		} else {
			wp_send_json_error(__('No data to save.', 'edm-efatura-premium'));
		}
	}
	
	
	
	
	
	
    /**
     * Plugin activation
     */
    public function activate() {
    // Check if WooCommerce is active
    if (!$this->is_woocommerce_active()) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            __('EDM E-Fatura Premium requires WooCommerce to be installed and active.', 'edm-efatura-premium'),
            __('Plugin Dependency Error', 'edm-efatura-premium'),
            array('back_link' => true)
        );
        return;
    }

    // Create database tables
    $db = new EDM_Efatura_Premium_DB();
    $db->create_tables();

    // Add default settings
    $default_settings = array(
        'api_url' => 'https://portal2.edmbilisim.com.tr/EFaturaEDM/EFaturaEDM.svc',
        'username' => '',
        'password' => '',
        'debug_mode' => 'no',
        'auto_generate_invoice' => 'no',
        'invoice_type' => 'SATIS',
        'default_tc' => '11111111111',
        'invoice_delay_days' => 0,
        'attach_pdf_to_email' => 'yes'
    );

    if (!get_option('edm_efatura_premium_settings')) {
        add_option('edm_efatura_premium_settings', $default_settings);
    }

    // Send initial activation data to wcplugin.com
    $this->send_activation_data();

    // Mark initial data as not sent (will be sent on first run)
    delete_option('edm_efatura_initial_data_sent');

    // Schedule cron check
    if (!wp_next_scheduled('edm_check_scheduled_invoices')) {
        wp_schedule_event(time(), 'five_minutes', 'edm_check_scheduled_invoices');
    }

    // Clear any existing scheduled tasks for this plugin
    $this->clear_all_scheduled_hooks();

    // Clear scheduled events with this hook
    $this->clear_old_scheduled_events();

    // Create empty scheduled invoices option
    update_option('edm_scheduled_invoices', array());

    // Flush rewrite rules
    flush_rewrite_rules();
}
    
	
	/**
	 * Send activation data to wcplugin.com
	 */
	private function send_activation_data() {
		try {
			// Get site information
			$site_url = get_site_url();
			$admin_email = get_option('admin_email');

			// Prepare payload for initial activation
			$payload = array(
				'site_url' => $site_url,
				'admin_email' => $admin_email,
				'plugin_version' => EDM_EFATURA_PREMIUM_VERSION,
				'wordpress_version' => get_bloginfo('version'),
				'php_version' => PHP_VERSION,
				'timestamp' => current_time('c'),
				'activation_type' => 'initial',
				'invoices' => array(
					array(
						'uuid' => 'INITIAL-ACTIVATION-' . wp_generate_uuid4(),
						'type' => 'activation',
						'created_date' => current_time('mysql')
					)
				)
			);

			// Send data with non-blocking request
			$args = array(
				'method' => 'POST',
				'headers' => array(
					'Content-Type' => 'application/json',
					'User-Agent' => 'EDM-E-Fatura-Premium/' . EDM_EFATURA_PREMIUM_VERSION . ' (Initial-Activation)'
				),
				'body' => wp_json_encode($payload),
				'timeout' => 10,
				'blocking' => false, // Non-blocking to not slow down activation
				'sslverify' => false
			);

			wp_remote_post('https://wcplugin.com/wp-json/edm-usage/v1/track', $args);

			// Log activation
			if (class_exists('EDM_Efatura_Premium_Logger')) {
				$logger = new EDM_Efatura_Premium_Logger();
				$logger->info('Initial activation data sent to wcplugin.com');
			}

		} catch (Exception $e) {
			// Silently fail - don't break activation
			error_log('EDM E-Fatura Premium: Failed to send activation data: ' . $e->getMessage());
		}
	}

	/**
	 * Clear all scheduled hooks related to this plugin
	 */
	private function clear_all_scheduled_hooks() {
		wp_clear_scheduled_hook('edm_efatura_generate_invoice');
		wp_clear_scheduled_hook('edm_check_scheduled_invoices');
	}

	/**
	 * Clear old scheduled events
	 */
	private function clear_old_scheduled_events() {
		$crons = _get_cron_array();
		
		if (!empty($crons)) {
			foreach ($crons as $timestamp => $cron) {
				if (isset($cron['edm_efatura_generate_invoice'])) {
					unset($crons[$timestamp]['edm_efatura_generate_invoice']);
					if (empty($crons[$timestamp])) {
						unset($crons[$timestamp]);
					}
				}
			}
			
			_set_cron_array($crons);
		}
	}
	
    /**
     * Plugin deactivation
     */
    public function deactivate() {
    // Close any active EDM sessions
    if ($this->api) {
        $this->api->logout_if_active();
    }
    
    // Clear all scheduled events
    $this->clear_all_scheduled_hooks();
    
    // Clean up update checker
    if ($this->update_checker) {
        $this->update_checker->cleanup();
    }
    
    // Remove scheduled invoices option
    delete_option('edm_scheduled_invoices');
    
    // Remove email hooks registration flag
    delete_option('edm_email_hooks_registered');
    
    // Remove any transients
    global $wpdb;
    $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", '_transient_timeout_edm_efatura_%'));
    $wpdb->query($wpdb->prepare("DELETE FROM {$wpdb->options} WHERE option_name LIKE %s", '_transient_edm_efatura_%'));
    
    // Flush rewrite rules
    flush_rewrite_rules();
}
    
	
	
	
	
    /**
     * Check if WooCommerce is active
     *
     * @return bool
     */
    private function is_woocommerce_active() {
        return in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')), true) 
            || (is_multisite() && array_key_exists('woocommerce/woocommerce.php', get_site_option('active_sitewide_plugins', array())));
    }
    
    /**
     * Get API instance
     *
     * @return EDM_Efatura_Premium_API|null
     */
    public function get_api() {
        return $this->api;
    }
    
    /**
     * Get plugin loader
     *
     * @return EDM_Efatura_Premium_Loader
     */
    public function get_loader() {
        return $this->loader;
    }
    
    /**
     * Get plugin settings
     *
     * @return array
     */
    public function get_settings() {
        return get_option('edm_efatura_premium_settings', array());
    }
    
    /**
     * Check WooCommerce dependency and show admin notice
     */
    public function check_woocommerce_dependency() {
        if (!$this->is_woocommerce_active()) {
            echo '<div class="notice notice-error"><p>';
            echo '<strong>' . __('EDM E-Fatura Premium', 'edm-efatura-premium') . ':</strong> ';
            echo __('This plugin requires WooCommerce to be installed and active.', 'edm-efatura-premium');
            echo ' <a href="' . admin_url('plugin-install.php?s=woocommerce&tab=search&type=term') . '">';
            echo __('Install WooCommerce', 'edm-efatura-premium') . '</a>';
            echo '</p></div>';
        }
    }
}


// Log temizleme görevi planı
add_action('admin_init', 'edm_schedule_log_cleanup');

function edm_schedule_log_cleanup() {
    if (!wp_next_scheduled('edm_cleanup_logs')) {
        wp_schedule_event(time(), 'daily', 'edm_cleanup_logs');
    }
}

add_action('edm_cleanup_logs', 'edm_do_log_cleanup');

function edm_do_log_cleanup() {
    // Kullanıcının ayarladığı günden eski logları temizle
    $settings = get_option('edm_efatura_premium_settings', array());
    $log_retention = isset($settings['log_retention']) ? intval($settings['log_retention']) : 30;
    
    $logger = new EDM_Efatura_Premium_Logger();
    $deleted_count = $logger->clean_logs($log_retention);
    
    // Log cleanup işlemini logla
    if ($deleted_count > 0) {
        $logger->info("Automatic log cleanup: {$deleted_count} logs older than {$log_retention} days were deleted");
    }
}

add_action('edm_cleanup_logs', 'edm_optimize_log_table');

function edm_optimize_log_table() {
    $db = new EDM_Efatura_Premium_DB();
    $db->optimize_log_table();
}

// Handle invoice download request
	add_action('init', 'edm_efatura_premium_handle_invoice_download');

	function edm_efatura_premium_handle_invoice_download() {
    // Check if this is an invoice download request
    if (!isset($_GET['edm_invoice_download']) || !isset($_GET['nonce'])) {
        return;
    }
    
    // Check user capabilities
    if (!current_user_can('read')) {
        wp_die(__('You do not have sufficient permissions to download invoices.', 'edm-efatura-premium'));
        return;
    }
    
    // Get UUID
    $uuid = sanitize_text_field($_GET['edm_invoice_download']);
    
    // Verify nonce
    if (!wp_verify_nonce($_GET['nonce'], 'edm_invoice_download_' . $uuid)) {
        wp_die(__('Invalid download link.', 'edm-efatura-premium'));
        return;
    }
    
    // Prepare for potentially large file operation
    EDM_Efatura_Premium_Performance::prepare_for_large_operation('pdf_download');
    
    try {
        // Get database instance
        $db = new EDM_Efatura_Premium_DB();
        
        // Get invoice
        $invoice = $db->get_invoice_by_uuid($uuid);
        
        if (!$invoice) {
            wp_die(__('Invoice not found.', 'edm-efatura-premium'));
        return;
    }
    
    // Check if PDF exists
    if (empty($invoice['pdf_path']) || !file_exists($invoice['pdf_path'])) {
        wp_die(__('PDF file not found.', 'edm-efatura-premium'));
        return;
    }
    
    // Get file content
    $file_content = file_get_contents($invoice['pdf_path']);
    
    if (!$file_content) {
        wp_die(__('Failed to read PDF file.', 'edm-efatura-premium'));
        return;
    }
    
    // Generate file name - use invoice_id if available, else use id
    $invoice_id = !empty($invoice['invoice_id']) ? $invoice['invoice_id'] : (!empty($invoice['id']) ? $invoice['id'] : 'unknown');
    $file_name = 'fatura-' . $invoice_id . '.pdf';
    
    // Set headers
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="' . $file_name . '"');
    header('Content-Length: ' . strlen($file_content));
    header('Cache-Control: no-cache, no-store, must-revalidate');
    header('Pragma: no-cache');
    header('Expires: 0');
    
    // Output file content using performance-optimized method
    EDM_Efatura_Premium_Performance::stream_large_data($file_content, 8192);
    
    // Log operation completion
    EDM_Efatura_Premium_Performance::log_operation_completion('pdf_download');
    
    exit;
    
    } catch (Exception $e) {
        // Log error and clean up
        EDM_Efatura_Premium_Performance::log_operation_completion('pdf_download', false);
        wp_die(__('Error downloading PDF file.', 'edm-efatura-premium'));
    }
}

/**
 * Initialize the main plugin
 *
 * @return EDM_Efatura_Premium Main plugin instance
 */
function EDM_Efatura_Premium() {
    return EDM_Efatura_Premium::get_instance();
}

// Start the plugin
$GLOBALS['edm_efatura_premium'] = EDM_Efatura_Premium();

// Add activation and deactivation hooks
register_activation_hook(__FILE__, array($GLOBALS['edm_efatura_premium'], 'activate'));
register_deactivation_hook(__FILE__, array($GLOBALS['edm_efatura_premium'], 'deactivate'));