<?php
/**
 * EDM E-Fatura Premium API Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling API communication with EDM E-Fatura system
 */
class EDM_Efatura_Premium_API {
    /**
     * EDM API URL
     *
     * @var string
     */
    private $api_url;
    
    /**
     * EDM username
     *
     * @var string
     */
    private $username;
    
    /**
     * EDM password
     *
     * @var string
     */
    private $password;
    
    /**
     * Debug mode
     *
     * @var bool
     */
    private $debug = false;
    
    /**
     * Active session ID
     *
     * @var string|null
     */
    private $session_id = null;
    
    /**
     * Logger instance
     *
     * @var EDM_Efatura_Premium_Logger
     */
    private $logger;
    
    /**
     * Constructor
     *
     * @param string $api_url API URL
     * @param string $username Username
     * @param string $password Password
     * @param bool $debug Debug mode
     */
    public function __construct($api_url, $username, $password, $debug = false, $edm_email = '') {
		$this->api_url = rtrim($api_url, '/');
		$this->username = $username;
		$this->password = $password;
		$this->debug = $debug;
		$this->edm_email = $edm_email;
		$this->logger = new EDM_Efatura_Premium_Logger();
	}
    
    /**
     * Login to EDM system
     *
     * @return string|false Session ID on success, false on failure
     */
    public function login() {
		// Current date and time
		$now = date('Y-m-d\TH:i:s.uP');
		$client_txn_id = $this->generate_uuid();
		
		// Create login request
		$xml = '<?xml version="1.0" encoding="utf-8"?>
		<s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
			<s:Body>
				<LoginRequest xmlns="http://tempuri.org/">
					<REQUEST_HEADER xmlns="">
						<SESSION_ID>0</SESSION_ID>
						<CLIENT_TXN_ID>' . $client_txn_id . '</CLIENT_TXN_ID>
						<ACTION_DATE>' . $now . '</ACTION_DATE>
						<REASON>' . __('E-Invoice operations', 'edm-efatura-premium') . '</REASON>
						<APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
						<HOSTNAME>' . (function_exists('gethostname') ? gethostname() : 'localhost') . '</HOSTNAME>
						<CHANNEL_NAME>WP_PLUGIN</CHANNEL_NAME>
						<COMPRESSED>N</COMPRESSED>
					</REQUEST_HEADER>
					<USER_NAME xmlns="">' . $this->username . '</USER_NAME>
					<PASSWORD xmlns="">' . $this->password . '</PASSWORD>
					<SECRET_KEY xmlns=""></SECRET_KEY>
				</LoginRequest>
			</s:Body>
		</s:Envelope>';
		
		$this->log_debug('Login Request: ' . $xml);
		
		// Send SOAP request
		$headers = array(
			'Content-Type: text/xml; charset=utf-8',
			'SOAPAction: "LoginRequest"',
			'Content-Length: ' . strlen($xml)
		);
		
		try {
			$response = $this->send_soap_request($xml, $headers);
			
			$this->log_debug('Login Response: ' . $response);
			
			// Process response
			$session_id = $this->extract_value_from_xml($response, 'SESSION_ID');
			
			if ($session_id) {
				$this->session_id = $session_id;
				return $session_id;
			}
			
			// Check for error
			$error_code = $this->extract_value_from_xml($response, 'ERROR_CODE');
			$error_desc = $this->extract_value_from_xml($response, 'ERROR_SHORT_DES');
			
			if ($error_code) {
				$error_message = sprintf(__('EDM Login Error: %1$s (Code: %2$s)', 'edm-efatura-premium'), $error_desc, $error_code);
				$this->logger->error($error_message);
				throw new Exception($error_message);
			}
			
			// Generic login failure if no specific error was found
			$this->logger->error('EDM Login failed without specific error code');
			return false;
		}
		catch (Exception $e) {
			$this->logger->error('EDM Login error: ' . $e->getMessage());
			throw $e;
		}
	}
    
    /**
     * Get invoices
     *
     * @param string $start_date Start date (Y-m-d format)
     * @param string $end_date End date (Y-m-d format)
     * @param string $direction Direction (IN, OUT, OUT-EINVOICE, OUT-EARCHIVE)
     * @param int $limit Maximum number of invoices (default: 10)
     * @return array Invoice list
     */
    public function get_invoices($start_date, $end_date, $direction = 'IN', $limit = 10) {
        // Format dates correctly
        $start_datetime = date('Y-m-d\T00:00:00.000P', strtotime($start_date));
        $end_datetime = date('Y-m-d\T23:59:59.999P', strtotime($end_date));
        
        // Ensure we have a session
        $session_id = $this->ensure_session();
        
        // Current date and time
        $now = date('Y-m-d\TH:i:s.uP');
        $client_txn_id = $this->generate_uuid();
        
        // Create GetInvoice request
        $xml = '<?xml version="1.0" encoding="utf-8"?>
        <s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
            <s:Body>
                <GetInvoiceRequest xmlns="http://tempuri.org/">
                    <REQUEST_HEADER xmlns="">
                        <SESSION_ID>' . $session_id . '</SESSION_ID>
                        <CLIENT_TXN_ID>' . $client_txn_id . '</CLIENT_TXN_ID>
                        <ACTION_DATE>' . $now . '</ACTION_DATE>
                        <REASON>' . __('E-Invoice operations', 'edm-efatura-premium') . '</REASON>
                        <APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
                        <HOSTNAME>' . (function_exists('gethostname') ? gethostname() : 'localhost') . '</HOSTNAME>
                        <CHANNEL_NAME>WP_PLUGIN</CHANNEL_NAME>
                        <COMPRESSED>N</COMPRESSED>
                    </REQUEST_HEADER>
                    <INVOICE_SEARCH_KEY xmlns="">
                        <READ_INCLUDED>false</READ_INCLUDED>
                        <DIRECTION>' . $direction . '</DIRECTION>
                        <CR_START_DATE>' . $start_datetime . '</CR_START_DATE>
                        <CR_END_DATE>' . $end_datetime . '</CR_END_DATE>
                        <LIMIT>' . $limit . '</LIMIT>
                        <ISARCHIVED>false</ISARCHIVED>
                    </INVOICE_SEARCH_KEY>
                    <HEADER_ONLY xmlns="">Y</HEADER_ONLY>
                    <INVOICE_CONTENT_TYPE xmlns="">XML</INVOICE_CONTENT_TYPE>
                </GetInvoiceRequest>
            </s:Body>
        </s:Envelope>';
        
        $this->log_debug('GetInvoice Request: ' . $xml);
        
        // Send SOAP request
        $headers = array(
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: "GetInvoiceRequest"',
            'Content-Length: ' . strlen($xml)
        );
        
        $response = $this->send_soap_request($xml, $headers);
        
        $this->log_debug('GetInvoice Response: ' . substr($response, 0, 1000) . '...');
        
        // Parse and return invoices
        $invoices = $this->parse_invoices_from_xml($response);
        
        return $invoices;
    }
    
    /**
     * Get invoice status
     *
     * @param string $invoice_uuid Invoice UUID
     * @return array Invoice status
     */
    public function get_invoice_status($invoice_uuid, $start_date = null, $end_date = null) {
		// Ensure we have a session
		$session_id = $this->ensure_session();
		
		// Current date and time
		$now = date('Y-m-d\TH:i:s.uP');
		$client_txn_id = $this->generate_uuid();
		
		// Create GetInvoiceStatus request
		$xml = '<?xml version="1.0" encoding="utf-8"?>
		<s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
			<s:Body>
				<GetInvoiceStatusRequest xmlns="http://tempuri.org/">
					<REQUEST_HEADER xmlns="">
						<SESSION_ID>' . $session_id . '</SESSION_ID>
						<CLIENT_TXN_ID>' . $client_txn_id . '</CLIENT_TXN_ID>
						<ACTION_DATE>' . $now . '</ACTION_DATE>
						<REASON>' . __('E-Invoice operations', 'edm-efatura-premium') . '</REASON>
						<APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
						<HOSTNAME>' . (function_exists('gethostname') ? gethostname() : 'localhost') . '</HOSTNAME>
						<CHANNEL_NAME>WP_PLUGIN</CHANNEL_NAME>
						<COMPRESSED>N</COMPRESSED>
					</REQUEST_HEADER>
					<INVOICE TRXID="0" UUID="' . $invoice_uuid . '" xmlns=""/>';
		
		// Add date range if provided
		if ($start_date && $end_date) {
			$start_datetime = date('Y-m-d\T00:00:00.000P', strtotime($start_date));
			$end_datetime = date('Y-m-d\T23:59:59.999P', strtotime($end_date));
			
			$xml .= '
					<DATE_RANGE xmlns="">
						<START_DATE>' . $start_datetime . '</START_DATE>
						<END_DATE>' . $end_datetime . '</END_DATE>
					</DATE_RANGE>';
		}
		
		$xml .= '
				</GetInvoiceStatusRequest>
			</s:Body>
		</s:Envelope>';
		
		$this->log_debug('GetInvoiceStatus Request: ' . $xml);
		
		// Send SOAP request
		$headers = array(
			'Content-Type: text/xml; charset=utf-8',
			'SOAPAction: "GetInvoiceStatusRequest"',
			'Content-Length: ' . strlen($xml)
		);
		
		$response = $this->send_soap_request($xml, $headers);
		
		$this->log_debug('GetInvoiceStatus Response: ' . $response);
		
		// Process response
		$status = array(
			'id' => $this->extract_value_from_xml($response, 'ID'),
			'uuid' => $invoice_uuid,
			'status' => $this->extract_value_from_xml($response, 'STATUS'),
			'status_description' => $this->extract_value_from_xml($response, 'STATUS_DESCRIPTION'),
			'gib_status_code' => $this->extract_value_from_xml($response, 'GIB_STATUS_CODE'),
			'gib_status_description' => $this->extract_value_from_xml($response, 'GIB_STATUS_DESCRIPTION'),
			'creation_date' => $this->extract_value_from_xml($response, 'CDATE')
		);
		
		return $status;
	}
    
    /**
     * Cancel invoice (supports both e-Invoice and e-Archive)
     *
     * @param string $invoice_uuid Invoice UUID
     * @return array Cancel result
     */
    public function cancel_invoice($invoice_uuid) {
        // Ensure we have a session
        $session_id = $this->ensure_session();
        
        // Current date and time
        $now = date('Y-m-d\TH:i:s.uP');
        $client_txn_id = $this->generate_uuid();
        
        // Get invoice type from database to determine cancel method
        $db = new EDM_Efatura_Premium_DB();
        $invoice = $db->get_invoice_by_uuid($invoice_uuid);
        $is_archive = false;
        
        if ($invoice) {
            // Check if this is an e-Archive invoice
            // e-Archive invoices are usually identified by profile or specific settings
            $settings = get_option('edm_efatura_premium_settings', array());
            $invoice_profile = isset($settings['invoice_profile']) ? $settings['invoice_profile'] : 'EARSIVFATURA';
            
            // If profile is e-Archive or direction is outgoing to individual customers
            $is_archive = ($invoice_profile === 'EARSIVFATURA') || 
                         (isset($invoice['direction']) && $invoice['direction'] === 'OUT-EARCHIVE');
        }
        
        // Prepare invoice element based on type
        if ($is_archive) {
            // For e-Archive: use invoice_id (invoice number) - more reliable for e-Archive
            $invoice_element = '<INVOICE ID="' . ($invoice['invoice_id'] ?? $invoice_uuid) . '" xmlns=""/>';
            $this->log_debug('Cancelling e-Archive invoice with ID: ' . ($invoice['invoice_id'] ?? $invoice_uuid));
        } else {
            // For e-Invoice: use UUID - standard method
            $invoice_element = '<INVOICE TRXID="0" UUID="' . $invoice_uuid . '" xmlns=""/>';
            $this->log_debug('Cancelling e-Invoice with UUID: ' . $invoice_uuid);
        }
        
        // Create CancelInvoice request
        $xml = '<?xml version="1.0" encoding="utf-8"?>
        <s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
            <s:Body>
                <CancelInvoiceRequest xmlns="http://tempuri.org/">
                    <REQUEST_HEADER xmlns="">
                        <SESSION_ID>' . $session_id . '</SESSION_ID>
                        <CLIENT_TXN_ID>' . $client_txn_id . '</CLIENT_TXN_ID>
                        <ACTION_DATE>' . $now . '</ACTION_DATE>
                        <REASON>' . __('E-Invoice operations', 'edm-efatura-premium') . '</REASON>
                        <APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
                        <HOSTNAME>' . (function_exists('gethostname') ? gethostname() : 'localhost') . '</HOSTNAME>
                        <CHANNEL_NAME>WP_PLUGIN</CHANNEL_NAME>
                        <COMPRESSED>N</COMPRESSED>
                    </REQUEST_HEADER>
                    ' . $invoice_element . '
                </CancelInvoiceRequest>
            </s:Body>
        </s:Envelope>';
        
        $this->log_debug('CancelInvoice Request: ' . $xml);
        
        // Send SOAP request
        $headers = array(
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: "CancelInvoiceRequest"',
            'Content-Length: ' . strlen($xml)
        );
        
        $response = $this->send_soap_request($xml, $headers);
        
        $this->log_debug('CancelInvoice Response: ' . $response);
        
        // Process response
        $result = array(
            'success' => false,
            'message' => ''
        );
        
        // Check success status
        $return_code = $this->extract_value_from_xml($response, 'RETURN_CODE');
        
        if ($return_code === '0') {
            $result['success'] = true;
            $result['message'] = __('Invoice successfully cancelled.', 'edm-efatura-premium');
            
            // Log success
            $this->logger->info('Invoice cancelled successfully', array('uuid' => $invoice_uuid));
        } else {
            // Check for error
            $error_code = $this->extract_value_from_xml($response, 'ERROR_CODE');
            $error_desc = $this->extract_value_from_xml($response, 'ERROR_SHORT_DES');
            
            if ($error_code) {
                $result['message'] = sprintf(__('EDM CancelInvoice Error: %1$s (Code: %2$s)', 'edm-efatura-premium'), $error_desc, $error_code);
            } else {
                $result['message'] = __('EDM CancelInvoice Error: Unknown error occurred.', 'edm-efatura-premium');
            }
            
            // Log error
            $this->logger->error($result['message']);
        }
        
        return $result;
    }
    
    /**
     * Check user by VKN/TCKN with caching
     *
     * @param string $identifier VKN/TCKN
     * @return array User information
     */
    public function check_user($identifier) {
        // Check cache first
        $cached_result = EDM_Efatura_Premium_Cache::get_cached_api_response('check_user', array($identifier));
        if ($cached_result !== false) {
            return $cached_result;
        }
        
        // Ensure we have a session
        $session_id = $this->ensure_session();
        
        // Current date and time
        $now = date('Y-m-d\TH:i:s.uP');
        $client_txn_id = $this->generate_uuid();
        
        // Create CheckUser request
        $xml = '<?xml version="1.0" encoding="utf-8"?>
        <s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
            <s:Body>
                <CheckUserRequest xmlns="http://tempuri.org/">
                    <REQUEST_HEADER xmlns="">
                        <SESSION_ID>' . $session_id . '</SESSION_ID>
                        <CLIENT_TXN_ID>' . $client_txn_id . '</CLIENT_TXN_ID>
                        <ACTION_DATE>' . $now . '</ACTION_DATE>
                        <REASON>' . __('E-Invoice operations', 'edm-efatura-premium') . '</REASON>
                        <APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
                        <HOSTNAME>' . (function_exists('gethostname') ? gethostname() : 'localhost') . '</HOSTNAME>
                        <CHANNEL_NAME>WP_PLUGIN</CHANNEL_NAME>
                        <COMPRESSED>N</COMPRESSED>
                    </REQUEST_HEADER>
                    <USER xmlns="">
                        <IDENTIFIER>' . $identifier . '</IDENTIFIER>
                    </USER>
                </CheckUserRequest>
            </s:Body>
        </s:Envelope>';
        
        $this->log_debug('CheckUser Request: ' . $xml);
        
        // Send SOAP request
        $headers = array(
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: "CheckUserRequest"',
            'Content-Length: ' . strlen($xml)
        );
        
        try {
            $response = $this->send_soap_request($xml, $headers);
            
            $this->log_debug('CheckUser Response: ' . $response);
            
            // Parse user info
            $user = array();
            
            // Find all USER elements
            if (preg_match_all('/<USER[^>]*>(.*?)<\/USER>/s', $response, $users)) {
                foreach ($users[1] as $user_xml) {
                    $user['identifier'] = $this->extract_value_from_xml($user_xml, 'IDENTIFIER');
                    $user['alias'] = $this->extract_value_from_xml($user_xml, 'ALIAS');
                    $user['title'] = $this->extract_value_from_xml($user_xml, 'TITLE');
                    $user['type'] = $this->extract_value_from_xml($user_xml, 'TYPE');
                    $user['register_time'] = $this->extract_value_from_xml($user_xml, 'REGISTER_TIME');
                    $user['unit'] = $this->extract_value_from_xml($user_xml, 'UNIT');
                    $user['alias_creation_time'] = $this->extract_value_from_xml($user_xml, 'ALIAS_CREATION_TIME');
                    
                    // If we found at least identifier, we have valid data
                    if (!empty($user['identifier'])) {
                        break;
                    }
                }
            }
            
            // Check for error
            $error_code = $this->extract_value_from_xml($response, 'ERROR_CODE');
            $error_desc = $this->extract_value_from_xml($response, 'ERROR_SHORT_DES');
            
            if ($error_code) {
                $error_message = sprintf(__('EDM CheckUser Error: %1$s (Code: %2$s)', 'edm-efatura-premium'), $error_desc, $error_code);
                $this->logger->error($error_message);
                throw new Exception($error_message);
            }
            
            // Cache successful result for 1 hour
            EDM_Efatura_Premium_Cache::cache_api_response('check_user', array($identifier), $user, 3600);
            
            return $user;
        } catch (Exception $e) {
            $this->logger->error('CheckUser error: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Send invoice
     *
     * @param string $receiver_vkn Receiver VKN
     * @param string $receiver_alias Receiver alias
     * @param string $invoice_content Invoice content (XML)
     * @return array Send result
     */
    public function send_invoice($receiver_vkn, $receiver_alias, $invoice_content) {
		// Ensure we have a session
		$session_id = $this->ensure_session();
		
		// Current date and time
		$now = date('Y-m-d\TH:i:s.uP');
		$client_txn_id = $this->generate_uuid();
		
		// Base64 encode content
		$content_base64 = base64_encode($invoice_content);
		
		// Get sender alias from settings
		$settings = get_option('edm_efatura_premium_settings', array());
		$sender_alias = '';
		
		if (!empty($settings['edm_email'])) {
			$email_parts = explode('@', $settings['edm_email']);
			if (count($email_parts) === 2) {
				$username = $email_parts[0];
				$domain = $email_parts[1];
				$sender_alias = 'urn:mail:' . $username . '@' . $domain;
			}
		}
		
		// If no alias found in settings, extract from invoice content
		if (empty($sender_alias)) {
			$sender_alias = $this->extract_sender_alias($invoice_content);
		}
		
		// Fix receiver alias format if needed
		if (strpos($receiver_alias, ',') !== false) {
			$aliases = explode(',', $receiver_alias);
			$receiver_alias = trim($aliases[0]); // Use only first alias
		}
		
		// Convert email to GIB postbox format if it's a plain email
		if (!empty($receiver_alias) && strpos($receiver_alias, 'urn:mail:') !== 0 && filter_var($receiver_alias, FILTER_VALIDATE_EMAIL)) {
			$email_parts = explode('@', $receiver_alias);
			if (count($email_parts) === 2) {
				$receiver_alias = 'urn:mail:' . $email_parts[0] . '@' . $email_parts[1];
			}
		}
		
		// Create SendInvoice request
		$xml = '<?xml version="1.0" encoding="utf-8"?>
		<s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
			<s:Body>
				<SendInvoiceRequest xmlns="http://tempuri.org/">
					<REQUEST_HEADER xmlns="">
						<SESSION_ID>' . $session_id . '</SESSION_ID>
						<CLIENT_TXN_ID>' . $client_txn_id . '</CLIENT_TXN_ID>
						<ACTION_DATE>' . $now . '</ACTION_DATE>
						<REASON>' . __('E-Invoice operations', 'edm-efatura-premium') . '</REASON>
						<APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
						<HOSTNAME>' . (function_exists('gethostname') ? gethostname() : 'localhost') . '</HOSTNAME>
						<CHANNEL_NAME>WP_PLUGIN</CHANNEL_NAME>
						<COMPRESSED>N</COMPRESSED>
					</REQUEST_HEADER>
					<SENDER vkn="' . (isset($settings['company_vkn']) ? $settings['company_vkn'] : '') . '" 
							alias="' . $sender_alias . '" xmlns=""/>
					<RECEIVER vkn="' . $receiver_vkn . '"' . 
							(!empty($receiver_alias) ? ' alias="' . $receiver_alias . '"' : '') . ' xmlns=""/>
					<INVOICE TRXID="0" xmlns="">
						<HEADER>
							<SENDER>' . $this->extract_sender_vkn($invoice_content) . '</SENDER>
							<RECEIVER>' . $receiver_vkn . '</RECEIVER>
							<FROM>' . $sender_alias . '</FROM>
							<TO>' . $receiver_alias . '</TO>
							<INTERNETSALES>false</INTERNETSALES>
							<EARCHIVE>false</EARCHIVE>
							<EARCHIVE_REPORT_SENDDATE>0001-01-01</EARCHIVE_REPORT_SENDDATE>
							<CANCEL_EARCHIVE_REPORT_SENDDATE>0001-01-01</CANCEL_EARCHIVE_REPORT_SENDDATE>
						</HEADER>
						<CONTENT>' . $content_base64 . '</CONTENT>
					</INVOICE>
				</SendInvoiceRequest>
			</s:Body>
		</s:Envelope>';
		
		$this->log_debug('SendInvoice Request: ' . substr($xml, 0, 1000) . '...');
		
		// Send SOAP request
		$headers = array(
			'Content-Type: text/xml; charset=utf-8',
			'SOAPAction: "SendInvoiceRequest"',
			'Content-Length: ' . strlen($xml)
		);
		
		$response = $this->send_soap_request($xml, $headers);
		
		$this->log_debug('SendInvoice Response: ' . $response);
		
		// Process response
		$result = array(
			'success' => false,
			'message' => '',
			'trx_id' => '',
			'uuid' => ''
		);
		
		// Check success status
		$return_code = $this->extract_value_from_xml($response, 'RETURN_CODE');
		
		if ($return_code === '0') {
			$result['success'] = true;
			$result['trx_id'] = $this->extract_value_from_xml($response, 'INTL_TXN_ID');
			$result['uuid'] = $this->extract_value_from_xml($response, 'UUID');
			$result['invoice_id'] = $this->extract_value_from_xml($response, 'ID');
			$result['status'] = $this->extract_value_from_xml($response, 'STATUS');
			$result['status_description'] = $this->extract_value_from_xml($response, 'STATUS_DESCRIPTION');
			$result['message'] = sprintf(__('Invoice successfully sent. UUID: %s', 'edm-efatura-premium'), $result['uuid']);
			
			// Log success
			$this->logger->info('Invoice sent successfully', $result);
		} else {
			// Check for error
			$error_code = $this->extract_value_from_xml($response, 'ERROR_CODE');
			$error_desc = $this->extract_value_from_xml($response, 'ERROR_SHORT_DES');
			
			if ($error_code) {
				$result['message'] = sprintf(__('EDM SendInvoice Error: %1$s (Code: %2$s)', 'edm-efatura-premium'), $error_desc, $error_code);
			} else {
				$result['message'] = __('EDM SendInvoice Error: Unknown error occurred.', 'edm-efatura-premium');
			}
			
			// Log error
			$this->logger->error($result['message']);
		}
		
		return $result;
	}
    
	
	/**
	 * Get session ID
	 *
	 * @return string|null Session ID or null
	 */
	public function get_session_id() {
		return $this->session_id;
	}
	
    
    
    /**
     * Ensure an active session exists
     *
     * @return string Session ID
     */
    private function ensure_session() {
		if (!$this->session_id) {
			try {
				$this->session_id = $this->login();
				
				if (!$this->session_id) {
					$this->logger->error('Failed to establish EDM session with provided credentials.');
					throw new Exception(__('Failed to establish EDM session. Please check your API credentials.', 'edm-efatura-premium'));
				}
				
				// Log successful session
				$this->logger->info('Successfully established EDM session.');
			} catch (Exception $e) {
				$this->logger->error('Error during session establishment: ' . $e->getMessage());
				throw new Exception(__('Failed to establish EDM session: ', 'edm-efatura-premium') . $e->getMessage());
			}
		}
		
		return $this->session_id;
	}
    
    /**
     * Logout if session is active
     */
    public function logout_if_active() {
        if ($this->session_id) {
            try {
                $this->logout($this->session_id);
            } catch (Exception $e) {
                $this->logger->error(__('Error during logout: ', 'edm-efatura-premium') . $e->getMessage());
            }
        }
    }
    
    /**
     * Logout from EDM system
     *
     * @param string $session_id Session ID
     * @return bool Success status
     */
    public function logout($session_id) {
        // Current date and time
        $now = date('Y-m-d\TH:i:s.uP');
        $client_txn_id = $this->generate_uuid();
        
        // Create logout request
        $xml = '<?xml version="1.0" encoding="utf-8"?>
        <s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
            <s:Body>
                <LogoutRequest xmlns="http://tempuri.org/">
                    <REQUEST_HEADER xmlns="">
                        <SESSION_ID>' . $session_id . '</SESSION_ID>
                        <CLIENT_TXN_ID>' . $client_txn_id . '</CLIENT_TXN_ID>
                        <ACTION_DATE>' . $now . '</ACTION_DATE>
                        <REASON>' . __('E-Invoice operations', 'edm-efatura-premium') . '</REASON>
                        <APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
                        <HOSTNAME>' . (function_exists('gethostname') ? gethostname() : 'localhost') . '</HOSTNAME>
                        <CHANNEL_NAME>WP_PLUGIN</CHANNEL_NAME>
                        <COMPRESSED>N</COMPRESSED>
                    </REQUEST_HEADER>
                </LogoutRequest>
            </s:Body>
        </s:Envelope>';
        
        $this->log_debug('Logout Request: ' . $xml);
        
        // Send SOAP request
        $headers = array(
            'Content-Type: text/xml; charset=utf-8',
            'SOAPAction: "LogoutRequest"',
            'Content-Length: ' . strlen($xml)
        );
        
        $response = $this->send_soap_request($xml, $headers);
        
        $this->log_debug('Logout Response: ' . $response);
        
        // Process response
        $return_code = $this->extract_value_from_xml($response, 'RETURN_CODE');
        
        if ($return_code === '0') {
            // Clear session ID if it matches
            if ($this->session_id === $session_id) {
                $this->session_id = null;
            }
            return true;
        }
        
        // Check for error
        $error_code = $this->extract_value_from_xml($response, 'ERROR_CODE');
        $error_desc = $this->extract_value_from_xml($response, 'ERROR_SHORT_DES');
        
        if ($error_code) {
            $error_message = sprintf(__('EDM Logout Error: %1$s (Code: %2$s)', 'edm-efatura-premium'), $error_desc, $error_code);
            $this->logger->error($error_message);
            throw new Exception($error_message);
        }
        
        return false;
    }
    
    /**
     * Persistent CURL handle for connection reuse
     */
    private static $curl_handle = null;
    
    /**
     * Get or create persistent CURL handle
     *
     * @return resource CURL handle
     */
    private function get_curl_handle() {
        if (self::$curl_handle === null) {
            self::$curl_handle = curl_init();
        }
        return self::$curl_handle;
    }
    
    /**
     * Send SOAP request with connection pooling
     *
     * @param string $xml SOAP XML content
     * @param array $headers HTTP headers
     * @param int $timeout Timeout in seconds
     * @return string SOAP response
     * @throws Exception on error
     */
    private function send_soap_request($xml, $headers, $timeout = 30) {
        $ch = $this->get_curl_handle();
        
        // Reset handle for new request
        curl_reset($ch);
        
        curl_setopt($ch, CURLOPT_URL, $this->api_url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $xml);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, $timeout);
        curl_setopt($ch, CURLOPT_TIMEOUT, $timeout);
        
        // Enable connection reuse
        curl_setopt($ch, CURLOPT_FORBID_REUSE, false);
        curl_setopt($ch, CURLOPT_FRESH_CONNECT, false);
        
        // Send request
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curl_error = curl_error($ch);
        $curl_errno = curl_errno($ch);
        
        // Don't close the handle, keep it for reuse
        // curl_close($ch) is removed for connection pooling
        
        // Check for errors
        if ($response === false) {
            $error_message = sprintf(
                __('CURL Error %1$d: %2$s', 'edm-efatura-premium'), 
                $curl_errno,
                $curl_error
            );
            $this->logger->error($error_message);
            throw new Exception($error_message);
        }
        
        if ($http_code != 200) {
            $error_message = sprintf(
                __('HTTP Error: %1$s, Response: %2$s', 'edm-efatura-premium'), 
                $http_code, 
                $response
            );
            $this->logger->error($error_message);
            throw new Exception($error_message);
        }
        
        // Check for SOAP fault in the response
        if (strpos($response, '<faultcode>') !== false) {
            $fault_code = $this->extract_value_from_xml($response, 'faultcode');
            $fault_string = $this->extract_value_from_xml($response, 'faultstring');
            
            $error_message = sprintf(
                __('SOAP Fault: %1$s (%2$s)', 'edm-efatura-premium'),
                $fault_string,
                $fault_code
            );
            
            $this->logger->error($error_message);
            throw new Exception($error_message);
        }
        
        return $response;
    }
    
    /**
     * Extract value from XML
     *
     * @param string $xml XML content
     * @param string $tag Tag name
     * @return string|null Tag value or null
     */
    private function extract_value_from_xml($xml, $tag) {
        // Escape any regex-special chars in $tag
        $safe_tag = preg_quote($tag, '/');
        
        $pattern = '/<' 
                 . $safe_tag 
                 . '[^>]*>' 
                 . '(.*?)'
                 . '<\/' 
                 . $safe_tag 
                 . '>/s';
        
        if (preg_match($pattern, $xml, $matches)) {
            return trim($matches[1]);
        }
        
        return null;
    }
    
    /**
     * Extract attribute from XML
     *
     * @param string $xml XML content
     * @param string $element Element name
     * @param string $attribute Attribute name
     * @return string|null Attribute value or null
     */
    private function extract_attribute_from_xml($xml, $element, $attribute) {
        // Escape element & attribute names for regex
        $safe_element = preg_quote($element, '/');
        $safe_attribute = preg_quote($attribute, '/');
        
        // Build the pattern
        $pattern = '/'
                 . '<' . $safe_element
                 . '[^>]*\s+' . $safe_attribute . '=['
                 . '["\']' . '([^"\']*)' . '["\']'
                 . '[^>]*'
                 . '>'
                 . '/is';
        
        if (preg_match($pattern, $xml, $matches)) {
            return $matches[1];
        }
        
        return null;
    }
    
    /**
     * Parse invoices from XML response
     *
     * @param string $xml XML content
     * @return array Invoice list
     */
    private function parse_invoices_from_xml($xml) {
        $invoices = array();
        
        // For each invoice
        $pattern = '/<INVOICE[^>]*>(.*?)<\/INVOICE>/s';
        if (preg_match_all($pattern, $xml, $invoice_matches)) {
            foreach ($invoice_matches[0] as $invoice_xml) {
                $invoice = array(
                    'id' => $this->extract_value_from_xml($invoice_xml, 'ID'),
                    'uuid' => $this->extract_attribute_from_xml($invoice_xml, 'INVOICE', 'UUID'),
                    'issue_date' => $this->extract_value_from_xml($invoice_xml, 'ISSUE_DATE'),
                    'sender' => $this->extract_value_from_xml($invoice_xml, 'SENDER'),
                    'receiver' => $this->extract_value_from_xml($invoice_xml, 'RECEIVER'),
                    'supplier' => $this->extract_value_from_xml($invoice_xml, 'SUPPLIER'),
                    'customer' => $this->extract_value_from_xml($invoice_xml, 'CUSTOMER'),
                    'amount' => $this->extract_value_from_xml($invoice_xml, 'PAYABLE_AMOUNT'),
                    'currency' => $this->extract_attribute_from_xml($invoice_xml, 'PAYABLE_AMOUNT', 'currencyID'),
                    'status' => $this->extract_value_from_xml($invoice_xml, 'STATUS'),
                    'status_description' => $this->extract_value_from_xml($invoice_xml, 'STATUS_DESCRIPTION'),
                    'creation_date' => $this->extract_value_from_xml($invoice_xml, 'CDATE')
                );
                
                // Add content if present
                $content = $this->extract_value_from_xml($invoice_xml, 'CONTENT');
                if ($content) {
                    $invoice['content'] = $content;
                }
                
                $invoices[] = $invoice;
            }
        }
        
        return $invoices;
    }
    
    
    
    /**
     * Extract sender VKN from invoice XML
     *
     * @param string $xml XML content
     * @return string Sender VKN
     */
    private function extract_sender_vkn($xml) {
        // Try simple match first
        $vkn = $this->extract_value_from_xml($xml, 'vkn');
        if ($vkn) {
            return $vkn;
        }
        
        // Search in UBL format
        $pattern = '/<cac:PartyIdentification>.*?<cbc:ID[^>]*>(.*?)<\/cbc:ID>/s';
        if (preg_match($pattern, $xml, $matches)) {
            return $matches[1];
        }
        
        // Default value
        return '';
    }
    
    /**
     * Extract sender alias from invoice XML
     *
     * @param string $xml XML content
     * @return string Sender alias
     */
    private function extract_sender_alias($xml) {
		// If EDM email is set, create alias from that email
		if (!empty($this->edm_email)) {
			$email_parts = explode('@', $this->edm_email);
			if (count($email_parts) === 2) {
				$username = $email_parts[0];
				$domain = $email_parts[1];
				return 'urn:mail:' . $username . '@' . $domain;
			}
		}
		
		// Search for alias in XML
		$alias = $this->extract_value_from_xml($xml, 'alias');
		if ($alias) {
			return $alias;
		}
		
		// Search in UBL format - email address
		$pattern = '/<cbc:ElectronicMail>(.*?)<\/cbc:ElectronicMail>/s';
		if (preg_match($pattern, $xml, $matches)) {
			$email = $matches[1];
			$email_parts = explode('@', $email);
			if (count($email_parts) === 2) {
				$username = $email_parts[0];
				$domain = $email_parts[1];
				return 'urn:mail:' . $username . '@' . $domain;
			}
		}
		
		// Default value - use settings if available
		$settings = get_option('edm_efatura_premium_settings', array());
		if (!empty($settings['edm_email'])) {
			$email_parts = explode('@', $settings['edm_email']);
			if (count($email_parts) === 2) {
				$username = $email_parts[0];
				$domain = $email_parts[1];
				return 'urn:mail:' . $username . '@' . $domain;
			}
		}
		
		return 'urn:mail:defaultpk@edmbilisim.com';
	}
    
    /**
     * Generate UUID
     *
     * @return string UUID
     */
    private function generate_uuid() {
        if (function_exists('wp_generate_uuid4')) {
            return wp_generate_uuid4();
        }
        
        // Alternative UUID generation (PHP 5.3+)
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }
    
    /**
     * Debug log
     *
     * @param string $message Log message
     */
    private function log_debug($message) {
        if ($this->debug) {
            $this->logger->debug($message);
        }
    }
    
    /**
     * Check counter (kontör sorgulama)
     * Based on EDM official documentation format
     * 
     * @return array Response with counter information
     */
    public function check_counter() {
        try {
            // Ensure we have a session
            $session_id = $this->ensure_session();
            if (!$session_id) {
                return array(
                    'success' => false,
                    'message' => __('Login failed for counter check.', 'edm-efatura-premium')
                );
            }
            
            // Current date and time
            $current_datetime = date('c'); // ISO 8601 format
            
            // Build XML request exactly as shown in EDM documentation
            $xml = '<?xml version="1.0" encoding="UTF-8"?>
            <s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
                <s:Header>
                    <ActivityId CorrelationId="' . $this->generate_uuid() . '" xmlns="http://schemas.microsoft.com/2004/09/ServiceModel/Diagnostics">00000000-0000-0000-0000-000000000000</ActivityId>
                </s:Header>
                <s:Body xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                    <CheckCounterRequest xmlns="http://tempuri.org/">
                        <REQUEST_HEADER xmlns="">
                            <SESSION_ID>' . $session_id . '</SESSION_ID>
                            <ACTION_DATE>' . $current_datetime . '</ACTION_DATE>
                            <REASON>E-fatura/E-Arşiv kontör sorgulama</REASON>
                            <APPLICATION_NAME>' . EDM_EFATURA_PREMIUM_APPLICATION_NAME . '</APPLICATION_NAME>
                            <HOSTNAME>' . (isset($_SERVER['SERVER_NAME']) ? $_SERVER['SERVER_NAME'] : 'localhost') . '</HOSTNAME>
                            <CHANNEL_NAME>WEB</CHANNEL_NAME>
                            <COMPRESSED>N</COMPRESSED>
                        </REQUEST_HEADER>
                    </CheckCounterRequest>
                </s:Body>
            </s:Envelope>';
            
            $this->log_debug('CheckCounter Request: ' . $xml);
            
            // Send SOAP request with proper headers (same pattern as Login/Logout)
            $headers = array(
                'Content-Type: text/xml; charset=utf-8',
                'SOAPAction: "CheckCounterRequest"',
                'Content-Length: ' . strlen($xml)
            );
            
            $response = $this->send_soap_request($xml, $headers);
            
            $this->log_debug('CheckCounter Response: ' . $response);
            
            if ($response === false) {
                return array(
                    'success' => false,
                    'message' => __('Counter check request failed.', 'edm-efatura-premium'),
                    'debug' => 'SOAP request returned false'
                );
            }
            
            // Parse XML response
            $xml_response = simplexml_load_string($response);
            if ($xml_response === false) {
                return array(
                    'success' => false,
                    'message' => __('Invalid XML response for counter check.', 'edm-efatura-premium'),
                    'debug' => 'XML parse failed. Response: ' . substr($response, 0, 200)
                );
            }
            
            // Register namespaces as per EDM documentation
            $xml_response->registerXPathNamespace('s', 'http://schemas.xmlsoap.org/soap/envelope/');
            $xml_response->registerXPathNamespace('temp', 'http://tempuri.org/');
            
            // Check for SOAP fault
            $fault = $xml_response->xpath('//s:Fault');
            if (!empty($fault)) {
                $fault_string = isset($fault[0]->faultstring) ? (string) $fault[0]->faultstring : 'Unknown SOAP fault';
                return array(
                    'success' => false,
                    'message' => __('Counter check failed: ', 'edm-efatura-premium') . $fault_string,
                    'debug' => 'SOAP Fault detected'
                );
            }
            
            // Try to get counter value using different xpath patterns
            $counter_nodes = array();
            
            // Pattern 1: With tempuri namespace
            $counter_nodes = $xml_response->xpath('//temp:CheckCounterResponse/COUNTER_LEFT');
            if (empty($counter_nodes)) {
                // Pattern 2: Without namespace prefix
                $counter_nodes = $xml_response->xpath('//CheckCounterResponse/COUNTER_LEFT');
            }
            if (empty($counter_nodes)) {
                // Pattern 3: Direct COUNTER_LEFT search
                $counter_nodes = $xml_response->xpath('//COUNTER_LEFT');
            }
            
            if (!empty($counter_nodes)) {
                $counter_left = (int) $counter_nodes[0];
                return array(
                    'success' => true,
                    'counter_left' => $counter_left,
                    'message' => __('Counter check successful.', 'edm-efatura-premium')
                );
            }
            
            // If no counter found, log the full response for debugging
            $this->logger->error('Counter not found in response. Full response: ' . $response);
            
            return array(
                'success' => false,
                'message' => __('Counter value not found in response.', 'edm-efatura-premium'),
                'debug' => 'Response: ' . substr($response, 0, 500) . '...'
            );
            
        } catch (Exception $e) {
            $this->logger->error('Counter check error: ' . $e->getMessage());
            
            // Create user-friendly error message
            $user_message = __('API bağlantı hatası. Lütfen kullanıcı adı ve şifrenizi kontrol edin.', 'edm-efatura-premium');
            
            // Check for specific error patterns
            $error_msg = $e->getMessage();
            if (strpos($error_msg, 'Kullanıcı Bulunamadı') !== false || strpos($error_msg, 'Kullanıcı veya Şifre hatalı') !== false) {
                $user_message = __('Kullanıcı adı veya şifre hatalı.', 'edm-efatura-premium');
            } elseif (strpos($error_msg, 'HTTP Hatası') !== false) {
                $user_message = __('EDM sunucusu ile bağlantı kurulamadı.', 'edm-efatura-premium');
            } elseif (strpos($error_msg, 'timeout') !== false) {
                $user_message = __('Bağlantı zaman aşımına uğradı.', 'edm-efatura-premium');
            }
            
            return array(
                'success' => false,
                'message' => $user_message,
                'debug' => 'Exception: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine()
            );
        }
    }
    
    /**
     * Destructor - ensure active sessions are closed
     */
    public function __destruct() {
        // Automatically logout if session exists
        $this->logout_if_active();
        
        // Close persistent CURL handle
        if (self::$curl_handle !== null) {
            curl_close(self::$curl_handle);
            self::$curl_handle = null;
        }
    }
}