<?php
/**
 * EDM E-Fatura Premium Cache Management Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling cache operations
 */
class EDM_Efatura_Premium_Cache {
    
    /**
     * Cache group prefix
     */
    const CACHE_GROUP = 'edm_efatura';
    
    /**
     * Default cache expiration (1 hour)
     */
    const DEFAULT_EXPIRATION = 3600;
    
    /**
     * Cache statistics
     */
    private static $cache_stats = array(
        'hits' => 0,
        'misses' => 0,
        'sets' => 0,
        'deletes' => 0
    );
    
    /**
     * Initialize cache system
     */
    public static function init() {
        // Register cleanup hooks
        add_action('edm_efatura_cleanup_cache', array(__CLASS__, 'cleanup_expired_cache'));
        
        // Schedule cache cleanup if not already scheduled
        if (!wp_next_scheduled('edm_efatura_cleanup_cache')) {
            wp_schedule_event(time(), 'hourly', 'edm_efatura_cleanup_cache');
        }
        
        // Register cache groups with WordPress
        wp_cache_add_global_groups(array(
            self::CACHE_GROUP . '_api',
            self::CACHE_GROUP . '_db',
            self::CACHE_GROUP . '_settings',
            self::CACHE_GROUP . '_invoices'
        ));
    }
    
    /**
     * Get cached data
     *
     * @param string $key Cache key
     * @param string $group Cache group
     * @param mixed $default Default value if cache miss
     * @return mixed Cached data or default value
     */
    public static function get($key, $group = 'general', $default = false) {
        $cache_key = self::build_key($key);
        $cache_group = self::CACHE_GROUP . '_' . $group;
        
        $cached_data = wp_cache_get($cache_key, $cache_group);
        
        if ($cached_data !== false) {
            self::$cache_stats['hits']++;
            
            // Check if data has expiration and is expired
            if (isset($cached_data['expires']) && $cached_data['expires'] < time()) {
                wp_cache_delete($cache_key, $cache_group);
                self::$cache_stats['misses']++;
                return $default;
            }
            
            return isset($cached_data['data']) ? $cached_data['data'] : $cached_data;
        }
        
        self::$cache_stats['misses']++;
        return $default;
    }
    
    /**
     * Set cache data
     *
     * @param string $key Cache key
     * @param mixed $data Data to cache
     * @param string $group Cache group
     * @param int $expiration Cache expiration in seconds
     * @return bool Success status
     */
    public static function set($key, $data, $group = 'general', $expiration = self::DEFAULT_EXPIRATION) {
        $cache_key = self::build_key($key);
        $cache_group = self::CACHE_GROUP . '_' . $group;
        
        // Prepare cache data with expiration
        $cache_data = array(
            'data' => $data,
            'expires' => time() + $expiration,
            'created' => time()
        );
        
        $result = wp_cache_set($cache_key, $cache_data, $cache_group, $expiration);
        
        if ($result) {
            self::$cache_stats['sets']++;
        }
        
        return $result;
    }
    
    /**
     * Delete cached data
     *
     * @param string $key Cache key
     * @param string $group Cache group
     * @return bool Success status
     */
    public static function delete($key, $group = 'general') {
        $cache_key = self::build_key($key);
        $cache_group = self::CACHE_GROUP . '_' . $group;
        
        $result = wp_cache_delete($cache_key, $cache_group);
        
        if ($result) {
            self::$cache_stats['deletes']++;
        }
        
        return $result;
    }
    
    /**
     * Get or set cached data (lazy loading pattern)
     *
     * @param string $key Cache key
     * @param callable $callback Callback to generate data if not cached
     * @param string $group Cache group
     * @param int $expiration Cache expiration in seconds
     * @return mixed Cached or generated data
     */
    public static function remember($key, $callback, $group = 'general', $expiration = self::DEFAULT_EXPIRATION) {
        $cached_data = self::get($key, $group, null);
        
        if ($cached_data !== null) {
            return $cached_data;
        }
        
        // Generate data using callback
        $data = call_user_func($callback);
        
        // Cache the generated data
        self::set($key, $data, $group, $expiration);
        
        return $data;
    }
    
    /**
     * Cache API responses
     *
     * @param string $method API method
     * @param array $params API parameters
     * @param mixed $response API response
     * @param int $expiration Cache expiration
     */
    public static function cache_api_response($method, $params, $response, $expiration = 900) {
        $cache_key = 'api_' . $method . '_' . md5(serialize($params));
        self::set($cache_key, $response, 'api', $expiration);
    }
    
    /**
     * Get cached API response
     *
     * @param string $method API method
     * @param array $params API parameters
     * @return mixed Cached response or false
     */
    public static function get_cached_api_response($method, $params) {
        $cache_key = 'api_' . $method . '_' . md5(serialize($params));
        return self::get($cache_key, 'api', false);
    }
    
    /**
     * Cache database query results
     *
     * @param string $sql SQL query
     * @param array $params Query parameters
     * @param mixed $results Query results
     * @param int $expiration Cache expiration
     */
    public static function cache_db_query($sql, $params, $results, $expiration = 1800) {
        $cache_key = 'db_' . md5($sql . serialize($params));
        self::set($cache_key, $results, 'db', $expiration);
    }
    
    /**
     * Get cached database query results
     *
     * @param string $sql SQL query
     * @param array $params Query parameters
     * @return mixed Cached results or false
     */
    public static function get_cached_db_query($sql, $params) {
        $cache_key = 'db_' . md5($sql . serialize($params));
        return self::get($cache_key, 'db', false);
    }
    
    /**
     * Cache invoice data
     *
     * @param string $uuid Invoice UUID
     * @param array $invoice_data Invoice data
     * @param int $expiration Cache expiration
     */
    public static function cache_invoice($uuid, $invoice_data, $expiration = 1800) {
        self::set('invoice_' . $uuid, $invoice_data, 'invoices', $expiration);
    }
    
    /**
     * Get cached invoice data
     *
     * @param string $uuid Invoice UUID
     * @return array|false Cached invoice data or false
     */
    public static function get_cached_invoice($uuid) {
        return self::get('invoice_' . $uuid, 'invoices', false);
    }
    
    /**
     * Invalidate invoice cache
     *
     * @param string $uuid Invoice UUID
     */
    public static function invalidate_invoice($uuid) {
        self::delete('invoice_' . $uuid, 'invoices');
        
        // Also clear related caches
        self::flush_group('invoices');
    }
    
    /**
     * Cache settings
     *
     * @param array $settings Settings data
     * @param int $expiration Cache expiration
     */
    public static function cache_settings($settings, $expiration = 3600) {
        self::set('plugin_settings', $settings, 'settings', $expiration);
    }
    
    /**
     * Get cached settings
     *
     * @return array|false Cached settings or false
     */
    public static function get_cached_settings() {
        return self::get('plugin_settings', 'settings', false);
    }
    
    /**
     * Flush cache group
     *
     * @param string $group Cache group to flush
     * @return bool Success status
     */
    public static function flush_group($group) {
        $cache_group = self::CACHE_GROUP . '_' . $group;
        
        // WordPress doesn't have a native group flush, so we'll use a workaround
        // by incrementing a group version number
        $version_key = 'cache_version_' . $group;
        $current_version = wp_cache_get($version_key, self::CACHE_GROUP);
        
        if ($current_version === false) {
            $current_version = 1;
        } else {
            $current_version++;
        }
        
        return wp_cache_set($version_key, $current_version, self::CACHE_GROUP, 0);
    }
    
    /**
     * Flush all EDM caches
     *
     * @return bool Success status
     */
    public static function flush_all() {
        $groups = array('general', 'api', 'db', 'settings', 'invoices');
        $success = true;
        
        foreach ($groups as $group) {
            if (!self::flush_group($group)) {
                $success = false;
            }
        }
        
        return $success;
    }
    
    /**
     * Build cache key with version support
     *
     * @param string $key Original key
     * @param string $group Cache group
     * @return string Versioned cache key
     */
    private static function build_key($key, $group = 'general') {
        $version_key = 'cache_version_' . $group;
        $version = wp_cache_get($version_key, self::CACHE_GROUP);
        
        if ($version === false) {
            $version = 1;
            wp_cache_set($version_key, $version, self::CACHE_GROUP, 0);
        }
        
        return $key . '_v' . $version;
    }
    
    /**
     * Cleanup expired cache entries
     */
    public static function cleanup_expired_cache() {
        // This is handled by WordPress cache backend
        // But we can cleanup our custom transients
        global $wpdb;
        
        $expired_transients = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT option_name FROM {$wpdb->options} 
                 WHERE option_name LIKE %s 
                 AND option_name LIKE %s 
                 AND option_value < %d",
                '_transient_timeout_edm_%',
                '%timeout%',
                time()
            )
        );
        
        foreach ($expired_transients as $transient) {
            $key = str_replace('_transient_timeout_', '', $transient->option_name);
            delete_transient($key);
        }
    }
    
    /**
     * Get cache statistics
     *
     * @return array Cache statistics
     */
    public static function get_stats() {
        $hit_rate = 0;
        $total_requests = self::$cache_stats['hits'] + self::$cache_stats['misses'];
        
        if ($total_requests > 0) {
            $hit_rate = (self::$cache_stats['hits'] / $total_requests) * 100;
        }
        
        return array_merge(self::$cache_stats, array(
            'hit_rate' => round($hit_rate, 2),
            'total_requests' => $total_requests
        ));
    }
    
    /**
     * Get cache size estimation
     *
     * @return array Cache size information
     */
    public static function get_cache_size() {
        global $wpdb;
        
        // Count EDM related transients
        $transient_count = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->options} 
                 WHERE option_name LIKE %s",
                '_transient_edm_%'
            )
        );
        
        // Estimate memory usage (rough calculation)
        $estimated_size = $transient_count * 1024; // 1KB average per cache entry
        
        return array(
            'entries' => (int) $transient_count,
            'estimated_size_bytes' => $estimated_size,
            'estimated_size_human' => size_format($estimated_size)
        );
    }
    
    /**
     * Check if cache is available and working
     *
     * @return bool Cache availability status
     */
    public static function is_cache_available() {
        $test_key = 'cache_test_' . time();
        $test_value = 'test_value_' . rand(1000, 9999);
        
        // Try to set and get a test value
        wp_cache_set($test_key, $test_value, self::CACHE_GROUP, 60);
        $retrieved_value = wp_cache_get($test_key, self::CACHE_GROUP);
        
        // Cleanup test value
        wp_cache_delete($test_key, self::CACHE_GROUP);
        
        return $retrieved_value === $test_value;
    }
    
    /**
     * Warm up cache with frequently accessed data
     */
    public static function warm_up_cache() {
        try {
            // Warm up settings cache
            $settings = get_option('edm_efatura_premium_settings', array());
            if (!empty($settings)) {
                self::cache_settings($settings);
            }
            
            // Warm up recent invoices cache
            global $wpdb;
            $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
            
            $recent_invoices = $wpdb->get_results(
                "SELECT * FROM {$table_name} 
                 ORDER BY created_at DESC 
                 LIMIT 10",
                ARRAY_A
            );
            
            foreach ($recent_invoices as $invoice) {
                self::cache_invoice($invoice['uuid'], $invoice, 1800);
            }
            
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
}