<?php
/**
 * EDM E-Fatura Premium Cron Checker
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for checking scheduled invoices and triggering them
 * Provides a reliable mechanism to ensure scheduled invoices are processed
 */
class EDM_Efatura_Premium_Cron_Checker {
    /**
     * Logger instance
     *
     * @var EDM_Efatura_Premium_Logger
     */
    private $logger;
    
    /**
     * Static flag to prevent multiple checks in the same request
     */
    private static $check_performed = false;
    
    /**
     * Constructor - Register hooks
     */
    public function __construct() {
        $this->logger = new EDM_Efatura_Premium_Logger();
        
        // Register one-time manual check on init with a very late priority
        add_action('init', array($this, 'manual_check_scheduled_invoices'), 999);
        
        // Add filter for cron schedules
        add_filter('cron_schedules', array($this, 'add_cron_interval'));
        
        // Register WP-Cron hook
        add_action('edm_check_scheduled_invoices', array($this, 'check_scheduled_invoices'));
        
        // Make sure cron is scheduled
        add_action('init', array($this, 'ensure_cron_scheduled'), 5);
    }
    
    /**
     * Ensure cron event is scheduled
     */
    public function ensure_cron_scheduled() {
        if (!wp_next_scheduled('edm_check_scheduled_invoices')) {
            wp_schedule_event(time(), 'five_minutes', 'edm_check_scheduled_invoices');
            
        }
    }
    
    /**
     * Add custom 5-minute schedule interval
     * 
     * @param array $schedules Existing schedules
     * @return array Modified schedules
     */
    public function add_cron_interval($schedules) {
        $schedules['five_minutes'] = array(
            'interval' => 300,
            'display'  => __('Every 5 Minutes', 'edm-efatura-premium'),
        );
        
        return $schedules;
    }
    
    /**
     * Manual check for scheduled invoices
     * This acts as a backup for WP-Cron
     */
    public function manual_check_scheduled_invoices() {
    // Only do this once per request
    if (self::$check_performed) {
        return;
    }
    
    self::$check_performed = true;
    
    // Don't run on admin-ajax requests
    if (defined('DOING_AJAX') && DOING_AJAX) {
        return;
    }
    
    // Don't run on REST API requests
    if (defined('REST_REQUEST') && REST_REQUEST) {
        return;
    }
    
    // GMT+3 için zamanı ayarla
    $current_time = time() + (3 * 3600); // 3 saat ekliyoruz
    
    // Check if it's time to run (last run was more than 4 minutes ago)
    $last_run = get_option('edm_last_manual_check', 0);
    
    if (($current_time - $last_run) < 240) { // 4 minutes
        return;
    }
    
    // Update the last run time (GMT+3 olarak)
    update_option('edm_last_manual_check', $current_time);
    
    
    // Check scheduled invoices
    $this->check_scheduled_invoices();
}
    
    /**
 * Check for scheduled invoices and process them
 */
public function check_scheduled_invoices() {
    
    // Get all scheduled invoices
    $scheduled_invoices = get_option('edm_scheduled_invoices', array());
    
    if (empty($scheduled_invoices)) {
        return;
    }
    
    // GMT+3 için zamanı ayarla
    $current_time = time() + (3 * 3600); // 3 saat ekliyoruz
    
    $this->logger->info('Checking scheduled invoices with GMT+3 timezone', array(
        'count' => count($scheduled_invoices),
        'current_time_gmt3' => date('Y-m-d H:i:s', $current_time)
    ));
    
    $updated = false;
    
    foreach ($scheduled_invoices as $order_id => $schedule) {
        // Order varlığını kontrol et
        $order = wc_get_order($order_id);
        if (!$order) {
            $this->logger->error('Order not found, removing from schedule', array('order_id' => $order_id));
            unset($scheduled_invoices[$order_id]);
            $updated = true;
            continue;
        }

        // GMT+3 için zamanlanmış tarihi logla
        $this->logger->info('Checking scheduled invoice (GMT+3)', array(
            'order_id' => $order_id,
            'scheduled_time' => date('Y-m-d H:i:s', $schedule['timestamp']),
            'current_time_gmt3' => date('Y-m-d H:i:s', $current_time),
            'time_remaining_seconds' => $schedule['timestamp'] - $current_time
        ));

        // ZAMANI GELMEDİYSE ATLAT!
        if ($current_time < $schedule['timestamp']) {
            $this->logger->info('Invoice not yet due, skipping', array(
                'order_id' => $order_id,
                'time_remaining_seconds' => $schedule['timestamp'] - $current_time
            ));
            continue;
        }

        // Zamanı geldi, işle
        $this->logger->info('Processing scheduled invoice (GMT+3)', array(
            'order_id' => $order_id,
            'scheduled_time' => date('Y-m-d H:i:s', $schedule['timestamp']),
            'current_time_gmt3' => date('Y-m-d H:i:s', $current_time)
        ));

        // Doğrudan process_direct_invoice_generation() metodunu çağır (asıl cronjob fonksiyonu)
        try {

            // Global ana plugin instance'ını kullan
            global $edm_efatura_premium;

            if ($edm_efatura_premium && method_exists($edm_efatura_premium, 'process_direct_invoice_generation')) {
                // Schedule key'i al
                $schedule_key = isset($schedule['key']) ? $schedule['key'] : 'cronjob_' . time();

                // Ana plugin'deki direct generation metodunu çağır
                $result = $edm_efatura_premium->process_direct_invoice_generation($order_id, $schedule_key);

                $this->logger->info('Invoice process result (direct generation)', array(
                    'order_id' => $order_id,
                    'result' => $result ? 'Success' : 'Failed',
                    'schedule_key' => $schedule_key
                ));
            } else {
                $this->logger->error('EDM Premium process_direct_invoice_generation method not available', array('order_id' => $order_id));
                $result = false;
            }
        } catch (Exception $e) {
            $this->logger->error('Exception during invoice processing (GMT+3)', array(
                'order_id' => $order_id,
                'error' => $e->getMessage()
            ));
        }

        // İşlendiyse zamanlanmış listeden kaldır
        unset($scheduled_invoices[$order_id]);
        $updated = true;
    }
    
    // Listeyi güncelle
    if ($updated) {
        update_option('edm_scheduled_invoices', $scheduled_invoices);
        $this->logger->info('Updated scheduled invoices list (GMT+3)', array(
            'remaining' => count($scheduled_invoices)
        ));
    }
}
    
    
}

// Initialize the cron checker
global $edm_cron_checker_instance;
if (!isset($edm_cron_checker_instance)) {
    $edm_cron_checker_instance = new EDM_Efatura_Premium_Cron_Checker();
}