<?php
/**
 * EDM E-Fatura Premium Database Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling database operations
 */
class EDM_Efatura_Premium_DB {
    /**
     * Check and create database tables
     */
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Invoices table
        $invoices_table = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
        $invoice_items_table = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICE_ITEMS;
        $log_table = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_LOG;
        
        // SQL for invoices table
        $invoices_sql = "CREATE TABLE $invoices_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            invoice_id varchar(50) NOT NULL,
            uuid varchar(255) NOT NULL,
            order_id bigint(20) DEFAULT NULL,
            type varchar(20) NOT NULL,
            direction varchar(20) NOT NULL,
            status varchar(50) NOT NULL,
            customer_vkn varchar(20) NOT NULL,
            customer_name varchar(255) NOT NULL,
            amount decimal(15,2) NOT NULL,
            currency varchar(5) NOT NULL,
            xml_content longtext DEFAULT NULL,
            pdf_path varchar(255) DEFAULT NULL,
            created_at datetime NOT NULL,
            updated_at datetime NOT NULL,
            extra_data longtext DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY invoice_id (invoice_id),
            KEY uuid (uuid),
            KEY order_id (order_id),
            KEY status (status),
            KEY customer_vkn (customer_vkn)
        ) $charset_collate;";
        
        // SQL for invoice items table
        $invoice_items_sql = "CREATE TABLE $invoice_items_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            invoice_id bigint(20) NOT NULL,
            product_id bigint(20) DEFAULT NULL,
            name varchar(255) NOT NULL,
            quantity decimal(15,4) NOT NULL,
            unit_price decimal(15,2) NOT NULL,
            tax_rate decimal(5,2) NOT NULL,
            tax_amount decimal(15,2) NOT NULL,
            total_amount decimal(15,2) NOT NULL,
            PRIMARY KEY  (id),
            KEY invoice_id (invoice_id),
            KEY product_id (product_id)
        ) $charset_collate;";
        
        // SQL for log table
        $log_sql = "CREATE TABLE $log_table (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            level varchar(20) NOT NULL,
            message text NOT NULL,
            context longtext DEFAULT NULL,
            created_at datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY level (level),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        // Load dbDelta function
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        
        // Create tables
        dbDelta($invoices_sql);
        dbDelta($invoice_items_sql);
        dbDelta($log_sql);
        
        // Store database version
        update_option('edm_efatura_premium_db_version', EDM_EFATURA_PREMIUM_DB_VERSION);
    }
    
    /**
     * Save invoice to database
     *
     * @param array $invoice Invoice data
     * @return int|false The ID of the inserted invoice or false on error
     */
    public function save_invoice($invoice) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
		
		$now = current_time('mysql');
		
		// Ensure required fields are present
		if (empty($invoice['uuid']) || empty($invoice['status'])) {
			$logger = new EDM_Efatura_Premium_Logger();
			$logger->error('Missing required fields for invoice save', $invoice);
			return false;
		}
		
		$data = array(
			'invoice_id' => isset($invoice['id']) ? $invoice['id'] : '',
			'uuid' => $invoice['uuid'],
			'order_id' => isset($invoice['order_id']) ? $invoice['order_id'] : null,
			'type' => isset($invoice['type']) ? $invoice['type'] : 'EINVOICE',
			'direction' => isset($invoice['direction']) ? $invoice['direction'] : 'OUT',
			'status' => $invoice['status'],
			'customer_vkn' => isset($invoice['customer_vkn']) ? $invoice['customer_vkn'] : '',
			'customer_name' => isset($invoice['customer_name']) ? $invoice['customer_name'] : '',
			'amount' => isset($invoice['amount']) ? $invoice['amount'] : 0,
			'currency' => isset($invoice['currency']) ? $invoice['currency'] : 'TRY',
			'xml_content' => isset($invoice['content']) ? $invoice['content'] : null,
			'pdf_path' => isset($invoice['pdf_path']) ? $invoice['pdf_path'] : null,
			'created_at' => isset($invoice['created_at']) ? $invoice['created_at'] : $now,
			'updated_at' => $now,
			'extra_data' => isset($invoice['extra_data']) ? (is_array($invoice['extra_data']) ? wp_json_encode($invoice['extra_data']) : $invoice['extra_data']) : null
		);
		
		$format = array(
			'%s', // invoice_id
			'%s', // uuid
			'%d', // order_id
			'%s', // type
			'%s', // direction
			'%s', // status
			'%s', // customer_vkn
			'%s', // customer_name
			'%f', // amount
			'%s', // currency
			'%s', // xml_content
			'%s', // pdf_path
			'%s', // created_at
			'%s', // updated_at
			'%s'  // extra_data
		);
		
		// Insert or update
		$existing_id = $this->get_invoice_id_by_uuid($invoice['uuid']);
		
		if ($existing_id) {
			// Update existing invoice
			unset($data['created_at']); // Don't update creation date
			
			$result = $wpdb->update(
				$table_name,
				$data,
				array('id' => $existing_id),
				$format,
				array('%d')
			);
			
			if ($result === false) {
				$logger = new EDM_Efatura_Premium_Logger();
				$logger->error('Failed to update invoice', array(
					'invoice_uuid' => $invoice['uuid'],
					'error' => $wpdb->last_error
				));
				return false;
			}
			
			// Invalidate cache on update
			EDM_Efatura_Premium_Cache::invalidate_invoice($invoice['uuid']);
			
			return $existing_id;
		} else {
			// Insert new invoice
			$result = $wpdb->insert(
				$table_name,
				$data,
				$format
			);
			
			if ($result === false) {
				$logger = new EDM_Efatura_Premium_Logger();
				$logger->error('Failed to insert invoice', array(
					'invoice_uuid' => $invoice['uuid'],
					'error' => $wpdb->last_error
				));
				return false;
			}
			
			// Cache the new invoice
			EDM_Efatura_Premium_Cache::cache_invoice($invoice['uuid'], $data, 1800);
			
			return $wpdb->insert_id;
		}
	}
    /**
	 * Optimize log tablosunu
	 */
	public function optimize_log_table() {
		global $wpdb;
		
		$table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_LOG;
		
		// Tabloyu optimize et
		$wpdb->query("OPTIMIZE TABLE {$table_name}");
		
		// İki haftadan eski logları temizle
		$date = date('Y-m-d H:i:s', strtotime("-14 days"));
		$wpdb->query($wpdb->prepare("DELETE FROM {$table_name} WHERE created_at < %s", $date));
	}
    /**
     * Save invoice items
     *
     * @param int $invoice_id Invoice ID
     * @param array $items Invoice items
     * @return bool Success status
     */
    public function save_invoice_items($invoice_id, $items) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICE_ITEMS;
        
        // Delete existing items for this invoice
        $wpdb->delete(
            $table_name,
            array('invoice_id' => $invoice_id),
            array('%d')
        );
        
        // Insert new items
        foreach ($items as $item) {
            $wpdb->insert(
                $table_name,
                array(
                    'invoice_id' => $invoice_id,
                    'product_id' => isset($item['product_id']) ? $item['product_id'] : null,
                    'name' => $item['name'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'],
                    'tax_amount' => $item['tax_amount'],
                    'total_amount' => $item['total_amount']
                ),
                array(
                    '%d', // invoice_id
                    '%d', // product_id
                    '%s', // name
                    '%f', // quantity
                    '%f', // unit_price
                    '%f', // tax_rate
                    '%f', // tax_amount
                    '%f'  // total_amount
                )
            );
        }
        
        return true;
    }
    
    /**
     * Get invoice by UUID with caching
     *
     * @param string $uuid Invoice UUID
     * @return array|null Invoice data or null if not found
     */
    public function get_invoice_by_uuid($uuid) {
        // Try cache first
        $cached_invoice = EDM_Efatura_Premium_Cache::get_cached_invoice($uuid);
        if ($cached_invoice !== false) {
            return $cached_invoice;
        }
        
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
        
        $invoice = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM {$table_name} WHERE uuid = %s LIMIT 1",
                $uuid
            ),
            ARRAY_A
        );
        
        if (!$invoice) {
            return false;
        }
        
        // Unserialize extra_data if it's serialized
        if (!empty($invoice['extra_data']) && is_serialized($invoice['extra_data'])) {
            $invoice['extra_data'] = maybe_unserialize($invoice['extra_data']);
        }
        
        // Make sure invoice_id is set (it might be stored in id field)
        if (empty($invoice['invoice_id']) && !empty($invoice['id'])) {
            $invoice['invoice_id'] = $invoice['id'];
        }
        
        // Cache for 30 minutes
        EDM_Efatura_Premium_Cache::cache_invoice($uuid, $invoice, 1800);
        
        return $invoice;
    }
    
    /**
     * Get invoice ID by UUID
     *
     * @param string $uuid Invoice UUID
     * @return int|null Invoice ID or null if not found
     */
    public function get_invoice_id_by_uuid($uuid) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
        
        return $wpdb->get_var(
            $wpdb->prepare(
                "SELECT id FROM $table_name WHERE uuid = %s",
                $uuid
            )
        );
    }
    
    /**
     * Get invoice by order ID
     *
     * @param int $order_id WooCommerce order ID
     * @return array|null Invoice data or null if not found
     */
    public function get_invoice_by_order_id($order_id) {
    global $wpdb;
    
    $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
    
    $invoice = $wpdb->get_row(
        $wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE order_id = %d ORDER BY created_at DESC LIMIT 1",
            $order_id
        ),
        ARRAY_A
    );
    
    if (!$invoice) {
        return false;
    }
    
    // Unserialize extra_data if it's serialized
    if (!empty($invoice['extra_data']) && is_serialized($invoice['extra_data'])) {
        $invoice['extra_data'] = maybe_unserialize($invoice['extra_data']);
    }
    
    // Make sure invoice_id is set (it might be stored in id field)
    if (empty($invoice['invoice_id']) && !empty($invoice['id'])) {
        $invoice['invoice_id'] = $invoice['id'];
    }
    
    return $invoice;
}
    
    /**
     * Get invoice items
     *
     * @param int $invoice_id Invoice ID
     * @return array Invoice items
     */
    public function get_invoice_items($invoice_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICE_ITEMS;
        
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE invoice_id = %d",
                $invoice_id
            ),
            ARRAY_A
        );
    }
    
    /**
     * Get invoices with optimized cursor-based pagination
     *
     * @param int $page Page number
     * @param int $per_page Items per page
     * @param array $filters Filters (direction, status, etc.)
     * @return array Invoices and pagination info
     */
    public function get_invoices($page = 1, $per_page = 20, $filters = array()) {
        // Try to get from cache first
        $cache_key = 'invoices_' . md5(serialize(array($page, $per_page, $filters)));
        $cached_result = EDM_Efatura_Premium_Cache::get($cache_key, 'db');
        
        if ($cached_result !== false) {
            return $cached_result;
        }
        
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
        
        // Use cursor pagination for better performance
        if (isset($filters['cursor']) && $filters['cursor'] > 0) {
            $result = EDM_Efatura_Premium_Performance::cursor_paginate(
                $table_name,
                $this->build_filter_conditions($filters),
                $per_page,
                $filters['cursor']
            );
            
            // Cache result for 5 minutes
            EDM_Efatura_Premium_Cache::set($cache_key, $result, 'db', 300);
            
            return $result;
        }
        
        // Fallback to traditional pagination for first page
        $offset = ($page - 1) * $per_page;
        
        // Start building query
        $query = "SELECT * FROM {$table_name} WHERE 1=1";
        $count_query = "SELECT COUNT(*) FROM {$table_name} WHERE 1=1";
        $query_args = array();
    
    // Apply filters
    if (!empty($filters['direction'])) {
        if ($filters['direction'] === 'OUT-EINVOICE') {
            $query .= " AND direction = %s AND type = %s";
            $count_query .= " AND direction = %s AND type = %s";
            $query_args[] = 'OUT';
            $query_args[] = 'EINVOICE';
        } else if ($filters['direction'] === 'OUT-EARCHIVE') {
            $query .= " AND direction = %s AND type = %s";
            $count_query .= " AND direction = %s AND type = %s";
            $query_args[] = 'OUT';
            $query_args[] = 'EARCHIVE';
        } else {
            $query .= " AND direction = %s";
            $count_query .= " AND direction = %s";
            $query_args[] = $filters['direction'];
        }
    }
    
    if (!empty($filters['status'])) {
        $query .= " AND status = %s";
        $count_query .= " AND status = %s";
        $query_args[] = $filters['status'];
    }
    
    if (!empty($filters['customer_name'])) {
        $query .= " AND customer_name LIKE %s";
        $count_query .= " AND customer_name LIKE %s";
        $query_args[] = '%' . $wpdb->esc_like($filters['customer_name']) . '%';
    }
    
    if (!empty($filters['start_date']) && !empty($filters['end_date'])) {
        $query .= " AND created_at BETWEEN %s AND %s";
        $count_query .= " AND created_at BETWEEN %s AND %s";
        $query_args[] = $filters['start_date'] . ' 00:00:00';
        $query_args[] = $filters['end_date'] . ' 23:59:59';
    }
    
    // Complete count query with args
    $total = $wpdb->get_var($wpdb->prepare($count_query, $query_args));
    
    // Add order and limit to main query
    $query .= " ORDER BY created_at DESC LIMIT %d OFFSET %d";
    $query_args[] = $per_page;
    $query_args[] = $offset;
    
        // Get invoices
        $invoices = $wpdb->get_results($wpdb->prepare($query, $query_args), ARRAY_A);
        
        // Calculate total pages
        $total_pages = ceil($total / $per_page);
        
        $result = array(
            'invoices' => $invoices,
            'total' => $total,
            'pages' => $total_pages
        );
        
        // Cache result for 5 minutes
        EDM_Efatura_Premium_Cache::set($cache_key, $result, 'db', 300);
        
        return $result;
    }
    
    /**
     * Build filter conditions for database queries
     *
     * @param array $filters Filter array
     * @return array Conditions for cursor pagination
     */
    private function build_filter_conditions($filters) {
        $conditions = array();
        
        if (!empty($filters['direction'])) {
            if ($filters['direction'] === 'OUT-EINVOICE') {
                $conditions['direction'] = 'OUT';
                $conditions['type'] = 'EINVOICE';
            } else if ($filters['direction'] === 'OUT-EARCHIVE') {
                $conditions['direction'] = 'OUT';
                $conditions['type'] = 'EARCHIVE';
            } else {
                $conditions['direction'] = $filters['direction'];
            }
        }
        
        if (!empty($filters['status'])) {
            $conditions['status'] = $filters['status'];
        }
        
        if (!empty($filters['customer_vkn'])) {
            $conditions['customer_vkn'] = $filters['customer_vkn'];
        }
        
        return $conditions;
    }
    /**
	 * Get total invoice count
	 *
	 * @param array $filters Filters (direction, status, etc.)
	 * @return int Total count
	 */
	public function get_invoice_count($filters = array()) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
		
		// Prepare query
		$sql = "SELECT COUNT(*) FROM {$table_name}";
		
		// Add filters
		$where = array();
		$where_values = array();
		
		if (!empty($filters['direction'])) {
			$where[] = 'direction = %s';
			$where_values[] = $filters['direction'];
		}
		
		if (!empty($filters['status'])) {
			$where[] = 'status = %s';
			$where_values[] = $filters['status'];
		}
		
		if (!empty($filters['customer_vkn'])) {
			$where[] = 'customer_vkn = %s';
			$where_values[] = $filters['customer_vkn'];
		}
		
		if (!empty($filters['start_date']) && !empty($filters['end_date'])) {
			$where[] = 'created_at BETWEEN %s AND %s';
			$where_values[] = $filters['start_date'] . ' 00:00:00';
			$where_values[] = $filters['end_date'] . ' 23:59:59';
		}
		
		// Add WHERE clause if filters exist
		if (!empty($where)) {
			$sql .= ' WHERE ' . implode(' AND ', $where);
			
			// Prepare SQL with values
			if (!empty($where_values)) {
				$sql = $wpdb->prepare($sql, $where_values);
			}
		}
		
		// Get result
		return (int) $wpdb->get_var($sql);
	}
    /**
     * Update invoice status
     *
     * @param string $uuid Invoice UUID
     * @param string $status New status
     * @param string $status_description Status description
     * @return bool Success status
     */
    public function update_invoice_status($uuid, $status, $status_description = '') {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_INVOICES;
        
        $result = $wpdb->update(
            $table_name,
            array(
                'status' => $status,
                'extra_data' => wp_json_encode(array(
                    'status_description' => $status_description,
                    'updated_at' => current_time('mysql')
                )),
                'updated_at' => current_time('mysql')
            ),
            array('uuid' => $uuid),
            array('%s', '%s', '%s'),
            array('%s')
        );
        
        return $result !== false;
    }
}