<?php
/**
 * EDM E-Fatura Premium - Merkezi İstisna Kodları Sınıfı
 * 
 * Bu sınıf tüm sistem genelinde kullanılan istisna kodlarını merkezi olarak yönetir.
 * Kod duplikasyonunu önler ve bakımı kolaylaştırır.
 * 
 * @package    EDM_EFatura_Premium
 * @subpackage EDM_EFatura_Premium/includes
 * @since      1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class EDM_EFatura_Premium_Exception_Codes {

    /**
     * Tam İstisna Kodları (Full Exemption Codes)
     * KDV Kanunu madde 11/1-a, 13, 14, 15, 19/2, 17/4 kapsamındaki istisna kodları
     * 
     * @since 1.0.0
     * @return array
     */
    public static function get_full_exemption_codes() {
        return array(
            '301' => __('11/1-a Mal İhracatı', 'edm-efatura-premium'),
            '302' => __('11/1-a Hizmet İhracatı', 'edm-efatura-premium'),
            '303' => __('11/1-a Roaming Hizmetleri', 'edm-efatura-premium'),
            '304' => __('13/a Deniz Hava ve Demiryolu Taşıma Araçlarının Teslimi İle İnşa, Tadil, Bakım ve Onarımları', 'edm-efatura-premium'),
            '305' => __('13/b Deniz ve Hava Taşıma Araçları İçin Liman Ve Hava Meydanlarında Yapılan Hizmetler', 'edm-efatura-premium'),
            '306' => __('13/c Petrol Aramaları ve Petrol Boru Hatlarının İnşa ve Modernizasyonuna İlişkin Yapılan Teslim ve Hizmetler', 'edm-efatura-premium'),
            '307' => __('13/c Maden Arama, Altın, Gümüş ve Platin Madenleri İçin İşletme, Zenginleştirme Ve Rafinaj Faaliyetlerine İlişkin Teslim Ve Hizmetler', 'edm-efatura-premium'),
            '308' => __('13/d Teşvikli Yatırım Mallarının Teslimi', 'edm-efatura-premium'),
            '309' => __('13/e Liman ve Hava Meydanı İnşası', 'edm-efatura-premium'),
            '310' => __('13/f Ulusal Güvenlik Amaçlı Teslimler', 'edm-efatura-premium'),
            '311' => __('14/1 Uluslararası Taşımacılık', 'edm-efatura-premium'),
            '312' => __('15/a Diplomatik Organ ve Misyonlar', 'edm-efatura-premium'),
            '313' => __('15/b Uluslararası Kuruluşlar', 'edm-efatura-premium'),
            '314' => __('19/2 Uluslararası Anlaşmalar', 'edm-efatura-premium'),
            '315' => __('14/3 İhraç Taşımacılarına Yapılan Motorin Teslimleri', 'edm-efatura-premium'),
            '316' => __('11/1-a Serbest Bölgelerdeki Hizmet İşlemleri', 'edm-efatura-premium'),
            '317' => __('17/4-s Engellilerin Eğitimleri, Meslekleri ve Günlük Yaşamlarına İlişkin Araç-Gereç ve Bilgisayar Programları', 'edm-efatura-premium'),
            '318' => __('Yap-İşlet-Devret Modeli Projeler', 'edm-efatura-premium'),
            '319' => __('13/g Başbakanlığa Yapılan Araç Teslimi', 'edm-efatura-premium'),
            '320' => __('ISMEP İstanbul Proje Koordinasyon Birimine Yapılan Teslimler', 'edm-efatura-premium'),
            '321' => __('Birleşmiş Milletler, NATO ve OECD\'ye Yapılan Teslimler', 'edm-efatura-premium'),
            '322' => __('11/1-a Yerleşik Olmayanlara Özel Teslim (Bavul Ticareti)', 'edm-efatura-premium'),
            '323' => __('13/ğ Ürün Senetlerinin İlk Teslimi', 'edm-efatura-premium'),
            '324' => __('13/h Türk Kızılayı Teslimleri ve Hizmetleri', 'edm-efatura-premium'),
            '325' => __('13/ı Yem Teslimleri', 'edm-efatura-premium'),
            '326' => __('13/ı Tescilli Gübre Teslimleri', 'edm-efatura-premium'),
            '327' => __('13/ı Gübre Üretimi İçin Hammadde', 'edm-efatura-premium'),
            '328' => __('13/i Konut veya İşyeri Teslimleri', 'edm-efatura-premium'),
            '329' => __('Rezerve Kod 329', 'edm-efatura-premium'), // Eksik kod eklendi
            '330' => __('13/j Organize Sanayi Bölgeleri İnşası', 'edm-efatura-premium'),
            '331' => __('13/m AR-GE, Yenilik ve Tasarım Makinaları', 'edm-efatura-premium'),
            '332' => __('İmalat Sanayii Makinaları', 'edm-efatura-premium'),
            '333' => __('13/k Kamu İdarelerine Bağışlar', 'edm-efatura-premium'),
            '334' => __('13/l Yabancılara Sağlık Hizmetleri', 'edm-efatura-premium'),
            '335' => __('13/n Basılı Kitap ve Süreli Yayınların Teslimleri', 'edm-efatura-premium'),
            '336' => __('UEFA Müsabakaları Teslimleri', 'edm-efatura-premium'),
            '337' => __('Türk Akımı Gaz Boru Hattı Hizmetleri', 'edm-efatura-premium'),
            '338' => __('İmalatçıların İhraç Malları', 'edm-efatura-premium'),
            '350' => __('Diğerleri', 'edm-efatura-premium'),
            '351' => __('KDV - Diğer İstisna Olmayan', 'edm-efatura-premium')
        );
    }

    /**
     * Kısmi İstisna Kodları (Partial Exemption Codes)
     * KDV Kanunu madde 17 kapsamındaki kısmi istisna kodları
     * 
     * @since 1.0.0
     * @return array
     */
    public static function get_partial_exemption_codes() {
        return array(
            '201' => __('17/1 Kültürel ve Eğitimsel Amaçlı İşlemler', 'edm-efatura-premium'),
            '202' => __('17/2-a Sağlık, Çevre ve Sosyal Yardım İşlemleri', 'edm-efatura-premium'),
            '203' => __('Rezerve Kod 203', 'edm-efatura-premium'), // Eksik kod eklendi
            '204' => __('17/2-c Yabancı Diplomatik Kurumlar ve Yardım Kuruluşları Bağışları', 'edm-efatura-premium'),
            '205' => __('17/2-d Taşınmaz Kültür Varlıkları ve Mimari Hizmetler', 'edm-efatura-premium'),
            '206' => __('17/2-e Meslek Örgütleri İşlemleri', 'edm-efatura-premium'),
            '207' => __('17/3 Askeri Fabrika, Tersane ve Atölye İşlemleri', 'edm-efatura-premium'),
            '208' => __('17/4-c Birleşme, Devir, Dönüşüm ve Bölünme İşlemleri', 'edm-efatura-premium'),
            '209' => __('17/4-e Banka ve Sigorta Muameleleri Vergisi Kapsamındaki İşlemler', 'edm-efatura-premium'),
            '210' => __('Rezerve Kod 210', 'edm-efatura-premium'), // Eksik kod eklendi
            '211' => __('17/4-h Tarımsal Sulama ve Köy İçme Suyu', 'edm-efatura-premium'),
            '212' => __('17/4-ı Serbest Bölgelerdeki Hizmetler', 'edm-efatura-premium'),
            '213' => __('17/4-j Petrol ve Gaz Boru Hattı Taşımacılığı', 'edm-efatura-premium'),
            '214' => __('17/4-k Organize Sanayi Bölgeleri ve Konut Kooperatifleri', 'edm-efatura-premium'),
            '215' => __('17/4-l Varlık Yönetim Şirketi İşlemleri', 'edm-efatura-premium'),
            '216' => __('17/4-m Tasarruf Mevduatı Sigorta Fonu İşlemleri', 'edm-efatura-premium'),
            '217' => __('17/4-n Basın-Medyaya Haber Hizmetleri', 'edm-efatura-premium'),
            '218' => __('17/4-o Gümrük Antrepoları ve Duty Free Mağazaları Hizmetleri', 'edm-efatura-premium'),
            '219' => __('17/4-p Hazine ve Arsa Ofisi İşlemleri', 'edm-efatura-premium'),
            '220' => __('17/4-r İki Yıl Elde Tutulan Gayrimenkul ve Hisse Satışı', 'edm-efatura-premium'),
            '221' => __('15 Kooperatiflere, Belediyelere Yapılan İnşaat Hizmetleri', 'edm-efatura-premium'),
            '222' => __('Rezerve Kod 222', 'edm-efatura-premium'), // Eksik kod eklendi
            '223' => __('20/1 Teknoloji Geliştirme Bölgeleri İşlemleri', 'edm-efatura-premium'),
            '224' => __('Rezerve Kod 224', 'edm-efatura-premium'), // Eksik kod eklendi
            '225' => __('23 Milli Eğitim Bakanlığına Bilgisayar Bağışları', 'edm-efatura-premium')
        );
    }

    /**
     * Tüm İstisna Kodları (Tam önce, sonra Kısmi)
     * 
     * @since 1.0.0
     * @return array
     */
    public static function get_all_exception_codes() {
        return array_merge(
            self::get_full_exemption_codes(),    // 301-351 (önce)
            self::get_partial_exemption_codes()  // 201-225 (sonra)
        );
    }

    /**
     * Form için formatlı istisna kodları (kod + açıklama)
     * 
     * @since 1.0.0
     * @return array
     */
    public static function get_formatted_exception_codes_for_forms() {
        // Direkt Türkçe açıklamalarla
        $turkish_descriptions = self::get_turkish_exception_descriptions();
        $formatted = array();
        
        foreach ($turkish_descriptions as $code => $description) {
            $formatted[$code] = $code . ' - ' . $description;
        }
        
        return $formatted;
    }

    /**
     * Türkçe istisna kodları açıklamaları (XML için)
     * 
     * @since 1.0.0
     * @return array
     */
    public static function get_turkish_exception_descriptions() {
        return array(
            // Tam İstisna Kodları
            '301' => '11/1-a Mal İhracatı',
            '302' => '11/1-a Hizmet İhracatı',
            '303' => '11/1-a Roaming Hizmetleri',
            '304' => '13/a Deniz Hava ve Demiryolu Taşıma Araçlarının Teslimi İle İnşa, Tadil, Bakım ve Onarımları',
            '305' => '13/b Deniz ve Hava Taşıma Araçları İçin Liman Ve Hava Meydanlarında Yapılan Hizmetler',
            '306' => '13/c Petrol Aramaları ve Petrol Boru Hatlarının İnşa ve Modernizasyonuna İlişkin Yapılan Teslim ve Hizmetler',
            '307' => '13/c Maden Arama, Altın, Gümüş ve Platin Madenleri İçin İşletme, Zenginleştirme Ve Rafinaj Faaliyetlerine İlişkin Teslim Ve Hizmetler',
            '308' => '13/d Teşvikli Yatırım Mallarının Teslimi',
            '309' => '13/e Liman ve Hava Meydanı İnşası',
            '310' => '13/f Ulusal Güvenlik Amaçlı Teslimler',
            '311' => '14/1 Uluslararası Taşımacılık',
            '312' => '15/a Diplomatik Organ ve Misyonlar',
            '313' => '15/b Uluslararası Kuruluşlar',
            '314' => '19/2 Uluslararası Anlaşmalar',
            '315' => '14/3 İhraç Taşımacılarına Yapılan Motorin Teslimleri',
            '316' => '11/1-a Serbest Bölgelerdeki Hizmet İşlemleri',
            '317' => '17/4-s Engellilerin Eğitimleri, Meslekleri ve Günlük Yaşamlarına İlişkin Araç-Gereç ve Bilgisayar Programları',
            '318' => 'Yap-İşlet-Devret Modeli Projeler',
            '319' => '13/g Başbakanlığa Yapılan Araç Teslimi',
            '320' => 'ISMEP İstanbul Proje Koordinasyon Birimine Yapılan Teslimler',
            '321' => 'Birleşmiş Milletler, NATO ve OECD\'ye Yapılan Teslimler',
            '322' => '11/1-a Yerleşik Olmayanlara Özel Teslim (Bavul Ticareti)',
            '323' => '13/ğ Ürün Senetlerinin İlk Teslimi',
            '324' => '13/h Türk Kızılayı Teslimleri ve Hizmetleri',
            '325' => '13/ı Yem Teslimleri',
            '326' => '13/ı Tescilli Gübre Teslimleri',
            '327' => '13/ı Gübre Üretimi İçin Hammadde',
            '328' => '13/i Konut veya İşyeri Teslimleri',
            '329' => 'Rezerve Kod 329',
            '330' => '13/j Organize Sanayi Bölgeleri İnşası',
            '331' => '13/m AR-GE, Yenilik ve Tasarım Makinaları',
            '332' => 'İmalat Sanayii Makinaları',
            '333' => '13/k Kamu İdarelerine Bağışlar',
            '334' => '13/l Yabancılara Sağlık Hizmetleri',
            '335' => '13/n Basılı Kitap ve Süreli Yayınların Teslimleri',
            '336' => 'UEFA Müsabakaları Teslimleri',
            '337' => 'Türk Akımı Gaz Boru Hattı Hizmetleri',
            '338' => 'İmalatçıların İhraç Malları',
            '350' => 'Diğerleri',
            '351' => 'KDV - Diğer İstisna Olmayan',

            // Kısmi İstisna Kodları
            '201' => '17/1 Kültürel ve Eğitimsel Amaçlı İşlemler',
            '202' => '17/2-a Sağlık, Çevre ve Sosyal Yardım İşlemleri',
            '203' => 'Rezerve Kod 203',
            '204' => '17/2-c Yabancı Diplomatik Kurumlar ve Yardım Kuruluşları Bağışları',
            '205' => '17/2-d Taşınmaz Kültür Varlıkları ve Mimari Hizmetler',
            '206' => '17/2-e Meslek Örgütleri İşlemleri',
            '207' => '17/3 Askeri Fabrika, Tersane ve Atölye İşlemleri',
            '208' => '17/4-c Birleşme, Devir, Dönüşüm ve Bölünme İşlemleri',
            '209' => '17/4-e Banka ve Sigorta Muameleleri Vergisi Kapsamındaki İşlemler',
            '210' => 'Rezerve Kod 210',
            '211' => '17/4-h Tarımsal Sulama ve Köy İçme Suyu',
            '212' => '17/4-ı Serbest Bölgelerdeki Hizmetler',
            '213' => '17/4-j Petrol ve Gaz Boru Hattı Taşımacılığı',
            '214' => '17/4-k Organize Sanayi Bölgeleri ve Konut Kooperatifleri',
            '215' => '17/4-l Varlık Yönetim Şirketi İşlemleri',
            '216' => '17/4-m Tasarruf Mevduatı Sigorta Fonu İşlemleri',
            '217' => '17/4-n Basın-Medyaya Haber Hizmetleri',
            '218' => '17/4-o Gümrük Antrepoları ve Duty Free Mağazaları Hizmetleri',
            '219' => '17/4-p Hazine ve Arsa Ofisi İşlemleri',
            '220' => '17/4-r İki Yıl Elde Tutulan Gayrimenkul ve Hisse Satışı',
            '221' => '15 Kooperatiflere, Belediyelere Yapılan İnşaat Hizmetleri',
            '222' => 'Rezerve Kod 222',
            '223' => '20/1 Teknoloji Geliştirme Bölgeleri İşlemleri',
            '224' => 'Rezerve Kod 224',
            '225' => '23 Milli Eğitim Bakanlığına Bilgisayar Bağışları'
        );
    }

    /**
     * Belirtilen kodun geçerli olup olmadığını kontrol eder
     * 
     * @since 1.0.0
     * @param string $code İstisna kodu
     * @return bool
     */
    public static function is_valid_exception_code($code) {
        $all_codes = self::get_all_exception_codes();
        return isset($all_codes[$code]);
    }

    /**
     * Belirtilen kodun açıklamasını döndürür
     * 
     * @since 1.0.0
     * @param string $code İstisna kodu
     * @param string $language Dil (en/tr)
     * @return string
     */
    public static function get_exception_description($code, $language = 'tr') {
        if ($language === 'tr') {
            $descriptions = self::get_turkish_exception_descriptions();
        } else {
            $descriptions = self::get_all_exception_codes();
        }
        
        return isset($descriptions[$code]) ? $descriptions[$code] : '';
    }

    /**
     * Varsayılan istisna kodunu döndürür (335 - Basılı Kitap)
     * 
     * @since 1.0.0
     * @return string
     */
    public static function get_default_exception_code() {
        return '335';
    }

    /**
     * Form seçenekleri için istisna kodlarını dropdown formatında döndürür
     * 
     * @since 1.0.0
     * @return array
     */
    public static function get_exception_codes_for_dropdown() {
        $codes = self::get_formatted_exception_codes_for_forms();
        $dropdown = array('' => __('Select Exception Code', 'edm-efatura-premium'));
        
        return array_merge($dropdown, $codes);
    }
}