<?php
/**
 * EDM E-Fatura Premium Logger Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling logging
 */
class EDM_Efatura_Premium_Logger {
    /**
     * Log levels
     *
     * @var array
     */
    private $levels = array(
        'emergency' => 0,
        'alert'     => 1,
        'critical'  => 2,
        'error'     => 3,
        'warning'   => 4,
        'notice'    => 5,
        'info'      => 6,
        'debug'     => 7,
    );
    private $log_count = 0;
	private $max_daily_logs = 100; // Günlük maksimum log sayısı
	
	/**
	 * Log limitini kontrol et
	 *
	 * @return bool Log yazılabilir mi?
	 */
	private function check_log_limit() {
		// Bugünün log sayacını kontrol et
		$today = date('Y-m-d');
		$daily_count_key = 'edm_daily_log_count_' . $today;
		$this->log_count = get_option($daily_count_key, 0);
		
		// Eğer limit aşıldıysa false dön
		if ($this->log_count >= $this->max_daily_logs) {
			return false;
		}
		
		// Log sayacını artır
		$this->log_count++;
		update_option($daily_count_key, $this->log_count, false);
		
		return true;
	}
	
    /**
     * Current log level
     *
     * @var string
     */
    private $current_level = 'info';
    
    /**
     * Store logged messages to prevent duplicates
     *
     * @var array
     */
    private static $logged_messages = array();
    
    /**
     * Constructor
     */
    public function __construct() {
        $settings = get_option('edm_efatura_premium_settings', array());
        if (isset($settings['log_level'])) {
            $this->current_level = $settings['log_level'];
        }
    }
    
    /**
     * Log an emergency message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function emergency($message, $context = array()) {
        $this->log('emergency', $message, $context);
    }
    
    /**
     * Log an alert message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function alert($message, $context = array()) {
        $this->log('alert', $message, $context);
    }
    
    /**
     * Log a critical message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function critical($message, $context = array()) {
        $this->log('critical', $message, $context);
    }
    
    /**
     * Log an error message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function error($message, $context = array()) {
        // Add stack trace to context when in debug mode
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $e = new Exception();
            $context['stack_trace'] = $e->getTraceAsString();
        }
        
        $this->log('error', $message, $context);
        
        // error_log is handled by the main log() method
    }
        
    /**
     * Log a warning message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function warning($message, $context = array()) {
        $this->log('warning', $message, $context);
    }
    
    /**
     * Log a notice message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function notice($message, $context = array()) {
        $this->log('notice', $message, $context);
    }
    
    /**
     * Log an info message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function info($message, $context = array()) {
        $this->log('info', $message, $context);
    }
    
    /**
     * Log a debug message
     *
     * @param string $message The log message
     * @param array $context Context data
     */
    public function debug($message, $context = array()) {
        $this->log('debug', $message, $context);
        
        // Debug messages only go to the database, not error_log to avoid clutter
    }
    
    /**
     * Log a message with any level
     *
     * @param string $level Log level
     * @param string $message The log message
     * @param array $context Context data
     */
    public function log($level, $message, $context = array()) {
		// Check if this level should be logged
		if (!$this->should_log($level)) {
			return;
		}
		
		// Sadece error ve warning düzeyindeki loglar için limiti aşmaya izin ver
		if (!in_array($level, ['error', 'warning']) && !$this->check_log_limit()) {
			return;
		}
        
        // Check for duplicate messages to avoid log flooding
        $context_key = !empty($context) ? md5(json_encode($context)) : '';
        $message_key = md5($level . $message . $context_key);
        
        // If this exact message was logged in the last 60 seconds, skip it
        if (isset(self::$logged_messages[$message_key])) {
            $time_diff = time() - self::$logged_messages[$message_key];
            if ($time_diff < 60) {
                return;
            }
        }
        
        // Update the timestamp for this message
        self::$logged_messages[$message_key] = time();
        
        // Keep the logged_messages array from growing too large
        if (count(self::$logged_messages) > 100) {
            // Keep only the 50 most recent messages
            self::$logged_messages = array_slice(self::$logged_messages, -50, 50, true);
        }
        
        // Format context data
        $context_string = '';
        if (!empty($context) && is_array($context)) {
            // Limit large context data to prevent overflow
            foreach ($context as $key => &$value) {
                if (is_string($value) && strlen($value) > 5000) {
                    $value = substr($value, 0, 5000) . '... [truncated]';
                }
            }
            
            $context_string = ' - ' . wp_json_encode($context, JSON_UNESCAPED_UNICODE);
        }
        
        // Format log entry
        $log_entry = array(
            'level' => strtoupper($level),
            'message' => $message,
            'context' => $context,
            'time' => current_time('mysql')
        );
        
        // Store in database
        try {
            $this->save_to_db($log_entry);
        } catch (Exception $e) {
            // If database logging fails, write to error log
            $error_context = array(
                'previous_log' => $log_entry,
                'db_error' => $e->getMessage()
            );
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log(sprintf('[EDM E-Fatura Premium] [DB ERROR] Failed to save log: %s', $e->getMessage()));
            }
        }
        
        // Logs are now only stored in database, not written to debug.log
    }

    
    /**
     * Save log entry to database
     *
     * @param array $log_entry Log entry data
     */
    private function save_to_db($log_entry) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_LOG;
        
        // Check if table exists, create if it doesn't
        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
            $this->create_log_table();
        }
        
        // Prepare context data
        $context_json = !empty($log_entry['context']) ? wp_json_encode($log_entry['context']) : null;
        
        // Insert log entry
        $result = $wpdb->insert(
            $table_name,
            array(
                'level' => $log_entry['level'],
                'message' => $log_entry['message'],
                'context' => $context_json,
                'created_at' => $log_entry['time']
            ),
            array(
                '%s',
                '%s',
                '%s',
                '%s'
            )
        );
        
        if ($result === false) {
            throw new Exception($wpdb->last_error);
        }
    }
    
    
    /**
     * Create log table if it doesn't exist
     */
    private function create_log_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_LOG;
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            level varchar(20) NOT NULL,
            message text NOT NULL,
            context longtext DEFAULT NULL,
            created_at datetime NOT NULL,
            PRIMARY KEY  (id),
            KEY level (level),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Check if the given level should be logged
     *
     * @param string $level Log level
     * @return bool Whether to log or not
     */
    private function should_log($level) {
        return isset($this->levels[$level]) && $this->levels[$level] <= $this->levels[$this->current_level];
    }
    
    /**
     * Get all logs with pagination
     *
     * @param int $page Page number
     * @param int $per_page Items per page
     * @param string $level Filter by level
     * @return array Logs and pagination info
     */
    public function get_logs($page = 1, $per_page = 20, $level = '') {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_LOG;
        
        // Prepare query
        $sql = "SELECT * FROM {$table_name}";
        $sql_count = "SELECT COUNT(*) FROM {$table_name}";
        
        // Add level filter if specified
        if (!empty($level)) {
            $sql .= $wpdb->prepare(" WHERE level = %s", $level);
            $sql_count .= $wpdb->prepare(" WHERE level = %s", $level);
        }
        
        // Add order and pagination
        $sql .= " ORDER BY created_at DESC";
        $sql .= $wpdb->prepare(" LIMIT %d OFFSET %d", $per_page, ($page - 1) * $per_page);
        
        // Get results
        $logs = $wpdb->get_results($sql, ARRAY_A);
        $total = $wpdb->get_var($sql_count);
        
        // Parse context JSON
        foreach ($logs as &$log) {
            $log['context'] = json_decode($log['context'], true);
        }
        
        return array(
            'logs' => $logs,
            'total' => (int) $total,
            'pages' => ceil($total / $per_page),
            'page' => $page,
            'per_page' => $per_page
        );
    }
    
    /**
     * Clean old logs
     *
     * @param int $days Keep logs for this many days
     * @return int Number of deleted logs
     */
    public function clean_logs($days = 30) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_LOG;
        
        $date = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        $result = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$table_name} WHERE created_at < %s",
                $date
            )
        );
        
        return $result;
    }
    
    /**
     * Delete all logs
     *
     * @return int Number of deleted logs
     */
    public function delete_all_logs() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . EDM_EFATURA_PREMIUM_TABLE_LOG;
        
        $result = $wpdb->query("DELETE FROM {$table_name}");
        
        // Reset auto-increment counter
        $wpdb->query("ALTER TABLE {$table_name} AUTO_INCREMENT = 1");
        
        return $result;
    }
}