<?php
/**
 * EDM E-Fatura Premium Performance Management Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling performance optimizations
 */
class EDM_Efatura_Premium_Performance {
    
    /**
     * Memory usage tracking
     */
    private static $memory_snapshots = array();
    
    /**
     * Execution time tracking
     */
    private static $time_markers = array();
    
    /**
     * Performance statistics
     */
    private static $performance_stats = array(
        'memory_peak' => 0,
        'execution_times' => array(),
        'db_queries' => 0,
        'cache_operations' => 0
    );
    
    /**
     * Initialize performance monitoring
     */
    public static function init() {
        // Hook into WordPress to track performance
        add_action('init', array(__CLASS__, 'start_monitoring'), 1);
        add_action('wp_footer', array(__CLASS__, 'log_performance_stats'), 9999);
        add_action('admin_footer', array(__CLASS__, 'log_performance_stats'), 9999);
        
        // Memory management hooks
        add_action('edm_before_large_operation', array(__CLASS__, 'prepare_for_large_operation'));
        add_action('edm_after_large_operation', array(__CLASS__, 'cleanup_after_large_operation'));
        
        // Database query optimization hooks
        add_filter('posts_request', array(__CLASS__, 'log_db_query'));
        
        self::mark_time('performance_init');
    }
    
    /**
     * Start performance monitoring
     */
    public static function start_monitoring() {
        self::take_memory_snapshot('initial');
        self::mark_time('monitoring_start');
        
        // Enable query logging if in debug mode
        if (defined('WP_DEBUG') && WP_DEBUG) {
            define('SAVEQUERIES', true);
        }
    }
    
    /**
     * Mark a time point for performance measurement
     *
     * @param string $marker_name Name of the time marker
     */
    public static function mark_time($marker_name) {
        self::$time_markers[$marker_name] = array(
            'time' => microtime(true),
            'memory' => memory_get_usage(),
            'memory_peak' => memory_get_peak_usage()
        );
    }
    
    /**
     * Mark memory usage for tracking
     * Alias for take_memory_snapshot for backward compatibility
     *
     * @param string $marker_name Name of the memory marker
     */
    public static function mark_memory($marker_name) {
        self::take_memory_snapshot($marker_name);
    }
    
    /**
     * Get elapsed time between two markers
     *
     * @param string $start_marker Start marker name
     * @param string $end_marker End marker name (current time if null)
     * @return float Elapsed time in seconds
     */
    public static function get_elapsed_time($start_marker, $end_marker = null) {
        if (!isset(self::$time_markers[$start_marker])) {
            return 0;
        }
        
        $start_time = self::$time_markers[$start_marker]['time'];
        
        if ($end_marker === null) {
            $end_time = microtime(true);
        } elseif (isset(self::$time_markers[$end_marker])) {
            $end_time = self::$time_markers[$end_marker]['time'];
        } else {
            return 0;
        }
        
        return $end_time - $start_time;
    }
    
    /**
     * Take memory usage snapshot
     *
     * @param string $snapshot_name Name of the snapshot
     */
    public static function take_memory_snapshot($snapshot_name) {
        self::$memory_snapshots[$snapshot_name] = array(
            'usage' => memory_get_usage(),
            'usage_real' => memory_get_usage(true),
            'peak' => memory_get_peak_usage(),
            'peak_real' => memory_get_peak_usage(true),
            'time' => microtime(true)
        );
        
        // Update peak memory tracking
        if (self::$memory_snapshots[$snapshot_name]['peak'] > self::$performance_stats['memory_peak']) {
            self::$performance_stats['memory_peak'] = self::$memory_snapshots[$snapshot_name]['peak'];
        }
    }
    
    /**
     * Get memory usage difference between snapshots
     *
     * @param string $start_snapshot Start snapshot name
     * @param string $end_snapshot End snapshot name (current if null)
     * @return array Memory usage difference
     */
    public static function get_memory_diff($start_snapshot, $end_snapshot = null) {
        if (!isset(self::$memory_snapshots[$start_snapshot])) {
            return array('usage' => 0, 'peak' => 0);
        }
        
        if ($end_snapshot === null) {
            $end_memory = array(
                'usage' => memory_get_usage(),
                'peak' => memory_get_peak_usage()
            );
        } elseif (isset(self::$memory_snapshots[$end_snapshot])) {
            $end_memory = self::$memory_snapshots[$end_snapshot];
        } else {
            return array('usage' => 0, 'peak' => 0);
        }
        
        return array(
            'usage' => $end_memory['usage'] - self::$memory_snapshots[$start_snapshot]['usage'],
            'peak' => $end_memory['peak'] - self::$memory_snapshots[$start_snapshot]['peak']
        );
    }
    
    /**
     * Prepare system for large operations
     *
     * @param string $operation_name Operation name
     */
    public static function prepare_for_large_operation($operation_name = 'large_operation') {
        // Take memory snapshot before operation
        self::take_memory_snapshot('before_' . $operation_name);
        self::mark_time('start_' . $operation_name);
        
        // Increase memory limit if needed
        $current_limit = ini_get('memory_limit');
        $current_bytes = self::convert_to_bytes($current_limit);
        $required_bytes = 256 * 1024 * 1024; // 256MB
        
        if ($current_bytes < $required_bytes) {
            ini_set('memory_limit', '256M');
        }
        
        // Increase execution time limit
        if (function_exists('set_time_limit')) {
            set_time_limit(300); // 5 minutes
        }
        
        // Clear any existing output buffers
        while (ob_get_level() > 0) {
            ob_end_clean();
        }
        
        // Force garbage collection
        if (function_exists('gc_collect_cycles')) {
            gc_collect_cycles();
        }
    }
    
    /**
     * Cleanup after large operations
     *
     * @param string $operation_name Operation name
     */
    public static function cleanup_after_large_operation($operation_name = 'large_operation') {
        // Take memory snapshot after operation
        self::take_memory_snapshot('after_' . $operation_name);
        self::mark_time('end_' . $operation_name);
        
        // Log performance metrics
        $memory_diff = self::get_memory_diff('before_' . $operation_name, 'after_' . $operation_name);
        $execution_time = self::get_elapsed_time('start_' . $operation_name, 'end_' . $operation_name);
        
        self::$performance_stats['execution_times'][$operation_name] = $execution_time;
        
        // Log if operation took too long or used too much memory
        if ($execution_time > 30) { // 30 seconds
        }
        
        if ($memory_diff['usage'] > 50 * 1024 * 1024) { // 50MB
        }
        
        // Force garbage collection
        if (function_exists('gc_collect_cycles')) {
            gc_collect_cycles();
        }
    }
    
    /**
     * Optimize database query performance
     *
     * @param string $query SQL query
     * @return string Optimized query
     */
    public static function optimize_db_query($query) {
        // Add LIMIT if not present in SELECT queries without LIMIT
        if (stripos($query, 'SELECT') === 0 && 
            stripos($query, 'LIMIT') === false && 
            stripos($query, 'COUNT(') === false) {
            
            // Add reasonable LIMIT to prevent runaway queries
            $query .= ' LIMIT 1000';
        }
        
        return $query;
    }
    
    /**
     * Log database query for performance monitoring
     *
     * @param string $query SQL query
     * @return string Unmodified query
     */
    public static function log_db_query($query) {
        self::$performance_stats['db_queries']++;
        
        // Log slow queries (this is a simplified check)
        $start_time = microtime(true);
        
        // We can't actually time the query here since this is just a filter
        // But we can log queries that might be problematic
        if (stripos($query, 'SELECT * FROM') !== false && 
            stripos($query, 'LIMIT') === false) {
            
        }
        
        return $query;
    }
    
    /**
     * Implement cursor-based pagination for large datasets
     *
     * @param string $table_name Table name
     * @param array $conditions Where conditions
     * @param int $limit Number of records per page
     * @param int $cursor Last record ID from previous page
     * @param string $order_column Column to order by (default: id)
     * @return array Query results
     */
    public static function cursor_paginate($table_name, $conditions = array(), $limit = 20, $cursor = 0, $order_column = 'id') {
        global $wpdb;
        
        $where_clauses = array();
        $where_values = array();
        
        // Add cursor condition
        if ($cursor > 0) {
            $where_clauses[] = "{$order_column} > %d";
            $where_values[] = $cursor;
        }
        
        // Add other conditions
        foreach ($conditions as $column => $value) {
            $where_clauses[] = "{$column} = %s";
            $where_values[] = $value;
        }
        
        // Build query
        $sql = "SELECT * FROM {$table_name}";
        
        if (!empty($where_clauses)) {
            $sql .= ' WHERE ' . implode(' AND ', $where_clauses);
        }
        
        $sql .= " ORDER BY {$order_column} ASC LIMIT %d";
        $where_values[] = $limit + 1; // Get one extra to check if there are more records
        
        $results = $wpdb->get_results($wpdb->prepare($sql, $where_values), ARRAY_A);
        
        $has_more = false;
        if (count($results) > $limit) {
            $has_more = true;
            array_pop($results); // Remove the extra record
        }
        
        $next_cursor = 0;
        if (!empty($results)) {
            $last_record = end($results);
            $next_cursor = $last_record[$order_column];
        }
        
        return array(
            'data' => $results,
            'has_more' => $has_more,
            'next_cursor' => $next_cursor,
            'count' => count($results)
        );
    }
    
    /**
     * Stream large data processing
     *
     * @param resource $handle File handle or stream
     * @param callable $processor Callback to process each chunk
     * @param int $chunk_size Chunk size in bytes
     */
    public static function stream_process($handle, $processor, $chunk_size = 8192) {
        if (!is_resource($handle)) {
            throw new InvalidArgumentException('Invalid handle provided');
        }
        
        while (!feof($handle)) {
            $chunk = fread($handle, $chunk_size);
            if ($chunk !== false && $chunk !== '') {
                call_user_func($processor, $chunk);
            }
            
            // Give other processes a chance and prevent memory buildup
            if (function_exists('fastcgi_finish_request')) {
                fastcgi_finish_request();
            }
            
            // Force garbage collection periodically
            if (mt_rand(1, 100) === 1) {
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
            }
        }
    }
    
    /**
     * Optimize PDF generation memory usage
     *
     * @param string $html HTML content
     * @param array $options PDF generation options
     * @return string PDF content or file path
     */
    public static function generate_pdf_optimized($html, $options = array()) {
        // Prepare for large operation
        self::prepare_for_large_operation('pdf_generation');
        
        try {
            // If HTML is too large, save to temp file first
            if (strlen($html) > 1024 * 1024) { // 1MB
                $temp_file = wp_tempnam('edm_html_');
                file_put_contents($temp_file, $html);
                
                // Process in chunks if needed
                $html = null; // Free memory
                unset($html);
                
                // Use file-based processing
                $pdf_content = self::process_large_html_file($temp_file, $options);
                
                // Cleanup
                unlink($temp_file);
            } else {
                // Regular processing for smaller HTML
                $pdf_content = self::process_html_to_pdf($html, $options);
            }
            
            return $pdf_content;
            
        } finally {
            // Always cleanup after operation
            self::cleanup_after_large_operation('pdf_generation');
        }
    }
    
    /**
     * Stream large data directly to output
     *
     * @param string $data Data to stream
     * @param int $chunk_size Chunk size for streaming
     */
    public static function stream_large_data($data, $chunk_size = 8192) {
        $length = strlen($data);
        $offset = 0;
        
        while ($offset < $length) {
            $chunk = substr($data, $offset, $chunk_size);
            echo $chunk;
            
            // Flush output to prevent memory buildup
            if (ob_get_level()) {
                ob_flush();
            }
            flush();
            
            $offset += $chunk_size;
            
            // Give system a break and prevent timeout
            usleep(1000); // 1ms pause
        }
    }
    
    /**
     * Process large HTML file to PDF
     *
     * @param string $html_file Path to HTML file
     * @param array $options Options
     * @return string PDF content
     */
    private static function process_large_html_file($html_file, $options) {
        // This would implement the actual PDF generation
        // For now, return a placeholder
        return "PDF generation optimized for file: {$html_file}";
    }
    
    /**
     * Process HTML to PDF (regular size)
     *
     * @param string $html HTML content
     * @param array $options Options
     * @return string PDF content
     */
    private static function process_html_to_pdf($html, $options) {
        // This would implement the actual PDF generation
        // For now, return a placeholder
        return "PDF generation for " . strlen($html) . " bytes of HTML";
    }
    
    /**
     * Get current performance statistics
     *
     * @return array Performance statistics
     */
    public static function get_performance_stats() {
        $stats = self::$performance_stats;
        
        // Add current memory usage
        $stats['current_memory'] = memory_get_usage();
        $stats['current_memory_peak'] = memory_get_peak_usage();
        
        // Add time markers
        $stats['time_markers'] = self::$time_markers;
        
        // Add memory snapshots
        $stats['memory_snapshots'] = self::$memory_snapshots;
        
        return $stats;
    }
    
    /**
     * Log performance statistics
     */
    public static function log_performance_stats() {
        if (!defined('WP_DEBUG') || !WP_DEBUG) {
            return;
        }
        
        $stats = self::get_performance_stats();
        
        // Only log if there was significant activity
        if ($stats['db_queries'] > 10 || $stats['current_memory_peak'] > 32 * 1024 * 1024) { // 32MB
            // Performance logging removed for cleaner output
        }
    }
    
    /**
     * Convert memory limit string to bytes
     *
     * @param string $memory_limit Memory limit string (e.g., "256M", "1G")
     * @return int Memory limit in bytes
     */
    private static function convert_to_bytes($memory_limit) {
        $memory_limit = trim($memory_limit);
        $last_char = strtolower($memory_limit[strlen($memory_limit) - 1]);
        $value = (int) $memory_limit;
        
        switch ($last_char) {
            case 'g':
                $value *= 1024;
            case 'm':
                $value *= 1024;
            case 'k':
                $value *= 1024;
        }
        
        return $value;
    }
    
    /**
     * Check if system resources are under pressure
     *
     * @return array Resource status
     */
    public static function check_resource_pressure() {
        $memory_limit = self::convert_to_bytes(ini_get('memory_limit'));
        $current_usage = memory_get_usage();
        $memory_percentage = ($current_usage / $memory_limit) * 100;
        
        $max_execution_time = ini_get('max_execution_time');
        $current_time = microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'];
        $time_percentage = $max_execution_time > 0 ? ($current_time / $max_execution_time) * 100 : 0;
        
        return array(
            'memory' => array(
                'usage' => $current_usage,
                'limit' => $memory_limit,
                'percentage' => $memory_percentage,
                'under_pressure' => $memory_percentage > 80
            ),
            'time' => array(
                'elapsed' => $current_time,
                'limit' => $max_execution_time,
                'percentage' => $time_percentage,
                'under_pressure' => $time_percentage > 80
            )
        );
    }
    
    /**
     * Optimize array processing for large datasets
     *
     * @param array $data Large dataset
     * @param callable $processor Processing function
     * @param int $batch_size Batch size for processing
     * @return array Processed results
     */
    public static function batch_process_array($data, $processor, $batch_size = 100) {
        $results = array();
        $chunks = array_chunk($data, $batch_size);
        
        foreach ($chunks as $chunk_index => $chunk) {
            $chunk_results = call_user_func($processor, $chunk);
            
            if (is_array($chunk_results)) {
                $results = array_merge($results, $chunk_results);
            }
            
            // Memory management between chunks
            if (($chunk_index + 1) % 10 === 0) {
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
            }
            
            // Check resource pressure
            $resource_status = self::check_resource_pressure();
            if ($resource_status['memory']['under_pressure']) {
                break;
            }
        }
        
        return $results;
    }
    
    /**
     * Optimize large CSV export processing
     *
     * @param array $data Data to export
     * @param string $filename Target filename
     * @param array $headers CSV headers
     * @return bool Success status
     */
    public static function export_large_csv($data, $filename, $headers = array()) {
        self::prepare_for_large_operation('csv_export');
        
        try {
            // Use output buffering to prevent memory issues
            $temp_file = wp_tempnam('edm_csv_');
            $handle = fopen($temp_file, 'w');
            
            if (!$handle) {
                return false;
            }
            
            // Write headers
            if (!empty($headers)) {
                fputcsv($handle, $headers);
            }
            
            // Process data in chunks
            $chunk_size = 1000;
            $chunks = array_chunk($data, $chunk_size);
            
            foreach ($chunks as $chunk) {
                foreach ($chunk as $row) {
                    fputcsv($handle, $row);
                }
                
                // Memory management
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
                
                // Check resource pressure
                $resource_status = self::check_resource_pressure();
                if ($resource_status['memory']['under_pressure']) {
                    fclose($handle);
                    unlink($temp_file);
                    return false;
                }
            }
            
            fclose($handle);
            
            // Move temp file to final location
            $upload_dir = wp_upload_dir();
            $target_path = $upload_dir['path'] . '/' . $filename;
            
            if (copy($temp_file, $target_path)) {
                unlink($temp_file);
                return $target_path;
            } else {
                unlink($temp_file);
                return false;
            }
            
        } finally {
            self::cleanup_after_large_operation('csv_export');
        }
    }
    
    /**
     * Handle large file uploads with progress tracking
     *
     * @param array $file_data File data from $_FILES
     * @param int $max_size Maximum file size in bytes
     * @return array Upload result
     */
    public static function handle_large_file_upload($file_data, $max_size = 50 * 1024 * 1024) { // 50MB default
        self::prepare_for_large_operation('file_upload');
        
        try {
            // Validate file
            if (!isset($file_data['tmp_name']) || !is_uploaded_file($file_data['tmp_name'])) {
                return array('success' => false, 'error' => 'Invalid file upload');
            }
            
            $file_size = filesize($file_data['tmp_name']);
            if ($file_size > $max_size) {
                return array('success' => false, 'error' => 'File too large');
            }
            
            // Process in chunks for large files
            if ($file_size > 5 * 1024 * 1024) { // 5MB
                return self::process_large_file_upload($file_data, $file_size);
            } else {
                return self::process_regular_file_upload($file_data);
            }
            
        } finally {
            self::cleanup_after_large_operation('file_upload');
        }
    }
    
    /**
     * Process large file upload
     *
     * @param array $file_data File data
     * @param int $file_size File size
     * @return array Result
     */
    private static function process_large_file_upload($file_data, $file_size) {
        $upload_dir = wp_upload_dir();
        $target_path = $upload_dir['path'] . '/' . sanitize_file_name($file_data['name']);
        
        $source = fopen($file_data['tmp_name'], 'rb');
        $target = fopen($target_path, 'wb');
        
        if (!$source || !$target) {
            return array('success' => false, 'error' => 'Cannot open file for processing');
        }
        
        $chunk_size = 8192;
        $bytes_copied = 0;
        
        while (!feof($source)) {
            $chunk = fread($source, $chunk_size);
            if ($chunk !== false) {
                fwrite($target, $chunk);
                $bytes_copied += strlen($chunk);
            }
            
            // Progress tracking (could be used with AJAX)
            $progress = ($bytes_copied / $file_size) * 100;
            
            // Memory management
            if ($bytes_copied % (1024 * 1024) === 0) { // Every 1MB
                if (function_exists('gc_collect_cycles')) {
                    gc_collect_cycles();
                }
            }
        }
        
        fclose($source);
        fclose($target);
        
        return array(
            'success' => true,
            'file_path' => $target_path,
            'file_size' => $bytes_copied,
            'file_url' => $upload_dir['url'] . '/' . sanitize_file_name($file_data['name'])
        );
    }
    
    /**
     * Process regular file upload
     *
     * @param array $file_data File data
     * @return array Result
     */
    private static function process_regular_file_upload($file_data) {
        $upload_dir = wp_upload_dir();
        $target_path = $upload_dir['path'] . '/' . sanitize_file_name($file_data['name']);
        
        if (move_uploaded_file($file_data['tmp_name'], $target_path)) {
            return array(
                'success' => true,
                'file_path' => $target_path,
                'file_size' => filesize($target_path),
                'file_url' => $upload_dir['url'] . '/' . sanitize_file_name($file_data['name'])
            );
        } else {
            return array('success' => false, 'error' => 'Failed to move uploaded file');
        }
    }
    
    /**
     * Log operation completion for performance monitoring
     *
     * @param string $operation_name Operation name
     * @param bool $success Whether operation was successful
     * @param array $additional_data Additional data to log
     */
    public static function log_operation_completion($operation_name, $success = true, $additional_data = array()) {
        // Take final memory snapshot
        self::take_memory_snapshot('completion_' . $operation_name);
        self::mark_time('completion_' . $operation_name);
        
        // Calculate performance metrics
        $start_marker = 'start_' . $operation_name;
        $completion_marker = 'completion_' . $operation_name;
        
        $execution_time = 0;
        $memory_usage = 0;
        
        if (isset(self::$time_markers[$start_marker])) {
            $execution_time = self::get_elapsed_time($start_marker, $completion_marker);
        }
        
        if (isset(self::$memory_snapshots['before_' . $operation_name])) {
            $memory_diff = self::get_memory_diff('before_' . $operation_name, 'completion_' . $operation_name);
            $memory_usage = $memory_diff['usage'];
        }
        
        // Store operation statistics
        if (!isset(self::$performance_stats['operations'])) {
            self::$performance_stats['operations'] = array();
        }
        
        self::$performance_stats['operations'][$operation_name] = array(
            'success' => $success,
            'execution_time' => $execution_time,
            'memory_usage' => $memory_usage,
            'timestamp' => current_time('timestamp'),
            'additional_data' => $additional_data
        );
        
        // Log if operation was significant or failed
        if (!$success || $execution_time > 5 || $memory_usage > 10 * 1024 * 1024) { // 5 seconds or 10MB
            $log_data = array(
                'operation' => $operation_name,
                'success' => $success,
                'execution_time' => round($execution_time, 3),
                'memory_usage' => self::format_bytes($memory_usage),
                'timestamp' => current_time('mysql')
            );
            
            if (!empty($additional_data)) {
                $log_data['additional_data'] = $additional_data;
            }
            
            // Performance data logged to system logger instead of error_log
        }
        
        // Force garbage collection after logging
        if (function_exists('gc_collect_cycles')) {
            gc_collect_cycles();
        }
    }
    
    /**
     * Format bytes to human readable format
     *
     * @param int $bytes Number of bytes
     * @param int $precision Decimal precision
     * @return string Formatted string
     */
    private static function format_bytes($bytes, $precision = 2) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
}