<?php
/**
 * EDM E-Fatura Premium Security Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling security operations
 */
class EDM_Efatura_Premium_Security {
    
    /**
     * Encrypt sensitive data
     *
     * @param string $data Data to encrypt
     * @return string Encrypted data
     */
    public static function encrypt($data) {
        if (empty($data)) {
            return '';
        }
        
        $key = self::get_encryption_key();
        $method = 'AES-256-CBC';
        $iv = openssl_random_pseudo_bytes(openssl_cipher_iv_length($method));
        $encrypted = openssl_encrypt($data, $method, $key, 0, $iv);
        
        return base64_encode($iv . $encrypted);
    }
    
    /**
     * Decrypt sensitive data
     *
     * @param string $data Encrypted data
     * @return string Decrypted data
     */
    public static function decrypt($data) {
        if (empty($data)) {
            return '';
        }
        
        // Check if data is already plain text (not base64 encoded)
        // If base64_decode fails or returns invalid data, assume it's plain text
        $decoded = base64_decode($data, true);
        if ($decoded === false || !self::is_encrypted_data($decoded)) {
            // Return original data as plain text
            return $data;
        }
        
        $key = self::get_encryption_key();
        $method = 'AES-256-CBC';
        $iv_length = openssl_cipher_iv_length($method);
        $iv = substr($decoded, 0, $iv_length);
        $encrypted = substr($decoded, $iv_length);
        
        $decrypted = openssl_decrypt($encrypted, $method, $key, 0, $iv);
        
        // If decryption fails, return original data
        return $decrypted !== false ? $decrypted : $data;
    }
    
    /**
     * Check if data is encrypted format
     *
     * @param string $data Data to check
     * @return bool True if data appears to be encrypted
     */
    private static function is_encrypted_data($data) {
        $method = 'AES-256-CBC';
        $iv_length = openssl_cipher_iv_length($method);
        
        // Check if data has minimum length for encrypted data
        return strlen($data) > $iv_length;
    }
    
    /**
     * Get or create encryption key
     *
     * @return string Encryption key
     */
    private static function get_encryption_key() {
        $key = get_option('edm_efatura_encryption_key');
        
        if (empty($key)) {
            $key = wp_generate_password(32, true, true);
            update_option('edm_efatura_encryption_key', $key, false);
        }
        
        return $key;
    }
    
    /**
     * Validate Turkish TC number
     *
     * @param string $tc_number TC number
     * @return bool Valid or not
     */
    public static function validate_tc_number($tc_number) {
        // Remove non-numeric characters
        $tc_number = preg_replace('/[^0-9]/', '', $tc_number);
        
        // Check if 11 digits
        if (strlen($tc_number) !== 11) {
            return false;
        }
        
        // Check if all digits are same
        if (preg_match('/^(\d)\1{10}$/', $tc_number)) {
            return false;
        }
        
        // Calculate checksum
        $digits = str_split($tc_number);
        $sum1 = $digits[0] + $digits[2] + $digits[4] + $digits[6] + $digits[8];
        $sum2 = $digits[1] + $digits[3] + $digits[5] + $digits[7];
        
        $check1 = ($sum1 * 7 - $sum2) % 10;
        if ($check1 != $digits[9]) {
            return false;
        }
        
        $total = array_sum(array_slice($digits, 0, 10));
        if (($total % 10) != $digits[10]) {
            return false;
        }
        
        return true;
    }
    
    /**
     * Validate Turkish VKN (Tax Number)
     *
     * @param string $vkn VKN number
     * @return bool Valid or not
     */
    public static function validate_vkn($vkn) {
        // Remove non-numeric characters
        $vkn = preg_replace('/[^0-9]/', '', $vkn);
        
        // Check if 10 or 11 digits
        if (strlen($vkn) < 10 || strlen($vkn) > 11) {
            return false;
        }
        
        // If 11 digits, validate as TC number
        if (strlen($vkn) === 11) {
            return self::validate_tc_number($vkn);
        }
        
        // VKN validation algorithm
        $digits = str_split($vkn);
        $v = array(9, 8, 7, 6, 5, 4, 3, 2, 1);
        $sum = 0;
        
        for ($i = 0; $i < 9; $i++) {
            $sum += $digits[$i] * $v[$i];
        }
        
        $mod = $sum % 11;
        $checksum = $mod < 2 ? $mod : 11 - $mod;
        
        return $checksum == $digits[9];
    }
    
    /**
     * Validate email address
     *
     * @param string $email Email address
     * @return bool Valid or not
     */
    public static function validate_email($email) {
        return is_email($email) !== false;
    }
    
    /**
     * Sanitize Turkish text
     *
     * @param string $text Text to sanitize
     * @return string Sanitized text
     */
    public static function sanitize_turkish_text($text) {
        // Allow Turkish characters
        $text = sanitize_text_field($text);
        
        // Additional validation for Turkish characters
        $allowed_chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789 çÇğĞıİöÖşŞüÜ.,!?()-_';
        $text = preg_replace('/[^' . preg_quote($allowed_chars, '/') . ']/', '', $text);
        
        return $text;
    }
    
    /**
     * Generate secure nonce for specific action and context
     *
     * @param string $action Action name
     * @param string $context Additional context
     * @return string Nonce
     */
    public static function create_nonce($action, $context = '') {
        return wp_create_nonce($action . '_' . $context . '_' . get_current_user_id());
    }
    
    /**
     * Verify secure nonce
     *
     * @param string $nonce Nonce to verify
     * @param string $action Action name
     * @param string $context Additional context
     * @return bool Valid or not
     */
    public static function verify_nonce($nonce, $action, $context = '') {
        return wp_verify_nonce($nonce, $action . '_' . $context . '_' . get_current_user_id()) !== false;
    }
    
    /**
     * Check if user has required capabilities for invoice operations
     *
     * @param string $operation Operation type
     * @return bool Has capability or not
     */
    public static function can_user_perform_invoice_operation($operation = 'view') {
        switch ($operation) {
            case 'create':
            case 'send':
            case 'cancel':
                return current_user_can('manage_woocommerce') || current_user_can('manage_options');
                
            case 'view':
            case 'download':
                return current_user_can('read_private_posts') || current_user_can('manage_woocommerce') || current_user_can('manage_options');
                
            case 'admin':
                return current_user_can('manage_options');
                
            default:
                return false;
        }
    }
    
    /**
     * Rate limiting for sensitive operations
     *
     * @param string $operation Operation name
     * @param int $limit Number of allowed operations
     * @param int $window Time window in seconds
     * @return bool Operation allowed
     */
    public static function is_rate_limited($operation, $limit = 10, $window = 300) {
        $user_id = get_current_user_id();
        if (!$user_id) {
            $user_id = $_SERVER['REMOTE_ADDR'];
        }
        
        $key = 'edm_rate_limit_' . $operation . '_' . md5($user_id);
        $current_count = get_transient($key);
        
        if ($current_count === false) {
            set_transient($key, 1, $window);
            return false;
        }
        
        if ($current_count >= $limit) {
            return true;
        }
        
        set_transient($key, $current_count + 1, $window);
        return false;
    }
    
    /**
     * Log security events
     *
     * @param string $event Event type
     * @param array $context Event context
     */
    public static function log_security_event($event, $context = array()) {
        $logger = new EDM_Efatura_Premium_Logger();
        
        $security_context = array_merge($context, array(
            'user_id' => get_current_user_id(),
            'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown',
            'timestamp' => current_time('mysql'),
            'security_event' => true
        ));
        
        $logger->warning('Security Event: ' . $event, $security_context);
    }
}