<?php
/**
 * EDM E-Fatura Premium Update Checker
 * 
 * Checks for plugin updates from wcplugin.com and shows admin notices
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class EDM_Efatura_Premium_Update_Checker {
    
    /**
     * Current plugin version
     */
    private $current_version;
    
    /**
     * API endpoint for version checking
     */
    private $api_endpoint = 'https://wcplugin.com/wp-json/edm-usage/v1/check-version';
    
    /**
     * Transient keys
     */
    private $transient_key = 'edm_update_data';
    private $dismiss_key = 'edm_update_dismissed';
    private $check_interval = 300; // 5 minutes for testing (was DAY_IN_SECONDS)
    
    /**
     * Logger instance
     */
    private $logger;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->current_version = EDM_EFATURA_PREMIUM_VERSION;
        $this->logger = new EDM_Efatura_Premium_Logger();
        
        // Hooks
        add_action('admin_init', array($this, 'check_for_updates'));
        add_action('admin_notices', array($this, 'show_update_notice'));
        add_action('wp_ajax_edm_dismiss_update_notice', array($this, 'dismiss_notice'));
        add_action('wp_ajax_edm_force_update_check', array($this, 'ajax_force_check'));
        add_action('edm_daily_update_check', array($this, 'daily_version_check'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_bar_menu', array($this, 'add_admin_bar_menu'), 999);
        
        // Schedule daily check if not already scheduled
        if (!wp_next_scheduled('edm_daily_update_check')) {
            wp_schedule_event(time(), 'daily', 'edm_daily_update_check');
        }
    }
    
    /**
     * Check for updates
     */
    public function check_for_updates($force = false) {
        // Don't check on non-admin pages
        if (!is_admin()) {
            return false;
        }
        
        // Check cache first unless forced
        if (!$force) {
            $cached = get_transient($this->transient_key);
            if ($cached !== false) {
                return $cached;
            }
        }
        
        // Check if API endpoint is reachable
        if (!$this->is_endpoint_reachable()) {
            $this->logger->error('Update check failed: API endpoint not reachable');
            return false;
        }
        
        // API call
        $response = wp_remote_post($this->api_endpoint, array(
            'body' => array(
                'current_version' => $this->current_version,
                'site_url' => get_site_url(),
                'wp_version' => get_bloginfo('version'),
                'php_version' => PHP_VERSION
            ),
            'timeout' => 15,
            'sslverify' => false
        ));
        
        if (is_wp_error($response)) {
            $this->logger->error('Update check failed: ' . $response->get_error_message());
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            $this->logger->error('Update check failed: HTTP ' . $response_code);
            return false;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!$data || !isset($data['success']) || !$data['success']) {
            $this->logger->error('Update check failed: Invalid response data');
            return false;
        }
        
        // Cache the result
        set_transient($this->transient_key, $data, $this->check_interval);
        
        $this->logger->info('Update check completed. Has update: ' . ($data['has_update'] ? 'Yes' : 'No'));
        
        return $data;
    }
    
    /**
     * Daily version check (background)
     */
    public function daily_version_check() {
        $this->check_for_updates(true);
    }
    
    /**
     * Check if API endpoint is reachable
     */
    private function is_endpoint_reachable() {
        $test_response = wp_remote_get('https://wcplugin.com', array(
            'timeout' => 10,
            'sslverify' => false
        ));
        
        if (is_wp_error($test_response)) {
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($test_response);
        return ($response_code >= 200 && $response_code < 400);
    }
    
    /**
     * Show update notice in admin
     */
    public function show_update_notice() {
        // Only show on admin pages
        if (!is_admin()) {
            return;
        }
        
        // Don't show on plugin pages to avoid conflicts
        $screen = get_current_screen();
        if ($screen && ($screen->id === 'plugins' || $screen->parent_file === 'plugins.php')) {
            return;
        }
        
        // Check if notice is dismissed
        $dismissed_until = get_option($this->dismiss_key);
        if ($dismissed_until && time() < $dismissed_until) {
            return;
        }
        
        // Get update data
        $update_data = $this->check_for_updates();
        if (!$update_data || !$update_data['has_update']) {
            return;
        }
        
        $this->render_notice($update_data);
    }
    
    /**
     * Render admin notice
     */
    private function render_notice($data) {
        if (!isset($data['update_data'])) {
            return;
        }
        
        $update_info = $data['update_data'];
        $version = $update_info['version'];
        $download_url = $update_info['download_url'];
        $is_critical = $update_info['is_critical'];
        $is_security = $update_info['is_security'];
        $changelog = $update_info['changelog'] ?? '';
        
        // Determine notice type and priority text
        $notice_class = 'notice-info';
        $priority_text = '';
        
        if ($is_security) {
            $notice_class = 'notice-error';
            $priority_text = '<strong style="color: #d63384;">[GÜVENLİK GÜNCELLEMESİ]</strong> ';
        } elseif ($is_critical) {
            $notice_class = 'notice-warning';
            $priority_text = '<strong style="color: #dc3545;">[KRİTİK GÜNCELLEME]</strong> ';
        }
        
        ?>
        <div class="notice <?php echo $notice_class; ?> is-dismissible edm-update-notice" style="position: relative;">
            <div style="display: flex; align-items: center; padding: 5px 0;">
                <div style="margin-right: 15px;">
                    <span class="dashicons dashicons-update" style="font-size: 24px; color: #0073aa;"></span>
                </div>
                <div style="flex: 1;">
                    <p style="margin: 0; font-size: 14px;">
                        <?php echo $priority_text; ?>
                        <strong>EDM E-Fatura Premium</strong> eklentisinin yeni sürümü (<strong><?php echo esc_html($version); ?></strong>) mevcut. 
                        Mevcut sürümünüz: <strong><?php echo esc_html($this->current_version); ?></strong>
                    </p>
                    
                    <?php if (!empty($changelog)): ?>
                    <div class="edm-changelog" style="margin: 8px 0 0 0;">
                        <details>
                            <summary style="cursor: pointer; color: #0073aa;"><strong>Değişiklikleri Gör</strong></summary>
                            <div style="background: #f9f9f9; padding: 10px; margin-top: 8px; border-left: 3px solid #0073aa;">
                                <?php echo wp_kses_post(wpautop($changelog)); ?>
                            </div>
                        </details>
                    </div>
                    <?php endif; ?>
                    
                    <div style="margin-top: 10px;">
                        <a href="<?php echo esc_url($download_url); ?>" class="button button-primary" target="_blank">
                            <span class="dashicons dashicons-download" style="font-size: 16px; margin-right: 5px;"></span>
                            Şimdi İndir
                        </a>
                        <a href="#" class="button edm-dismiss-update" data-nonce="<?php echo wp_create_nonce('edm_dismiss_update'); ?>" style="margin-left: 10px;">
                            Bu Hafta Hatırlatma
                        </a>
                        
                        <?php if ($is_security || $is_critical): ?>
                        <span style="margin-left: 15px; color: #dc3545; font-size: 12px;">
                            ⚠️ Bu güncelleme önemli güvenlik/hata düzeltmeleri içeriyor
                        </span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Dismiss update notice
     */
    public function dismiss_notice() {
        if (!wp_verify_nonce($_POST['nonce'], 'edm_dismiss_update')) {
            wp_die('Security check failed');
        }
        
        // Dismiss for 1 week
        update_option($this->dismiss_key, time() + WEEK_IN_SECONDS);
        
        $this->logger->info('Update notice dismissed by user');
        
        wp_die('OK');
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts() {
        if (!is_admin()) {
            return;
        }
        
        // Add inline script for dismiss functionality and force check
        $script = '
        jQuery(document).ready(function($) {
            // Dismiss functionality
            $(".edm-dismiss-update").click(function(e) {
                e.preventDefault();
                var $notice = $(this).closest(".edm-update-notice");
                var nonce = $(this).data("nonce");
                
                $.post(ajaxurl, {
                    action: "edm_dismiss_update_notice",
                    nonce: nonce
                }, function(response) {
                    if (response === "OK") {
                        $notice.fadeOut(300, function() {
                            $(this).remove();
                        });
                    }
                }).fail(function() {
                    alert("Hatırlatma kapatılamadı. Lütfen tekrar deneyin.");
                });
            });
            
            // Admin bar click handler (redirect to updates page)
            $("#wp-admin-bar-edm-update-check a").click(function(e) {
                // Allow normal link behavior to updates page
                // If Ctrl+Click or middle click, open in new tab
                if (e.ctrlKey || e.metaKey || e.which === 2) {
                    return true;
                }
                
                // Normal click - redirect to updates page
                window.location.href = $(this).attr("href");
                return false;
            });
        });
        ';
        
        wp_add_inline_script('jquery', $script);
        
        // Add CSS for admin bar styling
        $css = '
        #wp-admin-bar-edm-update-check .ab-icon:before {
            font-family: dashicons;
            content: "\\f463";
        }
        #wp-admin-bar-edm-update-check.loading .ab-icon:before {
            content: "\\f469";
            animation: edm-spin 1s linear infinite;
        }
        #wp-admin-bar-edm-update-check .dashicons-warning:before {
            content: "\\f534";
            color: #f56e28;
        }
        @keyframes edm-spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        #wp-admin-bar-edm-update-check a {
            cursor: pointer;
            text-decoration: none;
        }
        #wp-admin-bar-edm-update-check:hover {
            background-color: rgba(240, 245, 250, .04);
        }
        #wp-admin-bar-edm-update-check .ab-label {
            margin-left: 5px;
        }
        ';
        
        wp_add_inline_style('admin-bar', $css);
    }
    
    /**
     * Force check for updates (for manual trigger)
     */
    public function force_check() {
        delete_transient($this->transient_key);
        return $this->check_for_updates(true);
    }
    
    /**
     * Get current update status
     */
    public function get_update_status() {
        $update_data = get_transient($this->transient_key);
        
        if ($update_data === false) {
            return array(
                'status' => 'unknown',
                'message' => 'Update check not performed yet'
            );
        }
        
        if (!$update_data['has_update']) {
            return array(
                'status' => 'up_to_date',
                'message' => 'Plugin is up to date',
                'current_version' => $this->current_version,
                'latest_version' => $update_data['latest_version']
            );
        }
        
        return array(
            'status' => 'update_available',
            'message' => 'Update available',
            'current_version' => $this->current_version,
            'latest_version' => $update_data['latest_version'],
            'update_data' => $update_data['update_data']
        );
    }
    
    /**
     * Add admin bar menu for force check
     */
    public function add_admin_bar_menu($wp_admin_bar) {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Get current status
        $status = $this->get_update_status();
        
        $title = 'EDM Update Check';
        $icon = 'dashicons-update';
        
        if ($status['status'] === 'update_available') {
            $title = 'EDM Update Available!';
            $icon = 'dashicons-warning';
        }
        
        $wp_admin_bar->add_node(array(
            'id' => 'edm-update-check',
            'title' => '<span class="ab-icon ' . $icon . '"></span><span class="ab-label">' . $title . '</span>',
            'href' => admin_url('admin.php?page=edm-efatura-premium-updates'),
            'meta' => array(
                'class' => 'edm-update-check-link'
            )
        ));
    }
    
    /**
     * AJAX handler for force check
     */
    public function ajax_force_check() {
        // Security check
        if (!current_user_can('manage_options') || !wp_verify_nonce($_POST['nonce'], 'edm_force_check')) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => 'Security check failed'
            )));
        }
        
        // Force check
        $result = $this->force_check();
        
        if ($result === false) {
            wp_die(json_encode(array(
                'success' => false,
                'message' => 'Update check failed. Please try again later.'
            )));
        }
        
        $status = $this->get_update_status();
        
        wp_die(json_encode(array(
            'success' => true,
            'status' => $status['status'],
            'message' => $status['message'],
            'current_version' => $this->current_version,
            'latest_version' => $status['latest_version'] ?? 'Unknown',
            'has_update' => ($status['status'] === 'update_available'),
            'update_data' => $status['update_data'] ?? null
        )));
    }
    
    /**
     * Clean up on plugin deactivation
     */
    public function cleanup() {
        delete_transient($this->transient_key);
        delete_option($this->dismiss_key);
        wp_clear_scheduled_hook('edm_daily_update_check');
    }
}