<?php
/**
 * EDM E-Fatura Premium Usage Tracker
 * 
 * Tracks e-invoice creation usage and sends data to wcplugin.com
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class EDM_Efatura_Premium_Usage_Tracker {
    
    /**
     * Table name for usage tracking
     */
    private $table_name;
    
    /**
     * Logger instance
     */
    private $logger;
    
    /**
     * API endpoint for data submission
     */
    private $api_endpoint = 'https://wcplugin.com/wp-json/edm-usage/v1/track';
    
    /**
     * Check if API endpoint is reachable
     */
    private function is_endpoint_reachable() {
        $test_response = wp_remote_get('https://wcplugin.com', array(
            'timeout' => 10,
            'sslverify' => false
        ));
        
        if (is_wp_error($test_response)) {
            $this->logger->error('Endpoint not reachable: ' . $test_response->get_error_message());
            return false;
        }
        
        $response_code = wp_remote_retrieve_response_code($test_response);
        return ($response_code >= 200 && $response_code < 400);
    }
    
    /**
     * Constructor
     */
    public function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'edm_efatura_usage_tracking';
        $this->logger = new EDM_Efatura_Premium_Logger();
        
        // Create table if not exists
        $this->maybe_create_table();
        
        // Hook into invoice creation
        add_action('edm_efatura_invoice_created', array($this, 'track_invoice_creation'), 10, 3);
        
        // Schedule daily data sending
        add_action('wp', array($this, 'schedule_daily_sync'));
        add_action('edm_efatura_daily_usage_sync', array($this, 'send_pending_usage_data'));
        
        // Check if this is first run after activation
        add_action('wp', array($this, 'check_first_run'));
    }
    
    /**
     * Create usage tracking table
     */
    private function maybe_create_table() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            site_url varchar(255) NOT NULL,
            admin_email varchar(255) NOT NULL,
            invoice_uuid varchar(100) NOT NULL,
            invoice_type varchar(50) NOT NULL DEFAULT 'e-invoice',
            created_date datetime DEFAULT CURRENT_TIMESTAMP,
            sent_date datetime NULL,
            is_sent tinyint(1) DEFAULT 0,
            retry_count tinyint(3) DEFAULT 0,
            PRIMARY KEY (id),
            KEY site_url (site_url),
            KEY is_sent (is_sent),
            KEY created_date (created_date)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Track invoice creation
     * 
     * @param string $invoice_uuid Invoice UUID
     * @param string $order_id WooCommerce Order ID
     * @param array $invoice_data Invoice data
     */
    public function track_invoice_creation($invoice_uuid, $order_id = '', $invoice_data = array()) {
        global $wpdb;
        
        try {
            // Get site information
            $site_url = get_site_url();
            $admin_email = get_option('admin_email');
            
            // Determine invoice type
            $invoice_type = 'e-invoice'; // Default
            if (isset($invoice_data['direction'])) {
                if ($invoice_data['direction'] === 'OUT-EARCHIVE') {
                    $invoice_type = 'e-archive';
                }
            }
            
            // Check if this UUID is already tracked
            $existing = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$this->table_name} WHERE invoice_uuid = %s",
                $invoice_uuid
            ));
            
            if ($existing > 0) {
                $this->logger->info('Invoice already tracked, skipping: ' . $invoice_uuid);
                return;
            }
            
            // Insert tracking record
            $result = $wpdb->insert(
                $this->table_name,
                array(
                    'site_url' => $site_url,
                    'admin_email' => $admin_email,
                    'invoice_uuid' => $invoice_uuid,
                    'invoice_type' => $invoice_type,
                    'created_date' => current_time('mysql')
                ),
                array('%s', '%s', '%s', '%s', '%s')
            );
            
            if ($result === false) {
                $this->logger->error('Usage tracking insert failed: ' . $wpdb->last_error);
                return;
            }
            
            $this->logger->info('Invoice creation tracked: ' . $invoice_uuid);
            
            // Try immediate send (non-blocking)
            $this->try_immediate_send();
            
        } catch (Exception $e) {
            $this->logger->error('Usage tracking error: ' . $e->getMessage());
        }
    }
    
    /**
     * Try to send data immediately (non-blocking)
     */
    private function try_immediate_send() {
        // Quick endpoint check first (with cached result)
        static $endpoint_status = null;
        
        if ($endpoint_status === null) {
            $endpoint_status = $this->is_endpoint_reachable();
        }
        
        if (!$endpoint_status) {
            $this->logger->info('Endpoint not reachable, skipping immediate send');
            return;
        }
        
        // Use wp_remote_post with short timeout for immediate attempt
        $pending_data = $this->get_pending_data(5); // Limit to 5 records for immediate send
        
        if (empty($pending_data)) {
            return;
        }
        
        $this->send_usage_data($pending_data, true); // Non-blocking mode
    }
    
    /**
     * Schedule daily sync
     */
    public function schedule_daily_sync() {
        if (!wp_next_scheduled('edm_efatura_daily_usage_sync')) {
            wp_schedule_event(time(), 'daily', 'edm_efatura_daily_usage_sync');
        }
    }
    
    /**
     * Send pending usage data
     */
    public function send_pending_usage_data() {
        $this->logger->info('Starting daily usage data sync');
        
        $pending_data = $this->get_pending_data();
        
        if (empty($pending_data)) {
            $this->logger->info('No pending usage data to send');
            return;
        }
        
        $this->logger->info('Sending ' . count($pending_data) . ' usage records');
        $this->send_usage_data($pending_data);
    }
    
    /**
     * Get pending usage data
     * 
     * @param int $limit Limit number of records
     * @return array
     */
    private function get_pending_data($limit = 100) {
        global $wpdb;
        
        $sql = $wpdb->prepare(
            "SELECT * FROM {$this->table_name} 
             WHERE is_sent = 0 AND retry_count < 5 
             ORDER BY created_date ASC 
             LIMIT %d",
            $limit
        );
        
        return $wpdb->get_results($sql, ARRAY_A);
    }
    
    /**
     * Send usage data to wcplugin.com
     * 
     * @param array $data Usage data
     * @param bool $non_blocking Non-blocking mode
     */
    private function send_usage_data($data, $non_blocking = false) {
        if (empty($data)) {
            return;
        }
        
        // Group data by site for efficient sending
        $grouped_data = $this->group_data_by_site($data);
        
        foreach ($grouped_data as $site_url => $site_data) {
            $this->send_site_usage_data($site_data, $non_blocking);
        }
    }
    
    /**
     * Group data by site
     * 
     * @param array $data Usage data
     * @return array Grouped data
     */
    private function group_data_by_site($data) {
        $grouped = array();
        
        foreach ($data as $record) {
            $site_url = $record['site_url'];
            if (!isset($grouped[$site_url])) {
                $grouped[$site_url] = array(
                    'site_url' => $site_url,
                    'admin_email' => $record['admin_email'],
                    'records' => array()
                );
            }
            $grouped[$site_url]['records'][] = $record;
        }
        
        return $grouped;
    }
    
    /**
     * Send site usage data
     * 
     * @param array $site_data Site usage data
     * @param bool $non_blocking Non-blocking mode
     */
    private function send_site_usage_data($site_data, $non_blocking = false) {
        global $wpdb;
        
        // Check if endpoint is reachable (only for blocking requests to avoid slowdown)
        if (!$non_blocking && !$this->is_endpoint_reachable()) {
            $this->logger->warning('Usage data endpoint not reachable, skipping send for: ' . $site_data['site_url']);
            $this->increment_retry_count($site_data['records']);
            return;
        }
        
        // Prepare payload
        $payload = array(
            'site_url' => $site_data['site_url'],
            'admin_email' => $site_data['admin_email'],
            'plugin_version' => EDM_EFATURA_PREMIUM_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'timestamp' => current_time('c'),
            'invoices' => array()
        );
        
        // Add invoice data
        foreach ($site_data['records'] as $record) {
            $payload['invoices'][] = array(
                'uuid' => $record['invoice_uuid'],
                'type' => $record['invoice_type'],
                'created_date' => $record['created_date']
            );
        }
        
        // First test if endpoint is reachable with a simple GET request
        if (!$non_blocking) {
            $test_response = wp_remote_get($this->api_endpoint, array('timeout' => 5));
            $this->logger->info('Endpoint test response', array(
                'is_wp_error' => is_wp_error($test_response),
                'response_code' => is_wp_error($test_response) ? 'ERROR' : wp_remote_retrieve_response_code($test_response),
                'error_message' => is_wp_error($test_response) ? $test_response->get_error_message() : 'No error'
            ));
        }
        
        // Send data
        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'EDM-E-Fatura-Premium/' . EDM_EFATURA_PREMIUM_VERSION
            ),
            'body' => wp_json_encode($payload),
            'timeout' => $non_blocking ? 5 : 15,
            'blocking' => !$non_blocking,
            'sslverify' => false // SSL doğrulamasını geçici olarak kapat
        );
        
        $response = wp_remote_post($this->api_endpoint, $args);
        
        if (is_wp_error($response)) {
            $this->logger->error('Usage data send failed: ' . $response->get_error_message());
            $this->increment_retry_count($site_data['records']);
            return;
        }
        
        // For non-blocking requests, we can't check response code
        if (!$args['blocking']) {
            // Assume success for non-blocking requests since we can't wait for response
            $this->logger->info('Usage data sent (non-blocking) for ' . $site_data['site_url']);
            $this->mark_records_as_sent($site_data['records']);
            return;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Detailed debug logging only for blocking requests
        $debug_info = array(
            'endpoint' => $this->api_endpoint,
            'response_code' => $response_code,
            'response_headers' => wp_remote_retrieve_headers($response),
            'response_body_length' => strlen($response_body),
            'response_body_preview' => substr($response_body, 0, 200),
            'request_args' => array(
                'method' => $args['method'],
                'timeout' => $args['timeout'],
                'blocking' => $args['blocking'],
                'body_length' => strlen($args['body'])
            )
        );
        
        $this->logger->info('Usage data detailed response', $debug_info);
        
        if ($response_code === 200) {
            $this->logger->info('Usage data sent successfully for ' . $site_data['site_url']);
            $this->mark_records_as_sent($site_data['records']);
        } else {
            $this->logger->error('Usage data send failed with code ' . $response_code . ': ' . $response_body);
            $this->increment_retry_count($site_data['records']);
        }
    }
    
    /**
     * Mark records as sent
     * 
     * @param array $records Records to mark
     */
    private function mark_records_as_sent($records) {
        global $wpdb;
        
        $ids = array_column($records, 'id');
        $ids_placeholder = implode(',', array_fill(0, count($ids), '%d'));
        
        $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$this->table_name} 
                 SET is_sent = 1, sent_date = %s 
                 WHERE id IN ($ids_placeholder)",
                array_merge([current_time('mysql')], $ids)
            )
        );
    }
    
    /**
     * Increment retry count
     * 
     * @param array $records Records to increment
     */
    private function increment_retry_count($records) {
        global $wpdb;
        
        $ids = array_column($records, 'id');
        $ids_placeholder = implode(',', array_fill(0, count($ids), '%d'));
        
        $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$this->table_name} 
                 SET retry_count = retry_count + 1 
                 WHERE id IN ($ids_placeholder)",
                $ids
            )
        );
    }
    
    /**
     * Get usage statistics
     * 
     * @param int $days Number of days to get stats for
     * @return array Usage statistics
     */
    public function get_usage_stats($days = 30) {
        global $wpdb;
        
        $sql = $wpdb->prepare(
            "SELECT 
                COUNT(*) as total_invoices,
                COUNT(CASE WHEN invoice_type = 'e-invoice' THEN 1 END) as e_invoices,
                COUNT(CASE WHEN invoice_type = 'e-archive' THEN 1 END) as e_archives,
                COUNT(CASE WHEN is_sent = 1 THEN 1 END) as sent_records,
                COUNT(CASE WHEN is_sent = 0 THEN 1 END) as pending_records
             FROM {$this->table_name} 
             WHERE created_date >= DATE_SUB(NOW(), INTERVAL %d DAY)",
            $days
        );
        
        return $wpdb->get_row($sql, ARRAY_A);
    }
    
    /**
     * Clean up old sent records (older than 90 days)
     */
    public function cleanup_old_records() {
        global $wpdb;
        
        $wpdb->query(
            "DELETE FROM {$this->table_name} 
             WHERE is_sent = 1 AND sent_date < DATE_SUB(NOW(), INTERVAL 90 DAY)"
        );
        
        $this->logger->info('Cleaned up old usage tracking records');
    }
    
    /**
     * Check if this is first run after plugin activation
     */
    public function check_first_run() {
        // Only run once per page load and only in admin or front-end
        if (get_transient('edm_usage_first_run_checked')) {
            return;
        }
        
        // Set transient to prevent multiple checks
        set_transient('edm_usage_first_run_checked', true, HOUR_IN_SECONDS);
        
        // Check if we've already sent initial data
        $initial_data_sent = get_option('edm_efatura_initial_data_sent', false);
        
        if (!$initial_data_sent) {
            // Send initial site data
            $this->send_initial_site_data();
            
            // Mark as sent
            update_option('edm_efatura_initial_data_sent', true);
        }
    }
    
    /**
     * Send initial site data on plugin activation
     */
    private function send_initial_site_data() {
        try {
            // Get site information
            $site_url = get_site_url();
            $admin_email = get_option('admin_email');
            
            // Create initial tracking record with special UUID
            $initial_uuid = 'INITIAL-' . wp_generate_uuid4();
            
            global $wpdb;
            $result = $wpdb->insert(
                $this->table_name,
                array(
                    'site_url' => $site_url,
                    'admin_email' => $admin_email,
                    'invoice_uuid' => $initial_uuid,
                    'invoice_type' => 'initial-activation',
                    'created_date' => current_time('mysql')
                ),
                array('%s', '%s', '%s', '%s', '%s')
            );
            
            if ($result === false) {
                $this->logger->error('Initial usage tracking insert failed: ' . $wpdb->last_error);
                return;
            }
            
            $this->logger->info('Initial site data recorded for tracking');
            
            // Send immediately
            $this->send_initial_activation_data();
            
        } catch (Exception $e) {
            $this->logger->error('Initial site data tracking error: ' . $e->getMessage());
        }
    }
    
    /**
     * Send initial activation data immediately
     */
    private function send_initial_activation_data() {
        // Get site information
        $site_url = get_site_url();
        $admin_email = get_option('admin_email');
        
        // Prepare payload for initial activation
        $payload = array(
            'site_url' => $site_url,
            'admin_email' => $admin_email,
            'plugin_version' => EDM_EFATURA_PREMIUM_VERSION,
            'wordpress_version' => get_bloginfo('version'),
            'php_version' => PHP_VERSION,
            'timestamp' => current_time('c'),
            'activation_type' => 'initial',
            'invoices' => array(
                array(
                    'uuid' => 'INITIAL-ACTIVATION',
                    'type' => 'activation',
                    'created_date' => current_time('mysql')
                )
            )
        );
        
        // Send data with longer timeout for activation
        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'EDM-E-Fatura-Premium/' . EDM_EFATURA_PREMIUM_VERSION . ' (Initial-Activation)'
            ),
            'body' => wp_json_encode($payload),
            'timeout' => 10, // Longer timeout for initial activation
            'blocking' => false // Non-blocking to not slow down activation
        );
        
        $response = wp_remote_post($this->api_endpoint, $args);
        
        // Log result
        if (is_wp_error($response)) {
            $this->logger->info('Initial activation data send scheduled for retry: ' . $response->get_error_message());
        } else {
            $this->logger->info('Initial activation data sent to wcplugin.com');
        }
    }
    
    /**
     * Clean duplicate records from tracking table
     */
    public function clean_duplicate_records() {
        global $wpdb;
        
        try {
            // Find and remove duplicates, keeping only the earliest record for each UUID
            $deleted = $wpdb->query("
                DELETE t1 FROM {$this->table_name} t1
                INNER JOIN {$this->table_name} t2 
                WHERE t1.invoice_uuid = t2.invoice_uuid 
                AND t1.id > t2.id
            ");
            
            if ($deleted > 0) {
                $this->logger->info("Cleaned {$deleted} duplicate usage tracking records");
            }
            
            return $deleted;
            
        } catch (Exception $e) {
            $this->logger->error('Error cleaning duplicate records: ' . $e->getMessage());
            return false;
        }
    }
}