<?php
/**
 * Class WC_Email_EDM_Invoice file
 *
 * @package EDM_Efatura_Premium
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('WC_Email')) {
    return;
}

/**
 * E-Invoice Email
 *
 * Email sent to customers when an e-invoice is generated.
 *
 * @class    WC_Email_EDM_Invoice
 * @extends  WC_Email
 */
class WC_Email_EDM_Invoice extends WC_Email {
    /**
     * Invoice data
     *
     * @var array
     */
    public $invoice;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->id = 'edm_invoice';
        $this->customer_email = true;
        $this->title = __('E-Invoice Generated', 'edm-efatura-premium');
        $this->description = __('This email is sent to customers when an e-invoice is generated for their order.', 'edm-efatura-premium');
        $this->template_html = 'emails/customer-edm-invoice.php';
        $this->template_plain = 'emails/plain/customer-edm-invoice.php';
        $this->placeholders = array(
            '{order_date}' => '',
            '{order_number}' => '',
            '{invoice_id}' => '',
            '{invoice_date}' => '',
        );
        
        // Call parent constructor
        parent::__construct();
        
        // Add custom template path
        $this->template_base = EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'templates/';
    }
    
    /**
     * Get email subject
     *
     * @return string
     */
    public function get_default_subject() {
        return __('Your e-invoice for order {order_number} has been generated', 'edm-efatura-premium');
    }
    
    /**
     * Get email heading
     *
     * @return string
     */
    public function get_default_heading() {
        return __('Your e-invoice is ready', 'edm-efatura-premium');
    }
    
    /**
     * Trigger the sending of this email
     *
     * @param int $order_id Order ID
     * @param array $invoice Invoice data
     */
    public function trigger($order_id, $invoice = array()) {
        // Check if order exists
        $this->object = wc_get_order($order_id);
        
        if (!$this->object || !$this->object->get_billing_email()) {
            return;
        }
        
        $this->invoice = $invoice;
        
        // Set placeholders
        $this->placeholders['{order_date}'] = wc_format_datetime($this->object->get_date_created());
        $this->placeholders['{order_number}'] = $this->object->get_order_number();
        $this->placeholders['{invoice_id}'] = isset($invoice['invoice_id']) ? $invoice['invoice_id'] : '';
        $this->placeholders['{invoice_date}'] = isset($invoice['created_at']) ? date_i18n(get_option('date_format'), strtotime($invoice['created_at'])) : '';
        
        // Send email
        if ($this->is_enabled() && $this->get_recipient()) {
            $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());
        }
    }
    
    /**
     * Get content html
     *
     * @return string
     */
    public function get_content_html() {
        return wc_get_template_html(
            $this->template_html,
            array(
                'order' => $this->object,
                'invoice' => $this->invoice,
                'email_heading' => $this->get_heading(),
                'additional_content' => $this->get_additional_content(),
                'sent_to_admin' => false,
                'plain_text' => false,
                'email' => $this,
            ),
            '',
            $this->template_base
        );
    }
    
    /**
     * Get content plain
     *
     * @return string
     */
    public function get_content_plain() {
        return wc_get_template_html(
            $this->template_plain,
            array(
                'order' => $this->object,
                'invoice' => $this->invoice,
                'email_heading' => $this->get_heading(),
                'additional_content' => $this->get_additional_content(),
                'sent_to_admin' => false,
                'plain_text' => true,
                'email' => $this,
            ),
            '',
            $this->template_base
        );
    }
    
    /**
     * Get email attachments
     *
     * @return array
     */
    public function get_attachments() {
        $attachments = array();
        
        // Check if PDF exists
        if (!empty($this->invoice['pdf_path']) && file_exists($this->invoice['pdf_path'])) {
            $attachments[] = $this->invoice['pdf_path'];
        }
        
        return apply_filters('woocommerce_email_attachments', $attachments, $this->id, $this->object, $this);
    }
}