<?php
/**
 * EDM E-Fatura Premium WooCommerce Email Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling WooCommerce email operations
 */
class EDM_Efatura_Premium_Email {
    /**
     * Logger instance
     *
     * @var EDM_Efatura_Premium_Logger
     */
    private $logger;
    
    /**
     * DB instance
     *
     * @var EDM_Efatura_Premium_DB
     */
    private $db;
    
    /**
     * Settings
     *
     * @var array
     */
    private $settings;
    
    /**
     * Constructor
     */
    /**
     * Flag to prevent duplicate hook registrations (global across all instances)
     *
     * @var bool
     */
    private static $hooks_registered = false;
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = new EDM_Efatura_Premium_Logger();
        $this->db = new EDM_Efatura_Premium_DB();
        $this->settings = get_option('edm_efatura_premium_settings', array());
        
        // Register hooks - but only if not already done (use WordPress option for global state)
        $global_hooks_registered = get_option('edm_email_hooks_registered', false);
        
        if (!$global_hooks_registered && !self::$hooks_registered) {
            add_filter('woocommerce_email_classes', array($this, 'add_email_class'));
            
            // Sadece belirli durumlarda ek ekleyelim, tüm e-postalara değil
            add_filter('woocommerce_email_attachments', array($this, 'attach_invoice_to_specific_email'), 10, 3);
            
            // Fatura bilgisini e-postada göster
            add_action('woocommerce_email_before_order_table', array($this, 'add_invoice_info_to_email'), 10, 4);
            
            // Tamamlandı e-postalarının çift gönderilmemesi için
            add_filter('woocommerce_email_enabled_customer_completed_order', array($this, 'check_duplicate_completed_email'), 10, 3);
            
            // Mark as registered in both static and global state
            self::$hooks_registered = true;
            update_option('edm_email_hooks_registered', true);
        }
    }
    
	/**
     * Prevent duplicate completed order emails
     *
     * @param bool $enabled Whether this email is enabled
     * @param array $order Order data
     * @param WC_Email $email Email object
     * @return bool Modified enabled status
     */
	public function check_duplicate_completed_email($enabled, $order, $email) {
		// Eğer e-posta zaten devre dışıysa, o şekilde bırak
		if (!$enabled) {
			return $enabled;
		}
		
		$order_id = $order->get_id();
		
		// HPOS uyumluluğu kontrolü
		$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
		
		// Bu e-posta daha önce gönderildi mi?
		$email_sent = $hpos_enabled ? 
			$order->get_meta('_edm_completed_email_sent', true) : 
			get_post_meta($order_id, '_edm_completed_email_sent', true);
		
		// Eğer e-posta daha önce gönderildiyse, ikinci gönderimi engelle
		if ($email_sent === 'yes') {
			return false;
		}
		
		// İlk defa gönderiliyorsa, gönderildi olarak işaretle
		if ($hpos_enabled) {
			$order->update_meta_data('_edm_completed_email_sent', 'yes');
			$order->save();
		} else {
			update_post_meta($order_id, '_edm_completed_email_sent', 'yes');
		}
		
		// E-postanın gönderilmesine izin ver
		return true;
	}
	
	
    /**
     * Add custom email class
     *
     * @param array $email_classes Email classes
     * @return array Modified email classes
     */
    public function add_email_class($email_classes) {
        static $email_registered = false;
        
        // Only register the email class once
        if ($email_registered || isset($email_classes['edm_invoice'])) {
            return $email_classes;
        }
        
        // Make sure the email template file exists
        $template_path = EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'includes/emails/class-wc-email-edm-invoice.php';
        
        if (file_exists($template_path)) {
            // Include custom email class
            require_once $template_path;
            
            // Add custom email class if it exists
            if (class_exists('WC_Email_EDM_Invoice')) {
                $email_classes['edm_invoice'] = new WC_Email_EDM_Invoice();
                
                
                $email_registered = true;
            } else {
                // Log email class loading error
                $this->logger->error('Failed to load WC_Email_EDM_Invoice class');
            }
        } else {
            // Log missing email template file
            $this->logger->error('Email template file not found', array('path' => $template_path));
        }
        
        return $email_classes;
    }
    
    
    
    
    
    /**
     * Attach invoice PDF only to specific emails
     *
     * @param array $attachments Existing attachments
     * @param string $email_id Email ID
     * @param WC_Order $order Order object
     * @return array Modified attachments
     */
    public function attach_invoice_to_specific_email($attachments, $email_id, $order) {
    // Sadece tamamlanmış sipariş e-postalarına filtre uygula
    if ($email_id !== 'customer_completed_order' && $email_id !== 'edm_invoice') {
        return $attachments;
    }
    
    // Eğer sipariş objesi geçerli değilse, dön
    if (!$order || !($order instanceof WC_Order)) {
        return $attachments;
    }
    
    $order_id = $order->get_id();
    
    // PDF ekleme ayarları açık mı?
    if (!isset($this->settings['attach_pdf_to_email']) || $this->settings['attach_pdf_to_email'] !== 'yes') {
        return $attachments;
    }
    
    // Gecikme süresi var mı?
    $delay_days = isset($this->settings['invoice_delay_days']) ? intval($this->settings['invoice_delay_days']) : 0;
    
    // Eğer gecikme süresi varsa ve standart tamamlandı e-postasıysa, PDF ekleme
    if ($delay_days > 0 && $email_id === 'customer_completed_order') {
        return $attachments;
    }
    
    // Faturayı al
    $invoice = $this->db->get_invoice_by_order_id($order_id);
    
    // Fatura yoksa, çık
    if (!$invoice) {
        return $attachments;
    }
    
    // PDF varlığını kontrol et
    if (empty($invoice['pdf_path']) || !file_exists($invoice['pdf_path'])) {
        $this->logger->warning('E-postaya eklenecek PDF bulunamadı', [
            'order_id' => $order_id,
            'invoice_id' => $invoice['id']
        ]);
        return $attachments;
    }
    
    // PDF'yi eklere ekle
    $attachments[] = $invoice['pdf_path'];
    
    $this->logger->info('PDF fatura e-postaya eklendi', [
        'order_id' => $order_id,
        'email_id' => $email_id
    ]);
    
    return $attachments;
}
    
    /**
     * Add invoice information to email
     *
     * @param WC_Order $order Order object
     * @param bool $sent_to_admin Sent to admin
     * @param bool $plain_text Plain text
     * @param WC_Email $email Email object
     */
    public function add_invoice_info_to_email($order, $sent_to_admin, $plain_text, $email) {
		$order_id = $order->get_id();
		
		// Get settings
		$settings = get_option('edm_efatura_premium_settings', array());
		$delay_days = isset($settings['invoice_delay_days']) ? intval($settings['invoice_delay_days']) : 0;
		
		// Get invoice
		$invoice = $this->db->get_invoice_by_order_id($order_id);
		
		// Check if invoice generation is delayed
		if (!$invoice && $delay_days > 0) {
			// Check if there's a scheduled invoice generation
			$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
				\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
			
			$scheduled_time = false;
			if ($hpos_enabled) {
				$scheduled_time = $order->get_meta('edm_invoice_scheduled', true);
			} else {
				$scheduled_time = get_post_meta($order_id, 'edm_invoice_scheduled', true);
			}
			
			if ($scheduled_time) {
				if ($plain_text) {
					echo "\n\n" . __('Your invoice will be generated and sent to you within ', 'edm-efatura-premium') . $delay_days . __(' day(s).', 'edm-efatura-premium') . "\n\n";
				} else {
					echo '<div style="margin-top: 20px; margin-bottom: 20px; padding: 15px; background-color: #f8f8f8; border-left: 4px solid #0073aa;">';
					echo '<p>' . __('Your invoice will be generated and sent to you within ', 'edm-efatura-premium') . $delay_days . __(' day(s).', 'edm-efatura-premium') . '</p>';
					echo '</div>';
				}
			}
			
			return;
		}
		
		// If no invoice found, exit
		if (!$invoice) {
			return;
		}
		
		// Check if this is the email to show invoice information in
		$allowed_emails = array('customer_completed_order', 'customer_invoice', 'edm_invoice');
		if (!in_array($email->id, $allowed_emails)) {
			return;
		}
		
		// Display invoice information
		if ($plain_text) {
			echo "\n\n";
			echo __('Invoice Number', 'edm-efatura-premium') . ': ' . $invoice['invoice_id'] . "\n";
			echo __('Invoice Date', 'edm-efatura-premium') . ': ' . date_i18n(get_option('date_format'), strtotime($invoice['created_at'])) . "\n";
			echo __('Invoice Amount', 'edm-efatura-premium') . ': ' . wc_price($invoice['amount']) . "\n\n";
		} else {
			echo '<div style="margin-top: 20px; margin-bottom: 20px; padding: 15px; background-color: #f8f8f8; border-left: 4px solid #0073aa;">';
			echo '<h2 style="margin-top: 0;">' . __('E-Invoice Information', 'edm-efatura-premium') . '</h2>';
			echo '<p><strong>' . __('Invoice Number', 'edm-efatura-premium') . ':</strong> ' . esc_html($invoice['invoice_id']) . '</p>';
			echo '<p><strong>' . __('Invoice Date', 'edm-efatura-premium') . ':</strong> ' . esc_html(date_i18n(get_option('date_format'), strtotime($invoice['created_at']))) . '</p>';
			echo '<p><strong>' . __('Invoice Amount', 'edm-efatura-premium') . ':</strong> ' . wc_price($invoice['amount']) . '</p>';
			echo '</div>';
		}
	}
}