<?php
/**
 * EDM E-Fatura Premium WooCommerce Order Handler Class
 *
 * @package EDM_Efatura_Premium
 *
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for handling WooCommerce order operations
 */
class EDM_Efatura_Premium_Order_Handler {
    /**
     * API instance
     *
     * @var EDM_Efatura_Premium_API
     */
    private $api;
    
    /**
     * Logger instance
     *
     * @var EDM_Efatura_Premium_Logger
     */
    private $logger;
    
    /**
     * DB instance
     *
     * @var EDM_Efatura_Premium_DB
     */
    private $db;
    
    /**
     * Settings
     *
     * @var array
     */
    private $settings;
    
    /**
     * Constructor
     *
     * @param EDM_Efatura_Premium_API $api API instance
     */
    public function __construct($api) {
		$this->api = $api;
		$this->logger = new EDM_Efatura_Premium_Logger();
		$this->db = new EDM_Efatura_Premium_DB();
		$this->settings = get_option('edm_efatura_premium_settings', array());
		
		// Register hooks
		add_action('woocommerce_order_status_changed', array($this, 'handle_order_status_change'), 10, 4);
		add_action('woocommerce_admin_order_data_after_billing_address', array($this, 'display_invoice_info_in_order'), 10, 1);
		add_action('woocommerce_admin_order_actions_end', array($this, 'add_generate_invoice_action'), 10, 1);
		add_action('wp_ajax_edm_generate_invoice_for_order', array($this, 'ajax_generate_invoice_for_order'));

		// add_action('wp_ajax_edm_send_invoice_email', array($this, 'ajax_send_invoice_email')); // Removed: Duplicate with admin class

		// Email handler moved to main plugin only to prevent duplicates

		// Frontend customer account hooks
		add_action('woocommerce_order_details_after_order_table', array($this, 'display_invoice_download_button'), 10, 1);
		add_action('woocommerce_my_account_my_orders_actions', array($this, 'add_invoice_download_action'), 10, 2);
		add_action('wp_ajax_edm_download_invoice_pdf', array($this, 'ajax_download_invoice_pdf'));
		add_action('wp_ajax_nopriv_edm_download_invoice_pdf', array($this, 'ajax_download_invoice_pdf'));
		
	}
	
	/**
	 * Process scheduled invoice
	 * 
	 * @param int $order_id Order ID
	 */
	public function process_scheduled_invoice($order_id) {
		// Get order
		$order = wc_get_order($order_id);
		if (!$order) {
			$this->logger->error('Order not found for scheduled invoice', array('order_id' => $order_id));
			// Remove from scheduled invoices
			$this->remove_from_scheduled_invoices($order_id);
			return false;
		}
		
		// Check if order total is zero or the order has no items with value
		if ($order->get_total() <= 0) {
			$this->logger->info('Skipping scheduled invoice generation for zero-value order', array(
				'order_id' => $order_id,
				'total' => $order->get_total()
			));
			
			// Add private order note
			$order->add_order_note(
				__('EDM E-invoice generation skipped. Order has zero value.', 'edm-efatura-premium'),
				0 // Private note
			);
			
			// Remove from scheduled invoices
			$this->remove_from_scheduled_invoices($order_id);
			return false;
		}
		
		// Check if invoice was already sent
		$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
		
		if ($hpos_enabled) {
			$invoice_sent = $order->get_meta('edm_invoice_email_sent', true);
		} else {
			$invoice_sent = get_post_meta($order_id, 'edm_invoice_email_sent', true);
		}
		
		if ($invoice_sent === 'yes') {
			$this->logger->info('Invoice already sent for order (GMT+3)', array('order_id' => $order_id));
			// Remove from scheduled invoices
			$this->remove_from_scheduled_invoices($order_id);
			return true; // Invoice already sent, no need to process again
		}
		
		// Check if this order is actually scheduled for invoice generation
		$scheduled_time = false;
		
		if ($hpos_enabled) {
			$scheduled_time = $order->get_meta('edm_invoice_scheduled', true);
		} else {
			$scheduled_time = get_post_meta($order_id, 'edm_invoice_scheduled', true);
		}
		
		// If this order is not scheduled for invoice generation, log and exit
		if (!$scheduled_time) {
			$this->logger->error('Order not scheduled for invoice generation (GMT+3)', array('order_id' => $order_id));
			// Remove from scheduled invoices
			$this->remove_from_scheduled_invoices($order_id);
			return false;
		}
		
		// GMT+3 için zamanı ayarla
		$current_time = time() + (3 * 3600); // 3 saat ekliyoruz
		
		$this->logger->info('Processing scheduled invoice for order (GMT+3)', array(
			'order_id' => $order_id,
			'scheduled_time' => date('Y-m-d H:i:s', $scheduled_time),
			'current_time_gmt3' => date('Y-m-d H:i:s', $current_time),
			'is_due' => ($scheduled_time <= $current_time) ? 'YES' : 'NO'
		));
		
		// Check if invoice already exists for this order
		$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
		if (!$existing_invoice) {
			// Generate invoice
			$this->logger->info('Generating invoice for order (GMT+3)', array('order_id' => $order_id));
			$invoice = $this->generate_invoice_for_order($order);
		} else {
			$this->logger->info('Using existing invoice for order (GMT+3)', array(
				'order_id' => $order_id,
				'invoice_id' => isset($existing_invoice['id']) ? $existing_invoice['id'] : 'unknown'
			));
			$invoice = $existing_invoice;
		}
		
		if ($invoice) {
			// Make sure PDF is generated
			if (empty($invoice['pdf_path']) || !file_exists($invoice['pdf_path'])) {
				$this->logger->info('Generating PDF for invoice (GMT+3)', array(
					'order_id' => $order_id,
					'invoice_id' => isset($invoice['id']) ? $invoice['id'] : 'unknown'
				));
				$this->generate_invoice_pdf($invoice);
			}
			
			// Send invoice email
			$this->logger->info('Sending invoice email (GMT+3)', array(
				'order_id' => $order_id,
				'invoice_id' => isset($invoice['id']) ? $invoice['id'] : 'unknown',
				'current_time_gmt3' => date('Y-m-d H:i:s', $current_time)
			));
			
			$email_sent = $this->send_direct_invoice_email($order, $invoice);
			
			if ($email_sent) {
				// Mark as sent
				if ($hpos_enabled) {
					$order->update_meta_data('edm_invoice_email_sent', 'yes');
					$order->update_meta_data('edm_invoice_email_sent_date', current_time('mysql'));
					$order->delete_meta_data('edm_invoice_scheduled'); // Clear the scheduled flag - burası önemli
					$order->delete_meta_data('edm_invoice_schedule_key'); // Schedule key'i temizle
					$order->save();
				} else {
					update_post_meta($order_id, 'edm_invoice_email_sent', 'yes');
					update_post_meta($order_id, 'edm_invoice_email_sent_date', current_time('mysql'));
					delete_post_meta($order_id, 'edm_invoice_scheduled'); // Clear the scheduled flag - burası önemli
					delete_post_meta($order_id, 'edm_invoice_schedule_key'); // Schedule key'i temizle
				}
				
				// Add order note only once (private - not sent to customer)
				$order->add_order_note(
					sprintf(
						__('EDM E-invoice generated successfully and sent to customer (GMT+3). Invoice ID: %s', 'edm-efatura-premium'),
						isset($invoice['id']) ? $invoice['id'] : 'N/A'
					),
					0 // Private note - not sent to customer
				);
				
				$this->logger->info('Invoice successfully processed and sent (GMT+3)', array(
					'order_id' => $order_id,
					'invoice_id' => isset($invoice['id']) ? $invoice['id'] : 'unknown',
					'current_time_gmt3' => date('Y-m-d H:i:s', $current_time)
				));
				
				// Remove from scheduled invoices
				$this->remove_from_scheduled_invoices($order_id);
				
				return true;
			} else {
				$this->logger->error('Failed to send invoice email (GMT+3)', array(
					'order_id' => $order_id,
					'invoice_id' => isset($invoice['id']) ? $invoice['id'] : 'unknown',
					'current_time_gmt3' => date('Y-m-d H:i:s', $current_time)
				));
			}
		} else {
			$this->logger->error('Failed to generate invoice for order (GMT+3)', array(
				'order_id' => $order_id,
				'current_time_gmt3' => date('Y-m-d H:i:s', $current_time)
			));
			
			// Add private order note
			$order->add_order_note(
				__('EDM E-invoice generation failed (GMT+3). Please check logs for details.', 'edm-efatura-premium'),
				0 // Private note
			);
			
			// Remove from scheduled invoices after multiple attempts
			$attempts = $this->increment_failed_attempts($order_id);
			if ($attempts >= 3) {
				$this->logger->error('Removing order from scheduled invoices after 3 failed attempts (GMT+3)', array(
					'order_id' => $order_id,
					'attempts' => $attempts
				));
				$this->remove_from_scheduled_invoices($order_id);
			}
			
			return false;
		}
		
		// If we get here, something went wrong with sending the email
		// Add private order note
		$order->add_order_note(
			__('EDM E-invoice was generated but email sending failed (GMT+3). Please check logs for details.', 'edm-efatura-premium'),
			0 // Private note
		);
		
		// Increment failed attempts and possibly remove from schedule after multiple failures
		$attempts = $this->increment_failed_attempts($order_id);
		if ($attempts >= 3) {
			$this->logger->error('Removing order from scheduled invoices after 3 failed attempts (GMT+3)', array(
				'order_id' => $order_id,
				'attempts' => $attempts
			));
			$this->remove_from_scheduled_invoices($order_id);
		}
		
		return false;
	}
	
	/**
	 * Remove order from scheduled invoices list
	 * 
	 * @param int $order_id Order ID
	 */
	private function remove_from_scheduled_invoices($order_id) {
		$scheduled_invoices = get_option('edm_scheduled_invoices', array());
		if (isset($scheduled_invoices[$order_id])) {
			unset($scheduled_invoices[$order_id]);
			update_option('edm_scheduled_invoices', $scheduled_invoices);
			$this->logger->info('Removed order from scheduled invoices', array('order_id' => $order_id));
		}
	}

	/**
	 * Increment failed attempts for an order and return current count
	 * 
	 * @param int $order_id Order ID
	 * @return int Current number of attempts
	 */
	private function increment_failed_attempts($order_id) {
		$failed_attempts = get_option('edm_invoice_failed_attempts', array());
		
		if (!isset($failed_attempts[$order_id])) {
			$failed_attempts[$order_id] = 1;
		} else {
			$failed_attempts[$order_id]++;
		}
		
		update_option('edm_invoice_failed_attempts', $failed_attempts);
		
		$this->logger->info('Incremented failed attempts for order', array(
			'order_id' => $order_id,
			'attempts' => $failed_attempts[$order_id]
		));
		
		return $failed_attempts[$order_id];
	}
	
	/**
	 * Public wrapper for sending invoice email directly
	 *
	 * @param WC_Order $order Order object
	 * @param array $invoice Invoice data
	 * @return bool Success status
	 */
	public function send_invoice_email_directly($order, $invoice) {
		return $this->send_direct_invoice_email($order, $invoice);
	}
	
	/**
	 * Send direct invoice email with attached PDF
	 *
	 * @param WC_Order $order Order object
	 * @param array $invoice Invoice data
	 * @return bool Success status
	 */
	public function send_direct_invoice_email($order, $invoice) {
    // Check if email was already sent
    $order_id = $order->get_id();
    $hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
    
    // Use a specific transient key to prevent duplicate sends during the same request
    $email_lock_key = 'edm_email_lock_' . $order_id;
    $email_lock = get_transient($email_lock_key);
    
    if ($email_lock) {
        return true; // Email is already being sent by another process
    }
    
    // Set a short lock to prevent duplicate sends
    set_transient($email_lock_key, true, 30); // 30 seconds lock
    
    $email_sent = false;
    if ($hpos_enabled) {
        $email_sent = $order->get_meta('edm_invoice_email_sent', true);
    } else {
        $email_sent = get_post_meta($order_id, 'edm_invoice_email_sent', true);
    }
    
    if ($email_sent === 'yes') {
        delete_transient($email_lock_key);
        return true; // Email already sent, don't send again
    }
    
    try {
        // Verify invoice data
        if (empty($invoice) || empty($invoice['id'])) {
            delete_transient($email_lock_key);
            return false;
        }
        
        // Check if PDF exists
        $pdf_exists = !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path']);
        if (!$pdf_exists) {
            // Try to generate PDF
            $pdf_path = $this->generate_invoice_pdf($invoice);

            // After generating PDF, reload invoice from database to get updated pdf_path
            if ($pdf_path) {
                $db = new EDM_Efatura_Premium_DB();
                $updated_invoice = $db->get_invoice_by_order_id($order_id);
                if ($updated_invoice) {
                    $invoice = $updated_invoice;
                    $pdf_exists = !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path']);
                }
            }
        }
        
        // Prepare email content
        $subject = sprintf(
            __('Your e-invoice for order #%s is ready', 'edm-efatura-premium'),
            $order->get_order_number()
        );
        
        // Build HTML email
        $content = '<p>' . sprintf(__('Dear %s,', 'edm-efatura-premium'), $order->get_billing_first_name()) . '</p>';
        $content .= '<p>' . sprintf(__('Your e-invoice for order #%s has been generated.', 'edm-efatura-premium'), $order->get_order_number()) . '</p>';
        
        if (!empty($invoice['invoice_id'])) {
            $content .= '<h2>' . __('Invoice Details', 'edm-efatura-premium') . '</h2>';
            $content .= '<ul>';
            $content .= '<li><strong>' . __('Invoice ID:', 'edm-efatura-premium') . '</strong> ' . $invoice['invoice_id'] . '</li>';
            if (!empty($invoice['created_at'])) {
                $content .= '<li><strong>' . __('Date:', 'edm-efatura-premium') . '</strong> ' . date_i18n(get_option('date_format'), strtotime($invoice['created_at'])) . '</li>';
            }
            if (!empty($invoice['status'])) {
                $content .= '<li><strong>' . __('Status:', 'edm-efatura-premium') . '</strong> ' . $invoice['status'] . '</li>';
            }
            $content .= '</ul>';
        }
        
        if ($pdf_exists) {
            $content .= '<p>' . __('Your invoice is attached to this email.', 'edm-efatura-premium') . '</p>';
        } else {
            $content .= '<p>' . __('You can download your invoice from your account.', 'edm-efatura-premium') . '</p>';
        }
        
        $content .= '<p>' . __('Thank you for your business!', 'edm-efatura-premium') . '</p>';
        
        // Use WooCommerce mailer to send the email
        $mailer = WC()->mailer();
        if ($mailer) {
            // Format headers
            $headers = array(
                'Content-Type: text/html; charset=UTF-8',
                'From: ' . get_option('blogname') . ' <' . get_option('admin_email') . '>'
            );
            
            // Prepare attachments
            $attachments = array();
            if ($pdf_exists) {
                $attachments[] = $invoice['pdf_path'];
            }
            
            // Format message with WooCommerce template
            $formatted_content = $mailer->wrap_message(
                sprintf(__('Invoice for Order #%s', 'edm-efatura-premium'), $order->get_order_number()),
                $content
            );
            
            // Send the email
            $sent = $mailer->send($order->get_billing_email(), $subject, $formatted_content, $headers, $attachments);
            
            if ($sent) {
                // Mark as sent
                if ($hpos_enabled) {
                    $order->update_meta_data('edm_invoice_email_sent', 'yes');
                    $order->update_meta_data('edm_invoice_email_sent_date', current_time('mysql'));
                    $order->save();
                } else {
                    update_post_meta($order_id, 'edm_invoice_email_sent', 'yes');
                    update_post_meta($order_id, 'edm_invoice_email_sent_date', current_time('mysql'));
                }
                
                // E-posta gönderildi olarak işaretleyelim ama not eklemeyelim
        $this->logger->info('Invoice email sent successfully', [
            'order_id' => $order->get_id(),
            'email' => $order->get_billing_email()
        ]);
        
        delete_transient($email_lock_key);
			return true;
		} else {
			delete_transient($email_lock_key);
			return false;
		}
	}
        
        // Fallback to wp_mail if WooCommerce mailer fails
        // Set headers
        $headers = array(
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_option('blogname') . ' <' . get_option('admin_email') . '>'
        );
        
        // Set attachments
        $attachments = array();
        if ($pdf_exists) {
            $attachments[] = $invoice['pdf_path'];
        }
        
        // Send email
        $sent = wp_mail(
            $order->get_billing_email(),
            $subject,
            $content,
            $headers,
            $attachments
        );
        
        // Log result
        if ($sent) {
            // Mark as sent
            if ($hpos_enabled) {
                $order->update_meta_data('edm_invoice_email_sent', 'yes');
                $order->update_meta_data('edm_invoice_email_sent_date', current_time('mysql'));
                $order->save();
            } else {
                update_post_meta($order_id, 'edm_invoice_email_sent', 'yes');
                update_post_meta($order_id, 'edm_invoice_email_sent_date', current_time('mysql'));
            }
            
            // Add private order note
            $order->add_order_note(
                sprintf(
                    __('EDM E-invoice email sent to customer (%s).', 'edm-efatura-premium'),
                    $order->get_billing_email()
                ),
                0 // Private note
            );
            
            // Add admin notification note (visible in order notes, no email)
            $order->add_order_note(
                sprintf(
                    __('📧 Invoice PDF has been sent to customer via email (%s). Invoice ID: %s', 'edm-efatura-premium'),
                    $order->get_billing_email(),
                    !empty($invoice['invoice_id']) ? $invoice['invoice_id'] : 'N/A'
                ),
                0 // Private note - visible in order notes but no email
            );
            
            delete_transient($email_lock_key);
            return true;
        } else {
            delete_transient($email_lock_key);
            return false;
        }
    } catch (Exception $e) {
        delete_transient($email_lock_key);
        return false;
    }
}


	
    
	
	

	/**
	 * Cancel any scheduled invoice for an order
	 *
	 * @param int $order_id Order ID
	 */
	private function cancel_scheduled_invoice($order_id) {
		// HPOS compatibility check
		$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
		
		// Remove from scheduled cron events
		wp_clear_scheduled_hook('edm_efatura_generate_invoice', array($order_id));
		
		// Remove from order meta
		$order = wc_get_order($order_id);
		if ($order) {
			if ($hpos_enabled) {
				$order->delete_meta_data('edm_invoice_scheduled');
				$order->delete_meta_data('edm_invoice_schedule_key');
				$order->save();
			} else {
				delete_post_meta($order_id, 'edm_invoice_scheduled');
				delete_post_meta($order_id, 'edm_invoice_schedule_key');
			}
		}
		
		// Remove from scheduled invoices list
		$scheduled_invoices = get_option('edm_scheduled_invoices', array());
		if (isset($scheduled_invoices[$order_id])) {
			unset($scheduled_invoices[$order_id]);
			update_option('edm_scheduled_invoices', $scheduled_invoices);
			
			$this->logger->info('Cancelled scheduled invoice for order', array(
				'order_id' => $order_id,
				'reason' => 'Order status changed'
			));
		}
	}
    /**
     * Handle order status change
     *
     * @param int $order_id Order ID
     * @param string $status_from Old status
     * @param string $status_to New status
     * @param WC_Order $order Order object
     */
    public function handle_order_status_change($order_id, $status_from, $status_to, $order) {
		// Check if order total is zero or the order has no items with value
		if ($order->get_total() <= 0) {
			$this->logger->info('Skipping invoice generation for zero-value order', array(
				'order_id' => $order_id,
				'total' => $order->get_total()
			));
			
			// Add private order note
			$order->add_order_note(
				__('EDM E-invoice generation skipped. Order has zero value.', 'edm-efatura-premium'),
				0 // Private note
			);
			
			return;
		}
		
		// If status is changing to something other than target status, cancel any scheduled invoices
		if (isset($this->settings['order_status_for_invoice'])) {
			$target_status = $this->settings['order_status_for_invoice'];
			if ('wc-' . $target_status !== $status_to && $target_status !== $status_to) {
				// Remove any existing scheduled invoice
				$this->cancel_scheduled_invoice($order_id);
				return;
			}
		}
		
		// Check if auto-generate is enabled
		if (!isset($this->settings['auto_generate_invoice']) || $this->settings['auto_generate_invoice'] !== 'yes') {
			return;
		}
		
		// Get target status for invoice generation
		$target_status = isset($this->settings['order_status_for_invoice']) ? $this->settings['order_status_for_invoice'] : 'completed';
		
		// Check if order status matches target status
		if ('wc-' . $target_status !== $status_to && $target_status !== $status_to) {
			return;
		}
		
		// Check if invoice already exists to prevent duplicate processing
		$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
		if ($existing_invoice) {
			$this->logger->info('Invoice already exists for this order, skipping generation', array(
				'order_id' => $order_id,
				'invoice_id' => $existing_invoice['invoice_id']
			));
			return;
		}

		// Check if invoice already exists - if so, don't schedule or regenerate
		$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
		if ($existing_invoice) {
			$this->logger->info('Invoice already exists, skipping processing to prevent duplicates', array(
				'order_id' => $order_id,
				'invoice_id' => $existing_invoice['invoice_id'],
				'existing_status' => $existing_invoice['status']
			));
			
			// Remove from scheduled invoices if it exists there
			$scheduled_invoices = get_option('edm_scheduled_invoices', array());
			if (isset($scheduled_invoices[$order_id])) {
				unset($scheduled_invoices[$order_id]);
				update_option('edm_scheduled_invoices', $scheduled_invoices);
				
				$this->logger->info('Removed existing invoice from scheduled list', array(
					'order_id' => $order_id
				));
			}
			
			return;
		}

		// Check if we should delay invoice generation
		$delay_days = isset($this->settings['invoice_delay_days']) ? intval($this->settings['invoice_delay_days']) : 0;
		
		// HPOS compatibility check
		$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
		
		// Check if completed email was already sent
		$email_sent = $hpos_enabled ? 
			$order->get_meta('_edm_completed_email_sent', true) : 
			get_post_meta($order_id, '_edm_completed_email_sent', true);
		
		// Log whether email was sent
		if ($email_sent === 'yes') {
			$this->logger->info('Completed email already sent, continuing with invoice processing', [
				'order_id' => $order_id
			]);
		}
		
		// Process invoice based on delay setting
		if ($delay_days > 0) {
			// Clear any existing scheduled event to prevent duplicates
			wp_clear_scheduled_hook('edm_efatura_generate_invoice', array($order_id));
			
			// GMT+3 için zamanı ayarla
			$current_time = time() + (3 * 3600); // 3 saat ekliyoruz
			$timestamp = $current_time + ($delay_days * 86400); // Dakikaları saniyeye çevirip ekle
			
			// Log scheduling details
			$this->logger->info('Scheduling invoice generation with GMT+3 timezone', array(
				'order_id' => $order_id,
				'delay_days' => $delay_days,
				'current_time_gmt3' => date('Y-m-d H:i:s', $current_time),
				'scheduled_time_gmt3' => date('Y-m-d H:i:s', $timestamp)
			));
			
			// Add private order note
			$order->add_order_note(
				sprintf(
					__('EDM E-invoice generation scheduled for %s (GMT+3).', 'edm-efatura-premium'),
					date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $timestamp)
				),
				0 // Private note
			);
			
			// Store a unique key for this scheduled event
			$schedule_key = 'edm_schedule_' . md5($order_id . time());
			
			// Save scheduled info in order meta
			if ($hpos_enabled) {
				$order->update_meta_data('edm_invoice_scheduled', $timestamp);
				$order->update_meta_data('edm_invoice_schedule_key', $schedule_key);
				$order->save();
			} else {
				update_post_meta($order_id, 'edm_invoice_scheduled', $timestamp);
				update_post_meta($order_id, 'edm_invoice_schedule_key', $schedule_key);
			}
			
			// Send notification about delayed invoice
			$this->maybe_send_delayed_invoice_notification($order);
			
			// Instead of using WordPress cron, use a custom option to track this
			$scheduled_invoices = get_option('edm_scheduled_invoices', array());
			$scheduled_invoices[$order_id] = array(
				'timestamp' => $timestamp,
				'key' => $schedule_key
			);
			update_option('edm_scheduled_invoices', $scheduled_invoices);
			
			$this->logger->info('Added invoice to scheduled list (GMT+3)', array(
				'order_id' => $order_id,
				'scheduled_time_gmt3' => date('Y-m-d H:i:s', $timestamp),
				'scheduled_invoices_count' => count($scheduled_invoices)
			));
		} else {
			// No delay, generate invoice immediately
			try {
				// Generate invoice and send email right away
				$invoice = $this->generate_invoice_for_order($order);
				
				if ($invoice) {
					// Generate PDF if needed
					if (empty($invoice['pdf_path']) || !file_exists($invoice['pdf_path'])) {
						$this->generate_invoice_pdf($invoice);
					}
					
					// Mark as sent
					if ($hpos_enabled) {
						$order->update_meta_data('edm_invoice_email_sent', 'yes');
						$order->update_meta_data('edm_invoice_email_sent_date', current_time('mysql'));
						$order->save();
					} else {
						update_post_meta($order_id, 'edm_invoice_email_sent', 'yes');
						update_post_meta($order_id, 'edm_invoice_email_sent_date', current_time('mysql'));
					}
					
					// Add order note
					$order->add_order_note(
						sprintf(
							__('EDM E-invoice generated successfully. Invoice ID: %s', 'edm-efatura-premium'),
							$invoice['id']
						),
						0 // Private note - not sent to customer
					);
				}
			} catch (Exception $e) {
				// Log error
				$this->logger->error('Failed to generate invoice: ' . $e->getMessage(), array('order_id' => $order_id));
				
				// Add order note
				$order->add_order_note(
					sprintf(
						__('EDM E-invoice generation failed: %s', 'edm-efatura-premium'),
						$e->getMessage()
					)
				);
			}
		}
	}
	/**
 * Send notification about delayed invoice generation
 *
 * @param WC_Order $order Order object
 */
private function maybe_send_delayed_invoice_notification($order) {
    // Get delay days
    $delay_days = isset($this->settings['invoice_delay_days']) ? intval($this->settings['invoice_delay_days']) : 0;
    
    if ($delay_days <= 0) {
        return;
    }
    
    // Check if notification was already sent
    $notification_sent = false;
    $hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
    
    if ($hpos_enabled) {
        $notification_sent = $order->get_meta('edm_delayed_notification_sent', true);
    } else {
        $notification_sent = get_post_meta($order->get_id(), 'edm_delayed_notification_sent', true);
    }
    
    if ($notification_sent === 'yes') {
        return true; // Notification already sent
    }
    
    // Send notification email
    $subject = sprintf(
        __('Your invoice for order #%s will be generated soon', 'edm-efatura-premium'),
        $order->get_order_number()
    );
    
    // Email content
    $content = sprintf(
        __('Your order #%s has been completed. Your invoice will be generated and emailed to you within %d day(s). Please note that you will receive another email with your invoice when it is ready.', 'edm-efatura-premium'),
        $order->get_order_number(),
        $delay_days
    );
    
    // Email headers
    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: ' . get_option('blogname') . ' <' . get_option('admin_email') . '>'
    );
    
    // Try to use WooCommerce mailer instead of wp_mail directly
    try {
        $mailer = WC()->mailer();
        if ($mailer) {
            // Format message with WooCommerce template
            $formatted_content = $mailer->wrap_message(
                sprintf(__('Invoice for Order #%s', 'edm-efatura-premium'), $order->get_order_number()),
                wpautop($content)
            );
            
            // Send the email
            $sent = $mailer->send($order->get_billing_email(), $subject, $formatted_content, $headers, array());
            
            if ($sent) {
                // Mark notification as sent
                if ($hpos_enabled) {
                    $order->update_meta_data('edm_delayed_notification_sent', 'yes');
                    $order->save();
                } else {
                    update_post_meta($order->get_id(), 'edm_delayed_notification_sent', 'yes');
                }
                
                // Add order note
                $order->add_order_note(
                    sprintf(
                        __('Notification about delayed invoice generation sent to customer (%s).', 'edm-efatura-premium'),
                        $order->get_billing_email()
                    ),
                    false // Not customer-facing
                );
                
                return true;
            }
        }
    } catch (Exception $e) {
        // Log error but continue with fallback
        $this->logger->warning('WooCommerce mailer failed for delayed notification: ' . $e->getMessage());
    }
    
    // Fallback to standard wp_mail if WooCommerce mailer fails
    $sent = wp_mail($order->get_billing_email(), $subject, wpautop($content), $headers);
    
    // Log email sending
    if ($sent) {
        // Mark notification as sent
        if ($hpos_enabled) {
            $order->update_meta_data('edm_delayed_notification_sent', 'yes');
            $order->save();
        } else {
            update_post_meta($order->get_id(), 'edm_delayed_notification_sent', 'yes');
        }
        
        // Add order note
        $order->add_order_note(
            sprintf(
                __('Notification about delayed invoice generation sent to customer (%s).', 'edm-efatura-premium'),
                $order->get_billing_email()
            ),
            false // Not customer-facing
        );
        
        return true;
    } else {
        $this->logger->error('Failed to send delayed invoice notification', array(
            'order_id' => $order->get_id(),
            'email' => $order->get_billing_email()
        ));
        
        return false;
    }
}
    
    /**
     * Generate invoice for order
     *
     * @param WC_Order $order Order object
     * @return array|bool Invoice data on success, false on failure
     */
    public function generate_invoice_for_order($order) {
		$order_id = $order->get_id();
		
		// Check if order total is zero or the order has no items with value
		if ($order->get_total() <= 0) {
			$this->logger->info('Skipping invoice generation for zero-value order', array(
				'order_id' => $order_id,
				'total' => $order->get_total()
			));
			
			// Add private order note
			$order->add_order_note(
				__('EDM E-invoice generation skipped. Order has zero value.', 'edm-efatura-premium'),
				0 // Private note
			);
			
			return false;
		}
		
		// Check if invoice already exists
		$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
		if ($existing_invoice) {
			$this->logger->info('Invoice already exists for order', array(
				'order_id' => $order_id,
				'invoice_id' => $existing_invoice['id'],
				'uuid' => $existing_invoice['uuid']
			));
			return $existing_invoice;
		}
		
		// Check if API is available
		if (!$this->api) {
			// Düzeltme: API erişimi yoksa, yeni bir API bağlantısı oluşturmayı deneyelim
			$settings = get_option('edm_efatura_premium_settings', array());
			if (!empty($settings['api_url']) && !empty($settings['username']) && !empty($settings['password'])) {
				$edm_email = !empty($settings['edm_email']) ? $settings['edm_email'] : '';
				
				$this->api = new EDM_Efatura_Premium_API(
					$settings['api_url'],
					$settings['username'],
					$settings['password'],
					isset($settings['debug_mode']) && $settings['debug_mode'] === 'yes',
					$edm_email
				);
				
				// Yeni API bağlantısını kontrol et
				if (!$this->api) {
					$this->logger->error('API not available, cannot generate invoice', array('order_id' => $order_id));
					return false;
				}
			} else {
				$this->logger->error('API credentials not set, cannot generate invoice', array('order_id' => $order_id));
				return false;
			}
		}
		
		try {
			// Get order data for invoice
			$invoice_data = $this->prepare_invoice_data_from_order($order);
			
			
			// Generate UUID if not provided
			$invoice_uuid = wp_generate_uuid4();
			
			// Generate invoice number
			$invoice_id = $this->generate_invoice_number();
			
			// Generate XML with the proper ID and UUID
			$invoice_xml = $this->generate_invoice_xml_with_id_uuid($invoice_data, $invoice_id, $invoice_uuid);
			
			// Send invoice
			$result = $this->api->send_invoice(
				$invoice_data['receiver_vkn'],
				$invoice_data['receiver_alias'],
				$invoice_xml
			);
			
			if ($result['success']) {
				// Save invoice to database
				$invoice = array(
					'id' => $invoice_id,
					'uuid' => $invoice_uuid,
					'order_id' => $order_id,
					'type' => 'EINVOICE',
					'direction' => 'OUT',
					'status' => $result['status'],
					'customer_vkn' => $invoice_data['receiver_vkn'],
					'customer_name' => $invoice_data['customer_name'],
					'amount' => $order->get_total(),
					'currency' => get_woocommerce_currency(),
					'content' => $invoice_xml,
					'extra_data' => array(
						'status_description' => $result['status_description'],
						'order_number' => $order->get_order_number(),
						'invoice_type' => $invoice_data['invoice_type'],
						'profile_id' => $invoice_data['profile_id']
					)
				);
				
				$invoice_db_id = $this->db->save_invoice($invoice);
				
				if (!$invoice_db_id) {
					$this->logger->error('Failed to save invoice to database', array(
						'order_id' => $order_id,
						'invoice_id' => $invoice_id,
						'uuid' => $invoice_uuid
					));
					return false;
				}
				
				// Save items
				$items = array();
				foreach ($order->get_items() as $item) {
					$product = $item->get_product();
					$tax_rate = 0;
					$tax_amount = 0;

					// Calculate tax rate and amount
					if (wc_tax_enabled()) {
						$tax_data = $item->get_taxes();
						if (!empty($tax_data['total'])) {
							$tax_amount = array_sum($tax_data['total']);
							$tax_rate = round(($tax_amount / $item->get_subtotal()) * 100, 2);
						}
					}

					$items[] = array(
						'product_id' => $product ? $product->get_id() : null,
						'name' => $item->get_name(),
						'quantity' => $item->get_quantity(),
						'unit_price' => $item->get_subtotal() / $item->get_quantity(),
						'tax_rate' => $tax_rate,
						'tax_amount' => $tax_amount,
						'total_amount' => $item->get_total() + $tax_amount
					);
				}

				// Add shipping as a line item if applicable
				if ($order->get_shipping_total() > 0) {
					$shipping_tax = $order->get_shipping_tax();
					$shipping_method = $order->get_shipping_method();

					$shipping_name = __('Shipping', 'edm-efatura-premium');
					if (!empty($shipping_method)) {
						$shipping_name .= ': ' . $shipping_method;
					}

					// Use settings for shipping tax rate and unit code
					$settings = get_option('edm_efatura_premium_settings', array());
					$default_tax_rate = 20; // Kargo KDV oranı her zaman %20 sabit
					$shipping_unit_code = isset($settings['shipping_unit_code']) ? $settings['shipping_unit_code'] :
										 (isset($settings['default_unit_code']) ? $settings['default_unit_code'] : 'C62');

					$items[] = array(
						'product_id' => null,
						'name' => $shipping_name,
						'quantity' => 1,
						'unit_price' => $order->get_shipping_total(),
						'unit_code' => $shipping_unit_code,
						'tax_rate' => $default_tax_rate,
						'tax_amount' => $shipping_tax,
						'total_amount' => $order->get_shipping_total() + $shipping_tax,
						'is_shipping' => true
					);
				}

				$this->db->save_invoice_items($invoice_db_id, $items);
				
				// Log success
				$this->logger->info('Invoice generated successfully for order', array(
					'order_id' => $order_id,
					'invoice_id' => $invoice_id,
					'uuid' => $invoice_uuid
				));
				
				// Update order meta to indicate invoice was created
				$this->update_order_invoice_meta($order, $invoice_id, $invoice_uuid);
				
				// Generate PDF
				$this->generate_invoice_pdf(array_merge($invoice, array('db_id' => $invoice_db_id)));
				
				// Trigger usage tracking hook
				do_action('edm_efatura_invoice_created', 
					isset($invoice['uuid']) ? $invoice['uuid'] : '', 
					$order_id, 
					$invoice
				);
				
				// Return invoice data
				return array_merge($invoice, array('db_id' => $invoice_db_id));
			} else {
				// Check if this is a duplicate invoice error from GIB
				$is_duplicate_error = false;
				if (isset($result['message']) &&
				    (strpos($result['message'], 'Daha önce gönderdiğiniz bir faturayı tekrar göndermeye çalıştınız') !== false ||
				     strpos($result['message'], 'duplicate') !== false ||
				     (isset($result['error_code']) && $result['error_code'] == '11048'))) {
					$is_duplicate_error = true;
				}

				if ($is_duplicate_error) {
					// Check if we have this invoice in our database
					$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
					if ($existing_invoice && (!isset($existing_invoice['status']) || $existing_invoice['status'] !== 'CANCELLED')) {
						// We have a valid invoice, return it instead of failing
						$this->logger->info('Duplicate invoice error from GIB, but invoice exists in database', array(
							'order_id' => $order_id,
							'invoice_id' => $existing_invoice['id'],
							'uuid' => $existing_invoice['uuid']
						));
						return $existing_invoice;
					}
				}

				// Log error
				$this->logger->error('Failed to generate invoice for order', array(
					'order_id' => $order_id,
					'error' => $result['message']
				));

				return false;
			}
		} catch (Exception $e) {
			$this->logger->error('Exception while generating invoice', array(
				'order_id' => $order_id,
				'error' => $e->getMessage()
			));
			
			return false;
		}
	}
    
    /**
 * Prepare invoice data from order
 *
 * @param WC_Order $order Order object
 * @return array Invoice data
 */
private function prepare_invoice_data_from_order($order, $invoice_type_code = '') {
    // Check if HPOS is enabled
    $hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
    
    // Get global settings
    $settings = get_option('edm_efatura_premium_settings', array());
    
    // Get invoice type from order meta or global settings
    if ($hpos_enabled) {
        $invoice_type = $order->get_meta('edm_invoice_type', true);
        $invoice_profile = $order->get_meta('edm_invoice_profile', true);
        $invoice_type_code = $order->get_meta('edm_invoice_type_code', true);
        $exception_code = $order->get_meta('edm_exception_code', true);
    } else {
        $invoice_type = get_post_meta($order->get_id(), 'edm_invoice_type', true);
        $invoice_profile = get_post_meta($order->get_id(), 'edm_invoice_profile', true);
        $invoice_type_code = get_post_meta($order->get_id(), 'edm_invoice_type_code', true);
        $exception_code = get_post_meta($order->get_id(), 'edm_exception_code', true);
    }
    
    // If not set, default to individual for customer type
    if (empty($invoice_type)) {
        $invoice_type = 'individual';
    }
    
    // Use global settings if order meta values are not set
    if (empty($invoice_profile)) {
        $invoice_profile = isset($settings['invoice_profile']) ? $settings['invoice_profile'] : 'EARSIVFATURA';
    }
    
    // Use global settings for invoice type code if not set in order meta
    if (empty($invoice_type_code)) {
        $invoice_type_code = isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS';
    }
    
    // Use global settings for exception code if not set in order meta and invoice type is ISTISNA
    if (empty($exception_code) && $invoice_type_code === 'ISTISNA') {
        $exception_code = isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : '';
    }
    
    // Initialize data array with complete required fields
    $data = array(
        'order_id' => $order->get_id(),
        'order_number' => $order->get_order_number(),
        'invoice_type' => $invoice_type_code, // Use the actual invoice type code from meta or settings
        'profile_id' => $invoice_profile,
        'issue_date' => current_time('Y-m-d'), // Use WordPress current_time() for Turkey timezone
        'issue_time' => current_time('H:i:s'), // Use WordPress current_time() for Turkey timezone  
        'currency' => get_woocommerce_currency(),
        'note' => $order->get_customer_note()
    );
    
    // Add exception code if invoice type is ISTISNA and code is available
    if ($data['invoice_type'] === 'ISTISNA' && !empty($exception_code)) {
        $data['exception_code'] = $exception_code;
    }
    
    // Prepare data based on invoice type and profile
    if ($invoice_type === 'corporate' || $invoice_profile === 'TEMELFATURA') {
        // For corporate customers or BASIC invoice, we need VKN
        if ($hpos_enabled) {
            $vkn = $order->get_meta('edm_tax_number', true);
            $tax_office = $order->get_meta('edm_tax_office', true);
            $company_name = $order->get_meta('edm_company_name', true);
        } else {
            $vkn = get_post_meta($order->get_id(), 'edm_tax_number', true);
            $tax_office = get_post_meta($order->get_id(), 'edm_tax_office', true);
            $company_name = get_post_meta($order->get_id(), 'edm_company_name', true);
        }
        
        $data['customer_name'] = $company_name;
        $data['receiver_vkn'] = $vkn;
        $data['receiver_scheme_id'] = 'VKN';
        $data['tax_office'] = $tax_office;
        
        // For corporate invoices (TEMELFATURA), use GIB email from form if available
        $form_gib_email = '';
        if ($hpos_enabled) {
            $form_gib_email = $order->get_meta('edm_gib_email', true);
        } else {
            $form_gib_email = get_post_meta($order->get_id(), 'edm_gib_email', true);
        }
        
        if (!empty($form_gib_email)) {
            $data['receiver_alias'] = $form_gib_email;
        } else {
            // Fallback to default GIB postbox for TEMELFATURA
            $data['receiver_alias'] = 'urn:mail:defaultpk@gmail.com';
        }
    } else {
        // For individual customers (E-Archive Invoice)
        // First check if customer TC meta key is specified
        $customer_tc_meta_key = isset($settings['customer_tc_meta_key']) && !empty($settings['customer_tc_meta_key']) ? 
                                $settings['customer_tc_meta_key'] : '';
        
        if (!empty($customer_tc_meta_key)) {
            // Try to get TC number from the specified order meta key
            if ($hpos_enabled) {
                $vkn = $order->get_meta($customer_tc_meta_key, true);
            } else {
                $vkn = get_post_meta($order->get_id(), $customer_tc_meta_key, true);
            }
        } else {
            // Otherwise, check the default edm_tc_number
            if ($hpos_enabled) {
                $vkn = $order->get_meta('edm_tc_number', true);
            } else {
                $vkn = get_post_meta($order->get_id(), 'edm_tc_number', true);
            }
        }
        
        // If TC number still not available, use default
        if (empty($vkn)) {
            $vkn = isset($settings['default_tc']) ? $settings['default_tc'] : '11111111111';
        }
        
        $data['customer_name'] = $order->get_formatted_billing_full_name();
        $data['receiver_vkn'] = $vkn;
        $data['receiver_scheme_id'] = 'TCKN';
    }
    
    // receiver_alias is set above for corporate invoices, keep empty for individual invoices
    if (!isset($data['receiver_alias'])) {
        $data['receiver_alias'] = '';
    }
    
    // Customer address and contact information - EDM API için HTML etiketleri temizle
    $formatted_address = $order->get_formatted_billing_address();
    // WordPress'in kendi fonksiyonunu kullan - line break'leri korur
    $data['customer_address'] = wp_strip_all_tags($formatted_address, true);
    $data['customer_email'] = $order->get_billing_email();
    $data['customer_phone'] = $order->get_billing_phone();
    
    // KARMA FATURA LOGİĞİ: Kategori ayarlarına göre fatura tipi belirleme
    $has_category_taxed_items = false;
    $has_category_exception_items = false;
    $has_normal_vat_items = false;
    
    foreach ($order->get_items() as $item) {
        $product = $item->get_product();
        if ($product) {
            $has_category_setting = false;
            $category_ids = $product->get_category_ids();
            
            foreach ($category_ids as $category_id) {
                $tax_type = get_term_meta($category_id, 'edm_tax_type', true);
                if (!empty($tax_type)) {
                    $has_category_setting = true;
                    if ($tax_type === 'taxed') {
                        $has_category_taxed_items = true;
                    } elseif ($tax_type === 'exception') {
                        $has_category_exception_items = true;
                    }
                    break; // İlk kategori ayarını kullan
                }
            }
            
            // Kategori ayarı olmayan ürünler için global ayara bak
            if (!$has_category_setting && $data['invoice_type'] === 'SATIS') {
                $has_normal_vat_items = true;
            }
        }
    }
    
    // KARMA FATURA KURAL: Herhangi bir KDV'li item varsa SATIS, hepsi istisna ise ISTISNA
    
    if ($has_category_taxed_items || $has_normal_vat_items) {
        $data['invoice_type'] = 'SATIS';
    } elseif ($has_category_exception_items && !$has_category_taxed_items && !$has_normal_vat_items) {
        $data['invoice_type'] = 'ISTISNA';
    }
    
    // Items'ları güncellenmiş fatura tipi ile al
    $items_with_data = $this->get_order_items_with_exception_codes($order, $data['invoice_type']);
    $data['items'] = $items_with_data['items'];
    
    // Exception code'u ilk istisna item'dan al (eğer fatura tipi ISTISNA ise)
    if ($data['invoice_type'] === 'ISTISNA' && $items_with_data['has_exception'] && !empty($items_with_data['first_exception_code'])) {
        $data['exception_code'] = $items_with_data['first_exception_code'];
    }
    
    // Order totals
    $data['subtotal'] = $order->get_subtotal();
    $data['shipping'] = $order->get_shipping_total();
    $data['discount'] = $order->get_discount_total();
    $data['tax'] = $order->get_total_tax();
    $data['total'] = $order->get_total();
    
    // Add exception code if applicable
    if ($data['invoice_type'] === 'ISTISNA' && !isset($data['exception_code']) && isset($settings['invoice_exception_code'])) {
        $data['exception_code'] = $settings['invoice_exception_code'];
    }
    
    // Get default tax rate from settings if invoice type is SATIS
    if ($data['invoice_type'] === 'SATIS') {
        // Check for custom tax rate first - HIGHEST PRIORITY
        $custom_tax_rate = $order->get_meta('_edm_custom_tax_rate');
        
        if (!empty($custom_tax_rate) && is_numeric($custom_tax_rate) && $custom_tax_rate > 0) {
            $default_tax_rate = floatval($custom_tax_rate);
        } else {
            $default_tax_rate = isset($settings['default_tax_rate']) ? floatval($settings['default_tax_rate']) : 20;
        }
        
        // If tax rate is still 0 or not set, use the fallback
        if ($default_tax_rate <= 0) {
            $default_tax_rate = 20; // Fallback to 20% as default
        }
        
        // Check if WooCommerce prices include tax
        $prices_include_tax = isset($settings['woocommerce_prices_include_tax']) ? $settings['woocommerce_prices_include_tax'] : 'yes';
        
        // Apply this tax rate to all items if needed - NO REFERENCE to avoid PHP bugs
        $processed_items = array();
        
        foreach ($data['items'] as $item) {
            // Skip shipping items - they have their own tax calculation
            if (isset($item['is_shipping']) && $item['is_shipping']) {
                $processed_items[] = $item; // Keep shipping as-is
                continue;
            }
            
            // If tax rate is 0, set it to the default
            if (empty($item['tax_rate']) || $item['tax_rate'] <= 0) {
                $item['tax_rate'] = $default_tax_rate;
                
                // Get the original item total
                $original_total = $item['total'];
                
                // EXCEPTION ITEM CHECK: If this item has exception_code, skip this price recalculation
                // Exception items are already correctly calculated in get_order_items_with_exception_codes()
                if (isset($item['exception_code']) && !empty($item['exception_code'])) {
                    // Keep the exception item data as-is, don't recalculate
                } else {
                    // NORMAL ITEM: Calculate price based on store tax settings
                    if ($prices_include_tax === 'yes') {
                        // Mağaza fiyatları vergi dahil - vergi hariç fiyatı hesapla
                        $item['unit_price'] = $item['unit_price'] / (1 + ($default_tax_rate / 100));
                        $item['total'] = $item['unit_price'] * $item['quantity'];
                        
                        // Tax amount is the difference between original and tax-excluded price
                        $item['tax_amount'] = $original_total - $item['total'];
                        
                    } else {
                        // Mağaza fiyatları vergi hariç - vergi tutarını hesapla
                        $item['tax_amount'] = $item['total'] * ($default_tax_rate / 100);
                        // unit_price zaten vergi hariç, değiştirme
                        
                    }
                }
                
                // Total with tax
                $item['total_with_tax'] = $item['total'] + $item['tax_amount'];
            }
            
            $processed_items[] = $item;
        }
        
        // Replace the original items with processed ones
        $data['items'] = $processed_items;
        
        // Recalculate the totals
        $data['subtotal'] = 0;
        $data['tax'] = 0;
        
        foreach ($data['items'] as $item) {
            $data['subtotal'] += $item['total'];
            $data['tax'] += $item['tax_amount'];
        }
        
        // Reset shipping to 0 since it's now included in recalculated subtotal and tax
        $data['shipping'] = 0;
        
        // If prices include tax and this is a SATIS invoice, adjust the subtotal
        if ($prices_include_tax === 'yes') {
            // For shipping
            if ($data['shipping'] > 0) {
                $shipping_tax_rate = 20; // Kargo her zaman %20 KDV
                $data['shipping'] = $data['shipping'] / (1 + ($shipping_tax_rate / 100));
                $shipping_tax = $data['shipping'] * ($shipping_tax_rate / 100);
                $data['tax'] += $shipping_tax;
            }
        } else {
            // If prices exclude tax, add shipping tax
            if ($data['shipping'] > 0) {
                $shipping_tax_rate = 20; // Kargo her zaman %20 KDV
                $shipping_tax = $data['shipping'] * ($shipping_tax_rate / 100);
                $data['tax'] += $shipping_tax;
            }
        }
        
        // Update total - shipping already included in recalculated subtotal
        $data['total'] = $data['subtotal'] - $data['discount'] + $data['tax'];
    }
    
    // Default unit code
    $data['default_unit_code'] = isset($settings['default_unit_code']) ? $settings['default_unit_code'] : 'C62';
    
    // Add additional payment and shipping details for PDF
    $data['payment_method'] = '';
    $payment_gateway = $order->get_payment_method_title();
    if (!empty($payment_gateway)) {
        if (strpos(strtolower($payment_gateway), 'kredi') !== false || 
            strpos(strtolower($payment_gateway), 'kart') !== false) {
            $data['payment_method'] = 'KREDIKARTI/BANKAKARTI';
        } elseif (strpos(strtolower($payment_gateway), 'havale') !== false || 
                strpos(strtolower($payment_gateway), 'eft') !== false) {
            $data['payment_method'] = 'HAVALE/EFT';
        } elseif (strpos(strtolower($payment_gateway), 'kapı') !== false || 
                strpos(strtolower($payment_gateway), 'nakit') !== false) {
            $data['payment_method'] = 'KAPIDA ODEME';
        } else {
            $data['payment_method'] = $payment_gateway;
        }
    }
    
    // Payment date
    if ($order->get_date_paid()) {
        $data['payment_date'] = date('Y-m-d', $order->get_date_paid()->getTimestamp());
    } elseif ($order->get_date_created()) {
        $data['payment_date'] = date('Y-m-d', $order->get_date_created()->getTimestamp());
    } else {
        $data['payment_date'] = date('Y-m-d');
    }
    
    // Shipping date (default: order date + 1 day)
    if ($order->get_date_created()) {
        $data['shipping_date'] = date('Y-m-d', strtotime('+1 day', $order->get_date_created()->getTimestamp()));
    } else {
        $data['shipping_date'] = date('Y-m-d', strtotime('+1 day'));
    }
    
    return $data;
}
    
	/**
	 * Get order items with exception codes from product categories
	 *
	 * @param WC_Order $order Order object
	 * @return array Items with exception data
	 */
	private function get_order_items_with_exception_codes($order, $invoice_type_code = '') {
    $settings = get_option('edm_efatura_premium_settings', array());
    $default_unit_code = isset($settings['default_unit_code']) ? $settings['default_unit_code'] : 'C62';
    $prices_include_tax = isset($settings['woocommerce_prices_include_tax']) ? $settings['woocommerce_prices_include_tax'] : 'yes';
    
    // Check for custom tax rate first - HIGHEST PRIORITY (only for SATIS)
    if ($invoice_type_code === 'SATIS') {
        $custom_tax_rate = $order->get_meta('_edm_custom_tax_rate');
        if (!empty($custom_tax_rate) && is_numeric($custom_tax_rate) && $custom_tax_rate > 0) {
            $default_tax_rate = floatval($custom_tax_rate);
        } else {
            $default_tax_rate = isset($settings['default_tax_rate']) ? floatval($settings['default_tax_rate']) : 20;
        }
    } else {
        $default_tax_rate = isset($settings['default_tax_rate']) ? floatval($settings['default_tax_rate']) : 20;
    }
    
    // If tax rate is 0 or not set, use a default value
    if ($default_tax_rate <= 0) {
        $default_tax_rate = 20; // Fallback to 20% as default
    }
    
    $items = array();
    $has_exception = false;
    $first_exception_code = '';
    
    foreach ($order->get_items() as $item) {
        $product = $item->get_product();
        $tax_rate = 0;
        $tax_amount = 0;
        
        // Calculate tax rate and amount properly
        if (wc_tax_enabled()) {
            $tax_data = $item->get_taxes();
            if (!empty($tax_data['total'])) {
                $tax_amount = array_sum($tax_data['total']);
                if ($item->get_subtotal() > 0) {
                    $tax_rate = round(($tax_amount / $item->get_subtotal()) * 100, 2);
                }
            }
        }
        
        // Get unit code from product metadata if available
        $unit_code = $default_unit_code;
        if ($product) {
            $product_unit_code = get_post_meta($product->get_id(), '_unit_code', true);
            if (!empty($product_unit_code)) {
                $unit_code = $product_unit_code;
            }
        }
        
        // Check for category tax settings (new enhanced system)
        $exception_code = '';
        $category_tax_override = false;
        
        if ($product) {
            // Direct category tax settings check (without depending on global instance)
            $category_ids = $product->get_category_ids();
            
            foreach ($category_ids as $category_id) {
                $tax_type = get_term_meta($category_id, 'edm_tax_type', true);
                
                
                if (!empty($tax_type)) {
                    $category_tax_override = true;
                    
                    if ($tax_type === 'exception') {
                        // Category has exception setting
                        $exception_code = get_term_meta($category_id, 'edm_exception_code', true);
                        $has_exception = true;
                        
                        // Keep track of the first found exception code
                        if (empty($first_exception_code)) {
                            $first_exception_code = $exception_code;
                        }
                        
                        // Set tax rate to 0 for exception items
                        $tax_rate = 0;
                        $tax_amount = 0;
                        
                    } elseif ($tax_type === 'taxed') {
                        // Category has custom tax rate setting - NOT AN EXCEPTION!
                        $custom_tax_rate = floatval(get_term_meta($category_id, 'edm_tax_rate', true));
                        if ($custom_tax_rate > 0) {
                            $tax_rate = $custom_tax_rate;
                            
                            // Recalculate tax amount based on custom rate
                            if ($prices_include_tax === 'yes') {
                                // Price includes tax, calculate tax amount
                                $price_without_tax = $item->get_subtotal() / (1 + ($tax_rate / 100));
                                $tax_amount = $item->get_subtotal() - $price_without_tax;
                            } else {
                                // Price excludes tax, calculate tax amount
                                $tax_amount = ($item->get_subtotal() * $tax_rate) / 100;
                            }
                        }
                        
                        // IMPORTANT: This is NOT an exception - it's a taxed sale
                        // Do not set has_exception = true here!
                    }
                    
                    break; // Use first category with settings
                }
            }
            
            // Fallback to old system if no new category settings found
            if (!$category_tax_override) {
                $terms = get_the_terms($product->get_id(), 'product_cat');
                if (!empty($terms) && !is_wp_error($terms)) {
                    foreach ($terms as $term) {
                        $cat_exception_code = get_term_meta($term->term_id, 'edm_exception_code', true);
                        if (!empty($cat_exception_code)) {
                            $exception_code = $cat_exception_code;
                            $has_exception = true;
                            
                            // Keep track of the first found exception code
                            if (empty($first_exception_code)) {
                                $first_exception_code = $cat_exception_code;
                            }
                            
                            break; // Use the first exception code found
                        }
                    }
                }
                
                // KARMA FATURA FIX: Kategori ayarı olmayan ürünler için global settings kontrolü
                // Global invoice type ISTISNA ise, bu ürün istisna olmalı (karma faturada fatura tipi SATIS olsa bile)
                if (empty($exception_code)) {
                    $settings = get_option('edm_efatura_premium_settings', array());
                    $global_invoice_type = isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS';
                    
                    if ($global_invoice_type === 'ISTISNA') {
                        // Global settings'den exception code al
                        $global_exception_code = isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : '';
                        
                        if (!empty($global_exception_code)) {
                            $exception_code = $global_exception_code;
                            $has_exception = true;
                            
                            // ÖNEMLI: Global ISTISNA ürününde orjinal WooCommerce vergi oranını korumak gerekiyor
                            // Çünkü fiyat hesaplamasında original tax_rate kullanılacak, sonra vergi 0 yapılacak
                            $original_tax_rate = $tax_rate; // WooCommerce'den gelen vergi oranı
                            
                            // Keep track of the first found exception code
                            if (empty($first_exception_code)) {
                                $first_exception_code = $global_exception_code;
                            }
                        }
                    }
                }
            }
        }
        
        // Ensure quantity is not zero to prevent division by zero
        $quantity = max(1, $item->get_quantity());
        $unit_price = $item->get_subtotal() / $quantity;
        
        // Handle KDV included/excluded calculation for product items
        if (!empty($exception_code)) {
            // İSTISNA ÜRÜN: Fiyat hesaplaması özel yapılmalı
            if ($prices_include_tax === 'yes') {
                // İSTISNA ÜRÜNLER İÇİN: WooCommerce'deki vergi dahil fiyatı aynen kullan
                // Çünkü istisna ürünlerde gösterilecek birim fiyat = müşterinin ödediği vergi dahil fiyat
                $line_total = $item->get_subtotal() + $item->get_subtotal_tax(); // Vergi dahil toplam tutar
                $unit_price_without_tax = $line_total / $quantity; // Bu aslında vergi dahil unit price (istisna ürünlerde "vergi hariç" = "vergi dahil")
                $tax_amount = 0; // İstisna ürünlerde vergi 0
                $tax_rate = 0; // İstisna ürünlerde vergi oranı 0
            } else {
                // Fiyatlar KDV hariç ve istisna ürünlerde KDV 0
                $unit_price_without_tax = $unit_price;
                $line_total = $item->get_subtotal();
                $tax_amount = 0;
                $tax_rate = 0;
            }
        } else {
            // NORMAL ÜRÜN: Standart hesaplama
            if ($prices_include_tax === 'yes') {
                // If prices include tax, calculate the price without tax
                if ($tax_rate > 0) {
                    $unit_price_without_tax = $unit_price / (1 + ($tax_rate / 100));
                    $line_total = $unit_price_without_tax * $quantity;
                    $tax_amount = $item->get_subtotal() - $line_total;
                } else {
                    $unit_price_without_tax = $unit_price;
                    $line_total = $item->get_subtotal();
                    $tax_amount = 0;
                }
            } else {
                // If prices exclude tax, calculate tax amount
                $unit_price_without_tax = $unit_price;
                $line_total = $item->get_subtotal();
                $tax_amount = ($tax_rate > 0) ? ($line_total * ($tax_rate / 100)) : 0;
            }
        }
        
        $new_item = array(
            'id' => $product ? $product->get_id() : null,
            'sku' => $product ? $product->get_sku() : '',
            'name' => $item->get_name(),
            'quantity' => $quantity,
            'unit_price' => $unit_price_without_tax,
            'unit_code' => $unit_code,
            'tax_rate' => $tax_rate,
            'tax_amount' => $tax_amount,
            'total' => $line_total,
            'total_with_tax' => $line_total + $tax_amount,
            'exception_code' => $exception_code,
            'is_shipping' => false
        );
        
        $items[] = $new_item;
    }
    
    // Add shipping as a separate line item if applicable
    if ($order->get_shipping_total() > 0) {
		$shipping_tax = $order->get_shipping_tax();
		
		// For shipping, ALWAYS use 20% tax rate regardless of invoice type or admin settings
		$shipping_tax_rate = 20;
		
		// Get shipping total - handle KDV included/excluded calculation
		$shipping_total = $order->get_shipping_total();
		
		if ($prices_include_tax === 'yes') {
			// If prices include tax, calculate shipping without tax
			$shipping_total_without_tax = $shipping_total / (1 + ($shipping_tax_rate / 100));
			$shipping_tax_calculated = $shipping_total - $shipping_total_without_tax;
		} else {
			// If prices exclude tax, calculate the tax amount
			$shipping_total_without_tax = $shipping_total;
			$shipping_tax_calculated = $shipping_total * ($shipping_tax_rate / 100);
		}
		
		// Burada admin ayarlarından alınan unit_code değerini kullanıyoruz
		// Eğer ayarlar kısmında özel bir kargo unit_code varsa onu, yoksa default_unit_code'u kullan
		$shipping_unit_code = isset($settings['shipping_unit_code']) ? $settings['shipping_unit_code'] : $default_unit_code;
		
		$items[] = array(
			'id' => null,
			'sku' => 'SHIPPING',
			'name' => __('Shipping', 'edm-efatura-premium') . ': ' . $order->get_shipping_method(),
			'quantity' => 1,
			'unit_price' => $shipping_total_without_tax,
			'unit_code' => $shipping_unit_code, // Ayarlardan alınan unit_code değerini kullan
			'tax_rate' => $shipping_tax_rate, // Always 20% for shipping
			'tax_amount' => $shipping_tax_calculated,
			'total' => $shipping_total_without_tax,
			'total_with_tax' => $shipping_total_without_tax + $shipping_tax_calculated,
			'exception_code' => '', // Shipping typically doesn't have exception code
			'is_shipping' => true // Mark this as a shipping item
		);
	}
    
    return array(
        'items' => $items,
        'has_exception' => $has_exception,
        'first_exception_code' => $first_exception_code
    );
}
	
    /**
     * Get formatted order items
     *
     * @param WC_Order $order Order object
     * @return array Order items
     */
    private function get_order_items($order) {
        $items = array();
        
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $tax_rate = 0;
            $tax_amount = 0;
            
            // Calculate tax rate and amount
            if (wc_tax_enabled()) {
                $tax_data = $item->get_taxes();
                if (!empty($tax_data['total'])) {
                    $tax_amount = array_sum($tax_data['total']);
                    $tax_rate = round(($tax_amount / $item->get_subtotal()) * 100, 2);
                }
            }
            
            $items[] = array(
                'id' => $product ? $product->get_id() : null,
                'sku' => $product ? $product->get_sku() : '',
                'name' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'unit_price' => $item->get_subtotal() / $item->get_quantity(),
                'tax_rate' => $tax_rate,
                'tax_amount' => $tax_amount,
                'total' => $item->get_total(),
                'total_with_tax' => $item->get_total() + $tax_amount
            );
        }
        
        // Add shipping as a line item if applicable
        if ($order->get_shipping_total() > 0) {
			$shipping_tax = $order->get_shipping_tax();
			
			// For shipping, ALWAYS use 20% tax rate regardless of invoice type
			$shipping_tax_rate = 20; // Kargo her zaman %20 KDV
			
			// Get shipping total - handle KDV included/excluded calculation
			$shipping_total = $order->get_shipping_total();
			
			if ($prices_include_tax === 'yes') {
				// If prices include tax, calculate shipping without tax
				$shipping_total_without_tax = $shipping_total / (1 + ($shipping_tax_rate / 100));
				$shipping_tax_calculated = $shipping_total - $shipping_total_without_tax;
			} else {
				// If prices exclude tax, calculate the tax amount
				$shipping_total_without_tax = $shipping_total;
				$shipping_tax_calculated = $shipping_total * ($shipping_tax_rate / 100);
			}
			
			// Burada admin ayarlarından alınan unit_code değerini kullanıyoruz
			// Eğer ayarlar kısmında özel bir kargo unit_code varsa onu, yoksa default_unit_code'u kullan
			$shipping_unit_code = isset($settings['shipping_unit_code']) ? $settings['shipping_unit_code'] : $default_unit_code;
			
			$items[] = array(
				'id' => null,
				'sku' => 'SHIPPING',
				'name' => __('Shipping', 'edm-efatura-premium') . ': ' . $order->get_shipping_method(),
				'quantity' => 1,
				'unit_price' => $shipping_total_without_tax,
				'unit_code' => $shipping_unit_code, // Ayarlardan alınan unit_code değerini kullan
				'tax_rate' => $shipping_tax_rate, // Always 20% for shipping
				'tax_amount' => $shipping_tax_calculated,
				'total' => $shipping_total_without_tax,
				'total_with_tax' => $shipping_total_without_tax + $shipping_tax_calculated,
				'exception_code' => '', // Shipping typically doesn't have exception code
				'is_shipping' => true // Mark this as a shipping item
			);
		}
        
        return $items;
    }
    
/**
 * Sıralı fatura numarası oluşturma
 * 
 * @param string $prefix Şirket kodu (3 harf)
 * @param int $last_number Son oluşturulan fatura numarası
 * @return string Oluşturulan fatura numarası
 */
private function generate_invoice_number($prefix = '', $last_number = 0) {
    // Use prefix from settings if not provided
    if (empty($prefix)) {
        $prefix = isset($this->settings['company_prefix']) ? $this->settings['company_prefix'] : 'ONN';
    }
    
    // Limit company code to 3 letters and convert to uppercase
    $prefix = substr(strtoupper($prefix), 0, 3);
    
    // Current year
    $current_year = date('Y');
    
    // Create an option name that includes the year, so each year starts fresh
    $option_name = 'edm_last_invoice_number_' . $prefix . '_' . $current_year;
    
    // Get the last invoice number from the option, default to 0 if not set
    $last_invoice_number = get_option($option_name, 0);
    
    // Increment the counter
    $next_number = $last_invoice_number + 1;
    
    // Update the option with the new number
    update_option($option_name, $next_number);
    
    // Format with leading zeros to 9 digits
    $formatted_number = str_pad($next_number, 9, '0', STR_PAD_LEFT);
    
    // Complete invoice number
    $invoice_number = $prefix . $current_year . $formatted_number;
    
    return $invoice_number;
}





/**
 * E-Fatura XML oluşturma fonksiyonu
 *
 * @param array $data Fatura verileri
 * @param string $invoice_id Invoice ID
 * @param string $invoice_uuid Invoice UUID
 * @return string XML içeriği
 */
private function generate_invoice_xml_with_id_uuid($data, $invoice_id, $invoice_uuid) {
    // Exception code descriptions - available throughout the function
    $exception_descriptions = EDM_EFatura_Premium_Exception_Codes::get_turkish_exception_descriptions();

    // Current date and time using Turkey timezone
    $issue_date = $data['issue_date'] ?? current_time('Y-m-d');
    $issue_time = $data['issue_time'] ?? current_time('H:i:s');
    
    // Use provided ID and UUID
    $invoice_id = $invoice_id;
    $invoice_uuid = $invoice_uuid;
    
    // Company VKN/TCKN from settings
    $settings = get_option('edm_efatura_premium_settings', array());
    $company_vkn = isset($settings['company_vkn']) ? $settings['company_vkn'] : '';
    $company_name = isset($settings['company_name']) ? $settings['company_name'] : get_bloginfo('name');
    
    // Get profile ID from settings
    $profile_id = isset($data['profile_id']) ? $data['profile_id'] : 
                 (isset($settings['invoice_profile']) ? $settings['invoice_profile'] : 'EARSIVFATURA');
    
    // Get invoice type code - PRIORITIZE data over settings (IMPORTANT FIX)
    $invoice_type_code = isset($data['invoice_type']) && !empty($data['invoice_type']) ? $data['invoice_type'] : 
                        (isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS');
    
    // Get WooCommerce extra data for e-commerce information
    $order = null;
    $payment_method = isset($data['payment_method']) ? $data['payment_method'] : 'KREDIKARTI/BANKAKARTI'; // Default value
    $payment_date = isset($data['payment_date']) ? $data['payment_date'] : date('Y-m-d');
    $shipping_date = isset($data['shipping_date']) ? $data['shipping_date'] : date('Y-m-d', strtotime('+1 day'));
    $website_url = isset($settings['company_website']) ? $settings['company_website'] : get_site_url();
    
    // Get payment intermediary from settings
    $payment_intermediary = isset($settings['payment_intermediary']) ? $settings['payment_intermediary'] : 'sanal';
    
    // Get shipping company name from settings
    $shipping_company = isset($settings['shipping_company_name']) ? $settings['shipping_company_name'] : 'Kargo';
    
    // Start XML with correct namespaces
    $xml = '<?xml version="1.0" encoding="UTF-8"?>
<Invoice xmlns="urn:oasis:names:specification:ubl:schema:xsd:Invoice-2" 
         xmlns:cac="urn:oasis:names:specification:ubl:schema:xsd:CommonAggregateComponents-2" 
         xmlns:cbc="urn:oasis:names:specification:ubl:schema:xsd:CommonBasicComponents-2" 
         xmlns:ext="urn:oasis:names:specification:ubl:schema:xsd:CommonExtensionComponents-2">
    <cbc:UBLVersionID>2.1</cbc:UBLVersionID>
    <cbc:CustomizationID>TR1.2</cbc:CustomizationID>
    <cbc:ProfileID>' . $profile_id . '</cbc:ProfileID>
    <cbc:ID>' . $invoice_id . '</cbc:ID>
    <cbc:CopyIndicator>false</cbc:CopyIndicator>
    <cbc:UUID>' . $invoice_uuid . '</cbc:UUID>
    <cbc:IssueDate>' . $issue_date . '</cbc:IssueDate>
    <cbc:IssueTime>' . $issue_time . '</cbc:IssueTime>
    <cbc:InvoiceTypeCode>' . $invoice_type_code . '</cbc:InvoiceTypeCode>';
    
    // Add order amount in words if available
    if (isset($data['total'])) {
        $total_amount = $data['total'];
        $words_total = $this->convert_number_to_words($total_amount, $data['currency']);
        $xml .= '
    <cbc:Note>Yazı ile: # ' . $words_total . ' #</cbc:Note>';
    }
    
    // Add note if provided
    if (!empty($data['note'])) {
        $xml .= '
    <cbc:Note>' . htmlspecialchars($data['note']) . '</cbc:Note>';
    }
    
    // Add order number as a note
    if (!empty($data['order_number'])) {
        $xml .= '
    <cbc:Note>Sipariş Numarası: ' . $data['order_number'] . '</cbc:Note>';
    }
    
    // Add e-commerce information in a standardized format
    $xml .= '
    <cbc:Note> Web Adresi : ' . $website_url . '</cbc:Note>
    <cbc:Note> Ödeme Şekli : ' . $payment_method . '</cbc:Note>
    <cbc:Note>Ödeme Aracısı : ' . $payment_intermediary . '</cbc:Note>
    <cbc:Note> Ödeme Tarihi : ' . $payment_date . '</cbc:Note>
    <cbc:Note>Gönderim Tarihi : ' . $shipping_date . '</cbc:Note>
    <cbc:Note>Gönderi Taşıyan Unvan/İsim : ' . $shipping_company . '</cbc:Note>';
    
    // Add exception codes for all exception items (ISTISNA invoices and mixed invoices)
    $used_exception_codes = array();
    
    // Collect all unique exception codes from items
    foreach ($data['items'] as $item) {
        if (isset($item['exception_code']) && !empty($item['exception_code']) && !in_array($item['exception_code'], $used_exception_codes)) {
            $used_exception_codes[] = $item['exception_code'];
        }
    }
    
    // Add main exception code if invoice type is ISTISNA
    if ($invoice_type_code === 'ISTISNA') {
        $main_exception_code = isset($data['exception_code']) ? $data['exception_code'] : 
                              (isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : EDM_EFatura_Premium_Exception_Codes::get_default_exception_code());
        
        if (!in_array($main_exception_code, $used_exception_codes)) {
            $used_exception_codes[] = $main_exception_code;
        }
    }
    
    // Add notes for all exception codes
    foreach ($used_exception_codes as $exception_code) {
        $exception_description = isset($exception_descriptions[$exception_code]) ? 
                              $exception_descriptions[$exception_code] : 
                              __('Tax Exemption', 'edm-efatura-premium');
        
        $xml .= '
    <cbc:Note>Vergi İstisna Kodu: ' . $exception_code . ' - ' . $exception_description . '</cbc:Note>';
        
    }
    
    // Continue with the rest of the XML generation...
    $xml .= '
    <cbc:DocumentCurrencyCode>' . $data['currency'] . '</cbc:DocumentCurrencyCode>
    <cbc:LineCountNumeric>' . count($data['items']) . '</cbc:LineCountNumeric>';
    
    // Supplier (Seller) information
    $xml .= '
    <cac:AccountingSupplierParty>
        <cac:Party>
            <cbc:WebsiteURI>' . $website_url . '</cbc:WebsiteURI>';
    
    // Seller VKN/TCKN check and cac:Person preparation
    $company_person_element = '';
    if (strlen($company_vkn) == 10) {
        $xml .= '
            <cac:PartyIdentification>
                <cbc:ID schemeID="VKN">' . $company_vkn . '</cbc:ID>
            </cac:PartyIdentification>';
    } else if (strlen($company_vkn) == 11) {
        $xml .= '
            <cac:PartyIdentification>
                <cbc:ID schemeID="TCKN">' . $company_vkn . '</cbc:ID>
            </cac:PartyIdentification>';
        
        // Prepare Person element for TCKN
        $company_name_parts = explode(' ', $company_name, 2);
        $company_first_name = isset($company_name_parts[0]) ? $company_name_parts[0] : $company_name;
        $company_family_name = isset($company_name_parts[1]) ? $company_name_parts[1] : '';
        
        $company_person_element = '
            <cac:Person>
                <cbc:FirstName>' . htmlspecialchars($company_first_name) . '</cbc:FirstName>
                <cbc:FamilyName>' . htmlspecialchars($company_family_name) . '</cbc:FamilyName>
            </cac:Person>';
    }
    
    // Company identification fields
    $xml .= '
            <cac:PartyIdentification>
                <cbc:ID schemeID="MERSISNO">' . (isset($settings['company_mersis']) ? $settings['company_mersis'] : '') . '</cbc:ID>
            </cac:PartyIdentification>
            <cac:PartyIdentification>
                <cbc:ID schemeID="TICARETSICILNO">' . (isset($settings['company_trade_registry']) ? $settings['company_trade_registry'] : '') . '</cbc:ID>
            </cac:PartyIdentification>';
    
    // Company name and address from settings
    $xml .= '
            <cac:PartyName>
                <cbc:Name>' . htmlspecialchars($company_name) . '</cbc:Name>
            </cac:PartyName>
            <cac:PostalAddress>
                <cbc:StreetName>' . htmlspecialchars(isset($settings['company_street']) ? $settings['company_street'] : '') . '</cbc:StreetName>
                <cbc:BuildingNumber>' . htmlspecialchars(isset($settings['company_building']) ? $settings['company_building'] : '') . '</cbc:BuildingNumber>
                <cbc:CitySubdivisionName>' . htmlspecialchars(isset($settings['company_district']) ? $settings['company_district'] : '') . '</cbc:CitySubdivisionName>
                <cbc:CityName>' . htmlspecialchars(isset($settings['company_city']) ? $settings['company_city'] : '') . '</cbc:CityName>
                <cbc:PostalZone>' . htmlspecialchars(isset($settings['company_postal']) ? $settings['company_postal'] : '') . '</cbc:PostalZone>
                <cbc:Region>' . htmlspecialchars(isset($settings['company_region']) ? $settings['company_region'] : '') . '</cbc:Region>
                <cac:Country>
                    <cbc:IdentificationCode>' . htmlspecialchars(isset($settings['company_country_code']) ? $settings['company_country_code'] : 'TR') . '</cbc:IdentificationCode>
                    <cbc:Name>' . htmlspecialchars(isset($settings['company_country']) ? $settings['company_country'] : 'Türkiye') . '</cbc:Name>
                </cac:Country>
            </cac:PostalAddress>
            <cac:PartyTaxScheme>
                <cac:TaxScheme>
                    <cbc:Name>' . htmlspecialchars(isset($settings['company_tax_office']) ? $settings['company_tax_office'] : '') . '</cbc:Name>
                </cac:TaxScheme>
            </cac:PartyTaxScheme>
            <cac:Contact>
                <cbc:Telephone>' . htmlspecialchars(isset($settings['company_phone']) ? $settings['company_phone'] : '') . '</cbc:Telephone>
                <cbc:Telefax>' . htmlspecialchars(isset($settings['company_fax']) ? $settings['company_fax'] : '') . '</cbc:Telefax>
                <cbc:ElectronicMail>' . htmlspecialchars(isset($settings['company_email']) ? $settings['company_email'] : get_bloginfo('admin_email')) . '</cbc:ElectronicMail>
            </cac:Contact>';
    
    // Add Person element if it exists
    if (!empty($company_person_element)) {
        $xml .= $company_person_element;
    }
    
    $xml .= '
        </cac:Party>
    </cac:AccountingSupplierParty>';
    
    // Customer information - VKN/TCKN check
    $scheme_id = '';
    if (strlen($data['receiver_vkn']) == 10) {
        $scheme_id = 'VKN';
    } else if (strlen($data['receiver_vkn']) == 11) {
        $scheme_id = 'TCKN';
    } else {
        // Default value for invalid length
        $scheme_id = 'VKN';
    }
    
    // Customer information (without receiver_alias - not needed)
    $xml .= '
    <cac:AccountingCustomerParty>
        <cac:Party>
            <cac:PartyIdentification>
                <cbc:ID schemeID="' . $scheme_id . '">' . $data['receiver_vkn'] . '</cbc:ID>
            </cac:PartyIdentification>
            <cac:PartyName>
                <cbc:Name>' . htmlspecialchars($data['customer_name']) . '</cbc:Name>
            </cac:PartyName>
            <cac:PostalAddress>';
            
    // Parse customer address if available
    if (!empty($data['customer_address'])) {
        // Split address by lines
        $address_lines = explode("\n", $data['customer_address']);
        $street = isset($address_lines[0]) ? $address_lines[0] : '';
        $building = '1'; // Default building number
        
        // Try to extract city, region, and postal code from address
        $city = isset($data['billing_city']) ? $data['billing_city'] : 'İstanbul';
        $postal_code = isset($data['billing_postcode']) ? $data['billing_postcode'] : '34000';
        $region = isset($data['billing_state']) ? $data['billing_state'] : '';
        $district = isset($data['billing_district']) ? $data['billing_district'] : 'Merkez';
        
        $xml .= '
                <cbc:StreetName>' . htmlspecialchars($street) . '</cbc:StreetName>
                <cbc:BuildingNumber>' . htmlspecialchars($building) . '</cbc:BuildingNumber>
                <cbc:CitySubdivisionName>' . htmlspecialchars($district) . '</cbc:CitySubdivisionName>
                <cbc:CityName>' . htmlspecialchars($city) . '</cbc:CityName>
                <cbc:PostalZone>' . htmlspecialchars($postal_code) . '</cbc:PostalZone>';
                
        if (!empty($region)) {
            $xml .= '
                <cbc:Region>' . htmlspecialchars($region) . '</cbc:Region>';
        }
    } else {
        // Default values if no address available
        $xml .= '
                <cbc:StreetName>' . htmlspecialchars($data['customer_name']) . '</cbc:StreetName>
                <cbc:BuildingNumber>1</cbc:BuildingNumber>
                <cbc:CitySubdivisionName>Merkez</cbc:CitySubdivisionName>
                <cbc:CityName>İstanbul</cbc:CityName>
                <cbc:PostalZone>34000</cbc:PostalZone>';
    }
            
    // Country information - default to Turkey if not provided
    $country_code = isset($data['billing_country']) ? $data['billing_country'] : 'TR';
    $country_name = isset($data['billing_country_name']) ? $data['billing_country_name'] : 'Türkiye';
    
    $xml .= '
                <cac:Country>
                    <cbc:IdentificationCode>' . htmlspecialchars($country_code) . '</cbc:IdentificationCode>
                    <cbc:Name>' . htmlspecialchars($country_name) . '</cbc:Name>
                </cac:Country>
            </cac:PostalAddress>';
    
    // Contact information for customer - optional
    if (!empty($data['customer_phone']) || !empty($data['customer_email'])) {
        $xml .= '
            <cac:Contact>';
        
        if (!empty($data['customer_phone'])) {
            $xml .= '
                <cbc:Telephone>' . htmlspecialchars($data['customer_phone']) . '</cbc:Telephone>';
        }
        
        if (!empty($data['customer_email'])) {
            $xml .= '
                <cbc:ElectronicMail>' . htmlspecialchars($data['customer_email']) . '</cbc:ElectronicMail>';
        }
        
        $xml .= '
            </cac:Contact>';
    }
    
    // Add Person element for individuals (TCKN)
    if ($scheme_id === 'TCKN') {
        $customer_name_parts = explode(' ', $data['customer_name'], 2);
        $first_name = isset($customer_name_parts[0]) ? $customer_name_parts[0] : $data['customer_name'];
        $family_name = isset($customer_name_parts[1]) ? $customer_name_parts[1] : '';
        
        $xml .= '
            <cac:Person>
                <cbc:FirstName>' . htmlspecialchars($first_name) . '</cbc:FirstName>
                <cbc:FamilyName>' . htmlspecialchars($family_name) . '</cbc:FamilyName>
            </cac:Person>';
    }
    
    // Add tax office for corporate customers
    if ($scheme_id === 'VKN' && !empty($data['tax_office'])) {
        $xml .= '
            <cac:PartyTaxScheme>
                <cac:TaxScheme>
                    <cbc:Name>' . htmlspecialchars($data['tax_office']) . '</cbc:Name>
                </cac:TaxScheme>
            </cac:PartyTaxScheme>';
    }
    
    $xml .= '
        </cac:Party>
    </cac:AccountingCustomerParty>';
    
    // Payment Terms - for internet sales
    if ($profile_id === 'INTERNETSATIS' || strpos($profile_id, 'INTERNET') !== false) {
        $payment_date_formatted = date('Y-m-d', strtotime(str_replace('.', '-', $payment_date)));
        $xml .= '
    <cac:PaymentTerms>
        <cbc:PaymentDueDate>' . $payment_date_formatted . '</cbc:PaymentDueDate>
    </cac:PaymentTerms>';
    }
    
    // Tax totals - item'lardan gerçek vergi hesaplama
    $tax_totals = array();
    $total_calculated_tax = 0;
    foreach ($data['items'] as $item) {
        $tax_rate = isset($item['tax_rate']) ? $item['tax_rate'] : 0;
        $tax_amount = isset($item['tax_amount']) ? $item['tax_amount'] : 0;
        
        if (!isset($tax_totals[$tax_rate])) {
            $tax_totals[$tax_rate] = 0;
        }
        $tax_totals[$tax_rate] += $tax_amount;
        $total_calculated_tax += $tax_amount;
        
    }
    
    
    $xml .= '
    <cac:TaxTotal>
        <cbc:TaxAmount currencyID="' . $data['currency'] . '">' . number_format($total_calculated_tax, 2, '.', '') . '</cbc:TaxAmount>';
    
    // Tax details for each tax rate
    foreach ($tax_totals as $rate => $amount) {
        $xml .= '
        <cac:TaxSubtotal>
            <cbc:TaxableAmount currencyID="' . $data['currency'] . '">' . number_format($data['subtotal'], 2, '.', '') . '</cbc:TaxableAmount>
            <cbc:TaxAmount currencyID="' . $data['currency'] . '">' . number_format($amount, 2, '.', '') . '</cbc:TaxAmount>
           <cbc:CalculationSequenceNumeric>1</cbc:CalculationSequenceNumeric>
           <cbc:Percent>' . number_format($rate, 0, '.', '') . '</cbc:Percent>
           <cac:TaxCategory>';
           
       // Exception status - add TaxExemptionReasonCode and TaxExemptionReason for zero-tax items
       // This is REQUIRED by EDM API for items with 0% tax rate
       if ($rate == 0 && $invoice_type_code === 'ISTISNA') {
			$exception_code = isset($data['exception_code']) ? $data['exception_code'] : 
							(isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : EDM_EFatura_Premium_Exception_Codes::get_default_exception_code());
			
			$exception_description = isset($exception_descriptions[$exception_code]) ? 
								 $exception_descriptions[$exception_code] : 
								 'Basılı Kitap ve Süreli Yayınların Teslimleri';
			
			$xml .= '
				<cbc:TaxExemptionReasonCode>' . $exception_code . '</cbc:TaxExemptionReasonCode>
				<cbc:TaxExemptionReason>' . $exception_description . '</cbc:TaxExemptionReason>';
		}
       
       $xml .= '
               <cac:TaxScheme>
                   <cbc:Name>KDV</cbc:Name>
                   <cbc:TaxTypeCode>0015</cbc:TaxTypeCode>
               </cac:TaxScheme>
           </cac:TaxCategory>
       </cac:TaxSubtotal>';
   }
   
   $xml .= '
   </cac:TaxTotal>';
   
   // Total amounts - calculated tax kullan
   if ($data['invoice_type'] === 'SATIS') {
       // SATIS: subtotal already includes shipping from recalculation
       $line_extension_amount = $data['subtotal'];
       $tax_exclusive_amount = $data['subtotal'];  
       $tax_inclusive_amount = $data['subtotal'] + $total_calculated_tax;
   } else {
       // ISTISNA: manual shipping calculation needed
       $shipping_without_tax = $data['shipping'] / 1.20; // Assuming 20% shipping tax for ISTISNA
       $line_extension_amount = $data['subtotal'] + $shipping_without_tax;
       $tax_exclusive_amount = $data['subtotal'] + $shipping_without_tax;
       $tax_inclusive_amount = $data['subtotal'] + $data['shipping'] + $data['tax'];
   }
   
   $xml .= '
   <cac:LegalMonetaryTotal>
       <cbc:LineExtensionAmount currencyID="' . $data['currency'] . '">' . number_format($line_extension_amount, 2, '.', '') . '</cbc:LineExtensionAmount>
       <cbc:TaxExclusiveAmount currencyID="' . $data['currency'] . '">' . number_format($tax_exclusive_amount, 2, '.', '') . '</cbc:TaxExclusiveAmount>
       <cbc:TaxInclusiveAmount currencyID="' . $data['currency'] . '">' . number_format($tax_inclusive_amount, 2, '.', '') . '</cbc:TaxInclusiveAmount>';
   
   // Add discount if present
   if (!empty($data['discount']) && $data['discount'] > 0) {
       $xml .= '
       <cbc:AllowanceTotalAmount currencyID="' . $data['currency'] . '">' . number_format($data['discount'], 2, '.', '') . '</cbc:AllowanceTotalAmount>';
   } else {
       // Include with 0 value
       $xml .= '
       <cbc:AllowanceTotalAmount currencyID="' . $data['currency'] . '">0.00</cbc:AllowanceTotalAmount>';
   }
   
   $xml .= '
       <cbc:PayableAmount currencyID="' . $data['currency'] . '">' . number_format($data['total'], 2, '.', '') . '</cbc:PayableAmount>
   </cac:LegalMonetaryTotal>';
   
   // Invoice lines
   $line_number = 1;
   foreach ($data['items'] as $item) {
       // Unit code - use from item if valid, otherwise use default from settings
       $default_unit_code = isset($settings['default_unit_code']) ? $settings['default_unit_code'] : 'C62';
       $unit_code = isset($item['unit_code']) && !empty($item['unit_code']) ? $item['unit_code'] : $default_unit_code;
       
       // Check if unit code is valid
       $valid_unit_codes = array('B32', 'C62', 'CCT', 'PR', 'D30', 'D40', 'GFI', 'GRM', 'GT', 'CEN', 'KPO', 'MND', '3I', 'KFO', 'KGM', 'KHY', 'KMA', 'KNI', 'KPH', 'KSD', 'KSH', 'KUR', 'D32', 'GWH', 'MWH', 'KWH', 'KWT', 'LPA', 'LTR', 'MTK', 'DMK', 'MTQ', 'MTR', 'NCL', 'CTM', 'SM3', 'R9', 'SET', 'T3');
       
       if (!in_array($unit_code, $valid_unit_codes)) {
           $unit_code = $default_unit_code;
       }
       
       $tax_rate = isset($item['tax_rate']) ? $item['tax_rate'] : 0;
       $tax_amount = isset($item['tax_amount']) ? $item['tax_amount'] : 0;
       $line_total = isset($item['total']) ? $item['total'] : 0;
       
       // Exception code kontrolü - istisna ürünlerde vergi %0 olmalı
       if (isset($item['exception_code']) && !empty($item['exception_code'])) {
           $tax_rate = 0;
           $tax_amount = 0;
       }
       
       $xml .= '
   <cac:InvoiceLine>
       <cbc:ID>' . $line_number . '</cbc:ID>
       <cbc:InvoicedQuantity unitCode="' . $unit_code . '">' . $item['quantity'] . '</cbc:InvoicedQuantity>
       <cbc:LineExtensionAmount currencyID="' . $data['currency'] . '">' . number_format($line_total, 2, '.', '') . '</cbc:LineExtensionAmount>
       <cac:TaxTotal>
           <cbc:TaxAmount currencyID="' . $data['currency'] . '">' . number_format($tax_amount, 2, '.', '') . '</cbc:TaxAmount>
           <cac:TaxSubtotal>
               <cbc:TaxableAmount currencyID="' . $data['currency'] . '">' . number_format($line_total, 2, '.', '') . '</cbc:TaxableAmount>
               <cbc:TaxAmount currencyID="' . $data['currency'] . '">' . number_format($tax_amount, 2, '.', '') . '</cbc:TaxAmount>
               <cbc:CalculationSequenceNumeric>1</cbc:CalculationSequenceNumeric>
               <cbc:Percent>' . number_format($tax_rate, 0, '.', '') . '</cbc:Percent>
               <cac:TaxCategory>';
               
       // Add exception info to line if applicable - REQUIRED for zero-tax items
       // This is mandatory by EDM API for line items with 0% tax rate
       // Check for exception code in item (for mixed invoices) OR global ISTISNA
       if ($tax_rate == 0 && (isset($item['exception_code']) && !empty($item['exception_code']) || $invoice_type_code === 'ISTISNA')) {
			$exception_code = !empty($item['exception_code']) ? $item['exception_code'] : 
							 (isset($data['exception_code']) ? $data['exception_code'] : 
							 (isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : EDM_EFatura_Premium_Exception_Codes::get_default_exception_code()));
			
			$exception_description = isset($exception_descriptions[$exception_code]) ? 
								  $exception_descriptions[$exception_code] : 
								  'Basılı Kitap ve Süreli Yayınların Teslimleri';
			
			$xml .= '
					<cbc:TaxExemptionReasonCode>' . $exception_code . '</cbc:TaxExemptionReasonCode>
					<cbc:TaxExemptionReason>' . $exception_description . '</cbc:TaxExemptionReason>';
		}
               
       $xml .= '
                   <cac:TaxScheme>
                       <cbc:Name>KDV</cbc:Name>
                       <cbc:TaxTypeCode>0015</cbc:TaxTypeCode>
                   </cac:TaxScheme>
               </cac:TaxCategory>
           </cac:TaxSubtotal>
       </cac:TaxTotal>
       <cac:Item>
           <cbc:Name>' . htmlspecialchars($item['name']) . '</cbc:Name>';
           
       // Include SKU if available
       if (!empty($item['sku'])) {
           $xml .= '
           <cac:SellersItemIdentification>
               <cbc:ID>' . htmlspecialchars($item['sku']) . '</cbc:ID>
           </cac:SellersItemIdentification>';
       }
       
       $xml .= '
       </cac:Item>
       <cac:Price>
           <cbc:PriceAmount currencyID="' . $data['currency'] . '">' . number_format($item['unit_price'], 2, '.', '') . '</cbc:PriceAmount>
       </cac:Price>';
       
       
       $xml .= '
   </cac:InvoiceLine>';
       
       $line_number++;
   }
   
   $xml .= '
</Invoice>';
   
   return $xml;
}

/**
 * Sayıyı yazıya dönüştürür
 *
 * @param float $number Sayı
 * @param string $currency Para birimi
 * @return string Yazıya dönüştürülmüş metin
 */
private function convert_number_to_words($number, $currency = 'TRY') {
    $number = number_format($number, 2, '.', '');
    $parts = explode('.', $number);
    $whole = $parts[0];
    $fraction = isset($parts[1]) ? $parts[1] : '00';
    
    // Birler basamağı için kelimeler
    $words = array(
        0 => '', 1 => 'Bir', 2 => 'İki', 3 => 'Üç', 4 => 'Dört', 
        5 => 'Beş', 6 => 'Altı', 7 => 'Yedi', 8 => 'Sekiz', 9 => 'Dokuz'
    );
    
    // Onlar basamağı için kelimeler
    $tens = array(
        0 => '', 1 => 'On', 2 => 'Yirmi', 3 => 'Otuz', 4 => 'Kırk', 
        5 => 'Elli', 6 => 'Altmış', 7 => 'Yetmiş', 8 => 'Seksen', 9 => 'Doksan'
    );
    
    // Para birimleri
    $currency_names = array(
        'TRY' => 'TL',
        'USD' => 'Dolar',
        'EUR' => 'Euro',
        'GBP' => 'Sterlin'
    );
    
    // Para birimi adı
    $currency_name = isset($currency_names[$currency]) ? $currency_names[$currency] : $currency;
    
    // Tam kısmını yazıya çevir
    $result = '';
    if ($whole == 0) {
        $result = 'Sıfır';
    } else {
        // Milyar
        if ($whole >= 1000000000) {
            $billions = floor($whole / 1000000000);
            $result .= $this->convert_less_than_thousand($billions) . ' Milyar ';
            $whole = $whole % 1000000000;
        }
        
        // Milyon
        if ($whole >= 1000000) {
            $millions = floor($whole / 1000000);
            $result .= $this->convert_less_than_thousand($millions) . ' Milyon ';
            $whole = $whole % 1000000;
        }
        
        // Bin
        if ($whole >= 1000) {
            $thousands = floor($whole / 1000);
            // Türkçe'de "Bir Bin" değil, "Bin" denir
            if ($thousands == 1) {
                $result .= 'Bin ';
            } else {
                $result .= $this->convert_less_than_thousand($thousands) . ' Bin ';
            }
            $whole = $whole % 1000;
        }
        
        // Yüz ve altı
        $result .= $this->convert_less_than_thousand($whole);
    }
    
    // Para birimi ekle
    $result .= ' ' . $currency_name;
    
    // Kuruş kısmını ekle
    if ($fraction > 0) {
        $result .= ' ' . $this->convert_less_than_thousand(intval($fraction)) . ' Kuruş';
    }
    
    return $result;
}

/**
 * 1000'den küçük sayıyı yazıya dönüştürür
 * 
 * @param int $number Sayı
 * @return string Yazıya dönüştürülmüş metin
 */
private function convert_less_than_thousand($number) {
    $words = array(
        0 => '', 1 => 'Bir', 2 => 'İki', 3 => 'Üç', 4 => 'Dört', 
        5 => 'Beş', 6 => 'Altı', 7 => 'Yedi', 8 => 'Sekiz', 9 => 'Dokuz'
    );
    
    $tens = array(
        0 => '', 1 => 'On', 2 => 'Yirmi', 3 => 'Otuz', 4 => 'Kırk', 
        5 => 'Elli', 6 => 'Altmış', 7 => 'Yetmiş', 8 => 'Seksen', 9 => 'Doksan'
    );
    
    $result = '';
    
    // Yüzler basamağı
    if ($number >= 100) {
        $hundreds = floor($number / 100);
        // Türkçe'de "Bir Yüz" değil, "Yüz" denir
        if ($hundreds == 1) {
            $result .= 'Yüz ';
        } else {
            $result .= $words[$hundreds] . ' Yüz ';
        }
        $number = $number % 100;
    }
    
    // Onlar basamağı
    if ($number >= 10) {
        $result .= $tens[floor($number / 10)] . ' ';
        $number = $number % 10;
    }
    
    // Birler basamağı
    if ($number > 0) {
        $result .= $words[$number] . ' ';
    }
    
    return trim($result);
}
    
    /**
     * Update order meta with invoice information
     *
     * @param WC_Order $order Order object
     * @param string $invoice_id Invoice ID
     * @param string $uuid Invoice UUID
     */
    private function update_order_invoice_meta($order, $invoice_id, $uuid) {
        // Check if HPOS is enabled
        if (class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
            \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()) {
            // Update using HPOS
            $order->update_meta_data('edm_invoice_id', $invoice_id);
            $order->update_meta_data('edm_invoice_uuid', $uuid);
            $order->update_meta_data('edm_invoice_created', current_time('mysql'));
            $order->save();
        } else {
            // Update using legacy method
            update_post_meta($order->get_id(), 'edm_invoice_id', $invoice_id);
            update_post_meta($order->get_id(), 'edm_invoice_uuid', $uuid);
            update_post_meta($order->get_id(), 'edm_invoice_created', current_time('mysql'));
        }
    }
    
/**
 * Generate invoice PDF
 *
 * @param array $invoice Invoice data
 * @return string|bool PDF file path on success, false on failure
 */
public function generate_invoice_pdf($invoice) {
    
    // Basic logging
    $logger = new EDM_Efatura_Premium_Logger();
    $invoice_id_for_log = isset($invoice['invoice_id']) ? $invoice['invoice_id'] : (isset($invoice['id']) ? $invoice['id'] : 'unknown');
    $logger->debug('Generate Invoice PDF called', array('invoice_id' => $invoice_id_for_log));
    
    try {
        // Create directory
        $upload_dir = wp_upload_dir();
        $pdf_dir = $upload_dir['basedir'] . '/edm-invoices';
        
        if (!file_exists($pdf_dir)) {
            wp_mkdir_p($pdf_dir);
            
            // Prevent directory listing with index.php
            file_put_contents($pdf_dir . '/index.php', '<?php // Silence is golden');
            
            // Restrict direct access with .htaccess
            file_put_contents($pdf_dir . '/.htaccess', 'deny from all');
        }
        
        // Make sure we're using the right invoice_id
        $invoice_id = !empty($invoice['invoice_id']) ? $invoice['invoice_id'] : (!empty($invoice['id']) ? $invoice['id'] : 'unknown');
        
        // Generate filename
        $filename = $pdf_dir . '/fatura-' . $invoice_id . '-' . $invoice['uuid'] . '.pdf';
        
        // Load DOMPDF
        if (!class_exists('Dompdf\Dompdf')) {
            // If installed via Composer
            if (file_exists(EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'vendor/autoload.php')) {
                require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'vendor/autoload.php';
            } 
            // If installed directly to lib folder
            elseif (file_exists(EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'lib/dompdf/autoload.inc.php')) {
                require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'lib/dompdf/autoload.inc.php';
            }
            // Legacy dompdf version
            elseif (file_exists(EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'lib/dompdf/dompdf_config.inc.php')) {
                require_once EDM_EFATURA_PREMIUM_PLUGIN_DIR . 'lib/dompdf/dompdf_config.inc.php';
            }
            else {
                throw new Exception('DOMPDF library not found.');
            }
        }
        
        // Extract invoice data - use correct fields
        $invoice_uuid = isset($invoice['uuid']) ? $invoice['uuid'] : '';
        $customer_name = isset($invoice['customer_name']) ? $invoice['customer_name'] : '';
        $customer_vkn = isset($invoice['customer_vkn']) ? $invoice['customer_vkn'] : '';
        $date = isset($invoice['created_at']) ? date_i18n(get_option('date_format'), strtotime($invoice['created_at'])) : date_i18n(get_option('date_format'));
        $time = isset($invoice['created_at']) ? date_i18n(get_option('time_format'), strtotime($invoice['created_at'])) : date_i18n(get_option('time_format'));
        $amount = isset($invoice['amount']) ? $invoice['amount'] : 0;
        $currency = isset($invoice['currency']) ? $invoice['currency'] : 'TRY';
        
        // Get order object
        $order = false;
        if (!empty($invoice['order_id'])) {
            $order = wc_get_order($invoice['order_id']);
        }
        
        // Get invoice items from database
        $invoice_db_id = isset($invoice['db_id']) ? $invoice['db_id'] : (isset($invoice['id']) ? $invoice['id'] : 0);
        $db = new EDM_Efatura_Premium_DB();
        $items = $db->get_invoice_items($invoice_db_id);
        
        // Get settings
        $settings = get_option('edm_efatura_premium_settings', array());
        
        // Get company information from settings
        $company_name = isset($settings['company_name']) ? $settings['company_name'] : get_bloginfo('name');
        $company_vkn = isset($settings['company_vkn']) ? $settings['company_vkn'] : '';
        $company_tax_office = isset($settings['company_tax_office']) ? $settings['company_tax_office'] : '';
        
        // Get invoice type - prioritize order meta over invoice extra_data over global settings
        $invoice_type_code = '';
        if ($order) {
            // HPOS compatibility
            $hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
                \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
            
            if ($hpos_enabled) {
                $invoice_type_code = $order->get_meta('edm_invoice_type_code', true);
            } else {
                $invoice_type_code = get_post_meta($invoice['order_id'], 'edm_invoice_type_code', true);
            }
        }
        
        // Fallback to invoice extra data or global settings
        if (empty($invoice_type_code)) {
            $invoice_type_code = isset($invoice['extra_data']['invoice_type']) ? $invoice['extra_data']['invoice_type'] : 
                               (isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS');
        }
        
        $logger->info('PDF Generation using invoice type', array(
            'order_id' => $invoice['order_id'] ?? 'unknown',
            'invoice_type_code' => $invoice_type_code,
            'source' => $order && !empty($order->get_meta('edm_invoice_type_code', true)) ? 'order_meta' : 'fallback'
        ));
        
        // Check if prices include tax and get default tax rate 
        $prices_include_tax = isset($settings['woocommerce_prices_include_tax']) ? $settings['woocommerce_prices_include_tax'] : 'yes';
        
        // Get product tax rate based on invoice type
        if ($invoice_type_code === 'ISTISNA') {
            // ISTISNA invoices: products are tax-exempt
            $product_tax_rate = 0;
        } else {
            // SATIS invoices: check for custom tax rate first - HIGHEST PRIORITY
            $custom_tax_rate = $order->get_meta('_edm_custom_tax_rate');
            
            if (!empty($custom_tax_rate) && is_numeric($custom_tax_rate) && $custom_tax_rate > 0) {
                $product_tax_rate = floatval($custom_tax_rate);
            } else {
                $product_tax_rate = isset($settings['default_tax_rate']) ? floatval($settings['default_tax_rate']) : 20;
            }
            
            if ($product_tax_rate <= 0) {
                $product_tax_rate = 20; // Fallback to 20% default
            }
        }
        
        // Cargo ALWAYS has 20% tax regardless of invoice type
        $cargo_tax_rate = 20;
        
        // Build company address
        $company_address = '';
        
        if (isset($settings['company_street']) && !empty($settings['company_street'])) {
            $company_address .= 'Mahalle/Semt:' . $settings['company_street'] . ' ';
        }
        
        if (isset($settings['company_building']) && !empty($settings['company_building'])) {
            $company_address .= 'Cadde/Sokak:' . $settings['company_building'];
        }
        
        $company_address .= '<br/>';
        
        if (isset($settings['company_postal']) && !empty($settings['company_postal'])) {
            $company_address .= $settings['company_postal'] . ' ';
        }
        
        if (isset($settings['company_city']) && !empty($settings['company_city'])) {
            $company_address .= $settings['company_city'] . '/ ';
        }
        
        if (isset($settings['company_district']) && !empty($settings['company_district'])) {
            $company_address .= $settings['company_district'];
        }
        
        // Get company logo
        $logo_url = '';
        if (isset($settings['company_logo']) && !empty($settings['company_logo'])) {
            $logo_id = $settings['company_logo'];
            $logo_url = wp_get_attachment_url($logo_id);
        }
        
        // Get invoice profile from order meta first, then global settings
        $invoice_profile = '';
        if ($order) {
            if ($hpos_enabled) {
                $invoice_profile = $order->get_meta('edm_invoice_profile', true);
            } else {
                $invoice_profile = get_post_meta($order->get_id(), 'edm_invoice_profile', true);
            }
        }
        
        // Fallback to global settings if not set in order meta
        if (empty($invoice_profile)) {
            $invoice_profile = isset($settings['invoice_profile']) ? $settings['invoice_profile'] : 'EARSIVFATURA';
        }
        
        // Get exception code and description - only for ISTISNA type
        $exception_code = '';
        $exception_desc = '';
        
        if ($invoice_type_code === 'ISTISNA') {
            // Prioritize order meta over global settings for exception code
            if ($order) {
                if ($hpos_enabled) {
                    $exception_code = $order->get_meta('edm_exception_code', true);
                } else {
                    $exception_code = get_post_meta($invoice['order_id'], 'edm_exception_code', true);
                }
            }
            
            // Fallback to global settings
            if (empty($exception_code)) {
                $exception_code = isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : '';
            }
            
            // Get exception code and description
            $exception_descriptions = EDM_EFatura_Premium_Exception_Codes::get_turkish_exception_descriptions();
            
            if (!empty($exception_code)) {
                $exception_desc = isset($exception_descriptions[$exception_code]) ? $exception_descriptions[$exception_code] : 'Tax Exemption';
            }
        }
        
        // Format customer data (for PDF, keep HTML formatting)
        $customer_address = '';
        if ($order) {
            $customer_address = $order->get_formatted_billing_address();
            // For PDF generation, we want HTML breaks, so keep them as is
            // The address is already formatted with <br/> tags from WooCommerce
        }
        
        // Calculate totals and process items - Kargo hesaplamasını EDM API ile aynı yapıyoruz
        $subtotal = 0;
        $shipping_total = 0;
        $tax_total = 0;
        $shipping_tax = 0;
        $discount_total = 0;
        $total = $amount; // Amount from the invoice record
        $processed_items = array();
        $has_shipping_item = false;
        
        // REMOVED: WooCommerce order'dan shipping ekleme kısmı kaldırıldı
        // Çünkü database'den gelen items zaten shipping içeriyor
        // Bu duplication'ın ana kaynağıydı
        
        // KARMA FATURA LOGİĞİ PDF için: Kategori ayarlarına göre fatura tipi belirleme
        $has_category_taxed_items = false;
        $has_category_exception_items = false;
        $has_normal_vat_items = false;
        
        // Önce tüm items'ı kontrol et - kategori ayarları var mı?
        if (is_array($items) && !empty($items)) {
            foreach ($items as $item) {
                if (isset($item['product_id']) && !empty($item['product_id'])) {
                    $product = wc_get_product($item['product_id']);
                    if ($product) {
                        $has_category_setting = false;
                        $category_ids = $product->get_category_ids();
                        
                        foreach ($category_ids as $category_id) {
                            $tax_type = get_term_meta($category_id, 'edm_tax_type', true);
                            if (!empty($tax_type)) {
                                $has_category_setting = true;
                                if ($tax_type === 'exception') {
                                    $has_category_exception_items = true;
                                } elseif ($tax_type === 'taxed') {
                                    $has_category_taxed_items = true;
                                }
                                break; // İlk kategori ayarını kullan
                            }
                        }
                        
                        // Kategori ayarı olmayan ürünler için global ayara bak
                        if (!$has_category_setting && $invoice_type_code === 'SATIS') {
                            $has_normal_vat_items = true;
                        }
                    }
                }
            }
        }
        
        // KARMA FATURA KURAL: Herhangi bir KDV'li item varsa SATIS, hepsi istisna ise ISTISNA
        $original_invoice_type = $invoice_type_code;
        
        if ($has_category_taxed_items || $has_normal_vat_items) {
            $invoice_type_code = 'SATIS';
        } elseif ($has_category_exception_items && !$has_category_taxed_items && !$has_normal_vat_items) {
            $invoice_type_code = 'ISTISNA';
        }
        
        // Veritabanından gelen items'ı işle
        
        if (is_array($items) && !empty($items)) {
            foreach ($items as $item) {
                $processed_item = $item;
                
                // Exception code kontrolü ve item name'e ekleme (PDF için)
                if (isset($item['exception_code']) && !empty($item['exception_code'])) {
                    $exception_code = $item['exception_code'];
                    $original_name = isset($item['name']) ? $item['name'] : '';
                    $processed_item['name'] = $original_name . ' (İstisna Kodu: ' . $exception_code . ')';
                }
                
                // Quantity ve Unit Price doğru değerleri al
                $quantity = isset($item['quantity']) ? floatval($item['quantity']) : 0;
                $unit_price = isset($item['unit_price']) ? floatval($item['unit_price']) : 0;
                
                // Toplam tutarı hesapla
                $line_subtotal = $quantity * $unit_price;
                
                // Eğer total alanı varsa, onu tercih et
                $total_amount = isset($item['total_amount']) ? floatval($item['total_amount']) : 
                              (isset($item['total']) ? floatval($item['total']) : $line_subtotal);
                
                // Kargo öğesi olup olmadığını kontrol et
                $is_shipping = false;
                if (isset($item['is_shipping']) && $item['is_shipping']) {
                    $is_shipping = true;
                } else if (isset($item['sku']) && $item['sku'] === 'SHIPPING') {
                    $is_shipping = true;
                } else if (!empty($item['name'])) {
                    $name_lower = strtolower($item['name']);
                    if (strpos($name_lower, 'shipping') !== false || strpos($name_lower, 'kargo') !== false) {
                        $is_shipping = true;
                    }
                }
                
                // FIX: Her öğe için vergi bilgilerini doğru şekilde al
                $tax_rate = isset($item['tax_rate']) ? floatval($item['tax_rate']) : 0;
                $tax_amount = isset($item['tax_amount']) ? floatval($item['tax_amount']) : 0;

                // Calculate and process based on item type
                if ($is_shipping) {
                    // Kargo öğesi - ALWAYS 20% tax regardless of invoice type
                    $shipping_tax_rate = $cargo_tax_rate;
                    
                    // Vergi hesaplamalarını doğru şekilde yap
                    if ($prices_include_tax === 'yes') {
                        // Fiyatlar vergi dahil - vergiyi çıkar
                        $shipping_price_without_tax = $unit_price / (1 + ($shipping_tax_rate / 100));
                        $shipping_total_without_tax = $shipping_price_without_tax * $quantity;
                        $shipping_tax_amount = $total_amount - $shipping_total_without_tax;
                        
                        $processed_item['unit_price'] = $shipping_price_without_tax;
                        $processed_item['total'] = $shipping_total_without_tax;
                        $processed_item['tax_rate'] = $shipping_tax_rate;
                        $processed_item['tax_amount'] = $shipping_tax_amount;
                        $processed_item['total_with_tax'] = $total_amount;
                    } else {
                        // Fiyatlar vergi hariç - vergiyi ekle
                        $shipping_tax_amount = $total_amount * ($shipping_tax_rate / 100);
                        
                        $processed_item['unit_price'] = $unit_price;
                        $processed_item['total'] = $total_amount;
                        $processed_item['tax_rate'] = $shipping_tax_rate;
                        $processed_item['tax_amount'] = $shipping_tax_amount;
                        $processed_item['total_with_tax'] = $total_amount + $shipping_tax_amount;
                    }
                    
                    // Toplam değerlere ekle
                    $shipping_total += $processed_item['total'];
                    $shipping_tax += $processed_item['tax_amount'];
                } else {
                    // Normal ürün öğesi
                    // Check for category tax settings first (HIGHEST PRIORITY for products)
                    $product_tax_rate_for_item = $product_tax_rate; // Default fallback
                    $item_is_exception = false;
                    
                    // PDF için: product_id'den ürün objesini al
                    $product = null;
                    if (isset($item['product_id']) && !empty($item['product_id'])) {
                        $product = wc_get_product($item['product_id']);
                    }
                    
                    
                    if ($product) {
                        // Direct category tax settings check (for PDF)
                        $category_ids = $product->get_category_ids();
                        
                        foreach ($category_ids as $category_id) {
                            $tax_type = get_term_meta($category_id, 'edm_tax_type', true);
                            
                            if (!empty($tax_type)) {
                                if ($tax_type === 'exception') {
                                    // Category has exception setting - override invoice type logic
                                    $item_is_exception = true;
                                    $product_tax_rate_for_item = 0;
                                } elseif ($tax_type === 'taxed') {
                                    // Category has custom tax rate - use it regardless of invoice type
                                    $custom_tax_rate = floatval(get_term_meta($category_id, 'edm_tax_rate', true));
                                    if ($custom_tax_rate > 0) {
                                        $product_tax_rate_for_item = $custom_tax_rate;
                                    }
                                }
                                break; // Use first category with settings
                            }
                        }
                        
                        // PDF için Global ISTISNA kontrolü (kategori ayarı yoksa)
                        if (!$item_is_exception && $product_tax_rate_for_item == $default_tax_rate) {
                            // Kategori ayarı yok, global ayara bak
                            $settings = get_option('edm_efatura_premium_settings', array());
                            $global_invoice_type = isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS';
                            
                            if ($global_invoice_type === 'ISTISNA') {
                                $item_is_exception = true;
                                $product_tax_rate_for_item = 0;
                            }
                        }
                    }
                    
                    
                    // Apply tax logic - CATEGORY SETTINGS HAVE HIGHEST PRIORITY
                    $has_category_setting = ($item_is_exception || $product_tax_rate_for_item != $default_tax_rate);
                    
                    if ($item_is_exception) {
                        // Kategori istisna ayarı - her zaman %0 vergi
                        $processed_item['tax_rate'] = 0;
                        $processed_item['tax_amount'] = 0;
                        $processed_item['unit_price'] = $unit_price;
                        $processed_item['total'] = $unit_price * $quantity;
                        $processed_item['total_with_tax'] = $processed_item['total'];
                    } elseif ($has_category_setting) {
                        // Kategori özel vergi ayarı var
                        if ($tax_rate <= 0) {
                            $tax_rate = $product_tax_rate_for_item;
                        } else {
                            // If item already has a tax rate from WooCommerce but we have category override
                            $tax_rate = $product_tax_rate_for_item;
                        }
                        
                        // Vergi hesaplamalarını doğru şekilde yap
                        if ($prices_include_tax === 'yes') {
                            // Fiyatlar vergi dahil - vergiyi çıkar
                            $price_without_tax = $unit_price / (1 + ($tax_rate / 100));
                            $total_without_tax = $price_without_tax * $quantity;
                            $tax_amount = $total_amount - $total_without_tax;
                            
                            $processed_item['unit_price'] = $price_without_tax;
                            $processed_item['total'] = $total_without_tax;
                            $processed_item['tax_rate'] = $tax_rate;
                            $processed_item['tax_amount'] = $tax_amount;
                            $processed_item['total_with_tax'] = $total_amount;
                        } else {
                            // Fiyatlar vergi hariç - vergiyi ekle
                            $tax_amount = $total_amount * ($tax_rate / 100);
                            
                            $processed_item['unit_price'] = $unit_price;
                            $processed_item['total'] = $total_amount;
                            $processed_item['tax_rate'] = $tax_rate;
                            $processed_item['tax_amount'] = $tax_amount;
                            $processed_item['total_with_tax'] = $total_amount + $tax_amount;
                        }
                    } else {
                        // Kategori ayarı yok - global ayara bak
                        $settings = get_option('edm_efatura_premium_settings', array());
                        $global_invoice_type = isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS';
                        
                        if ($global_invoice_type === 'ISTISNA') {
                            // Global ISTISNA ayarı - istisna item
                            $processed_item['tax_rate'] = 0;
                            $processed_item['tax_amount'] = 0;
                            $processed_item['unit_price'] = $unit_price;
                            $processed_item['total'] = $unit_price * $quantity;
                            $processed_item['total_with_tax'] = $processed_item['total'];
                            
                            // Exception code ekleme
                            $global_exception_code = isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : '';
                            if (!empty($global_exception_code)) {
                                $original_name = isset($processed_item['name']) ? $processed_item['name'] : '';
                                $processed_item['name'] = $original_name . ' (İstisna Kodu: ' . $global_exception_code . ')';
                            }
                        } else {
                            // Global SATIS ayarı - normal vergi
                            if ($tax_rate <= 0) {
                                $tax_rate = $default_tax_rate;
                            }
                            
                            // Vergi hesaplamalarını doğru şekilde yap
                            if ($prices_include_tax === 'yes') {
                                // Fiyatlar vergi dahil - vergiyi çıkar
                                $price_without_tax = $unit_price / (1 + ($tax_rate / 100));
                                $total_without_tax = $price_without_tax * $quantity;
                                $tax_amount = $total_amount - $total_without_tax;
                                
                                $processed_item['unit_price'] = $price_without_tax;
                                $processed_item['total'] = $total_without_tax;
                                $processed_item['tax_rate'] = $tax_rate;
                                $processed_item['tax_amount'] = $tax_amount;
                                $processed_item['total_with_tax'] = $total_amount;
                            } else {
                                // Fiyatlar vergi hariç - vergiyi ekle
                                $tax_amount = $total_amount * ($tax_rate / 100);
                                
                                $processed_item['unit_price'] = $unit_price;
                                $processed_item['total'] = $total_amount;
                                $processed_item['tax_rate'] = $tax_rate;
                                $processed_item['tax_amount'] = $tax_amount;
                                $processed_item['total_with_tax'] = $total_amount + $tax_amount;
                            }
                        }
                    }
                    
                    // FIX: Toplam değerlere ekleme
                    $subtotal += $processed_item['total'];
                    $tax_total += $processed_item['tax_amount'];
                    
                }
                
                $processed_item['is_shipping'] = $is_shipping;
                $processed_items[] = $processed_item;
            }
        }
        
        // Kargo öğesini items listesine ekle (eğer varsa)
        if ($has_shipping_item && isset($shipping_item)) {
            $processed_items[] = $shipping_item;
        }
        
        // Eğer veritabanında hiç item yoksa, siparişten oluştur
        
        if (empty($processed_items) && $order) {
            // Eğer veritabanında kaydedilmiş öğe yoksa, siparişten oluştur
            foreach ($order->get_items() as $item) {
                $product = $item->get_product();
                $tax_rate = 0;
                $tax_amount = 0;
                
                // AJAX PDF Kategori vergi ayarlarını kontrol et (HIGHEST PRIORITY)
                $product_tax_rate_for_item = $default_tax_rate; // Default fallback
                $item_is_exception = false;
                
                if ($product) {
                    // Kategori vergi ayarlarını kontrol et
                    $category_ids = $product->get_category_ids();
                    
                    foreach ($category_ids as $category_id) {
                        $tax_type = get_term_meta($category_id, 'edm_tax_type', true);
                        
                        if (!empty($tax_type)) {
                            if ($tax_type === 'exception') {
                                // Kategori istisna ayarı - invoice type'ı override et
                                $item_is_exception = true;
                                $product_tax_rate_for_item = 0;
                            } elseif ($tax_type === 'taxed') {
                                // Kategori özel vergi oranı - invoice type'ı ignore et
                                $custom_tax_rate = floatval(get_term_meta($category_id, 'edm_tax_rate', true));
                                if ($custom_tax_rate > 0) {
                                    $product_tax_rate_for_item = $custom_tax_rate;
                                }
                            }
                            break; // İlk kategori ayarını kullan
                        }
                    }
                }
                
                // Vergi oranını ve miktarını hesapla
                if (wc_tax_enabled()) {
                    $tax_data = $item->get_taxes();
                    if (!empty($tax_data['total'])) {
                        $tax_amount = array_sum($tax_data['total']);
                        if ($item->get_subtotal() > 0) {
                            $tax_rate = round(($tax_amount / $item->get_subtotal()) * 100, 2);
                        }
                    }
                }
                
                // Miktar ve birim fiyat
                $quantity = max(1, $item->get_quantity());
                $unit_price = $item->get_subtotal() / $quantity;
                
                // Kategori ayarlarına ve invoice type'a göre vergi logic'ini uygula
                // İstisna (ISTISNA) fatura tiplerinde ürünlerin vergisi 0 olur
                if ($item_is_exception || $invoice_type_code === 'ISTISNA') {
                    $tax_rate = 0;
                    $tax_amount = 0;
                    $total_without_tax = $unit_price * $quantity;
                    $total_with_tax = $total_without_tax;
                } else {
                    // SATIS fatura tipleri için kategori vergi oranını veya varsayılanı kullan
                    if ($tax_rate <= 0) {
                        $tax_rate = $product_tax_rate_for_item;
                    } else {
                        // Eğer WooCommerce'dan gelen vergi oranı varsa ama kategori override varsa
                        $tax_rate = $product_tax_rate_for_item;
                    }
                    
                    if ($prices_include_tax === 'yes') {
                        // Fiyatlar vergi dahil - vergiyi çıkar
                        $price_without_tax = $unit_price / (1 + ($tax_rate / 100));
                        $total_without_tax = $price_without_tax * $quantity;
                        $tax_amount = ($unit_price * $quantity) - $total_without_tax;
                    } else {
                        // Fiyatlar vergi hariç - vergiyi ekle
                        $price_without_tax = $unit_price;
                        $total_without_tax = $price_without_tax * $quantity;
                        $tax_amount = $total_without_tax * ($tax_rate / 100);
                    }
                    
                    $total_with_tax = $total_without_tax + $tax_amount;
                }
                
                $processed_item = array(
                    'id' => $product ? $product->get_id() : null,
                    'name' => $item->get_name(),
                    'quantity' => $quantity,
                    'unit_price' => $price_without_tax,
                    'tax_rate' => $tax_rate,
                    'tax_amount' => $tax_amount,
                    'total' => $total_without_tax,
                    'total_with_tax' => $total_with_tax,
                    'is_shipping' => false
                );
                
                $processed_items[] = $processed_item;
                
                // Toplam değerlere ekle
                $subtotal += $total_without_tax;
                $tax_total += $tax_amount;
            }
            
            // Kargo zaten yukarıda hesaplandı, tekrar eklemeye gerek yok
        }
        
        // İşlenmiş öğeleri elementlere aktar
        $items = $processed_items;
        
        // İndirim miktarını al
        if ($order) {
            $discount_total = $order->get_discount_total();
        }
        
        // FIX: Nihai tutarları doğru şekilde hesapla
        // Subtotal tüm satırları (ürün + kargo) içermeli
        $grand_subtotal = $subtotal; // Kargo dahil toplam (doğru)
        $grand_tax_total = $tax_total + $shipping_tax;
        $grand_total = $grand_subtotal + $grand_tax_total - $discount_total;
        
        // Diğer bağlam verileri
        $order_number = '';
        $payment_method = '';
        $payment_date = '';
        $shipping_date = '';
        $shipping_company = '';
        $web_address = site_url();
        
        if ($order) {
            $order_number = $order->get_order_number();
            $payment_gateway = $order->get_payment_method_title();
            
            // Ödeme yöntemini ödeme geçidine göre ayarla
            if (!empty($payment_gateway)) {
                if (strpos(strtolower($payment_gateway), 'kredi') !== false || 
                    strpos(strtolower($payment_gateway), 'kart') !== false) {
                    $payment_method = 'Kredi Kartı / Banka Kartı';
                } elseif (strpos(strtolower($payment_gateway), 'havale') !== false || 
                        strpos(strtolower($payment_gateway), 'eft') !== false) {
                    $payment_method = 'HAVALE/EFT';
                } elseif (strpos(strtolower($payment_gateway), 'kapı') !== false || 
                        strpos(strtolower($payment_gateway), 'nakit') !== false) {
                    $payment_method = 'KAPIDA ODEME';
                } else {
                    $payment_method = $payment_gateway;
                }
            }
            
            // Ödeme tarihi
            if ($order->get_date_paid()) {
                $payment_date = date_i18n('Y-m-d', $order->get_date_paid()->getTimestamp());
            } elseif ($order->get_date_created()) {
                $payment_date = date_i18n('Y-m-d', $order->get_date_created()->getTimestamp());
            }
            
            // Gönderim tarihi (varsayılan: sipariş tarihi + 1 gün)
            if ($order->get_date_created()) {
                $shipping_date = date_i18n('Y-m-d', strtotime('+1 day', $order->get_date_created()->getTimestamp()));
            }
        }
        
        // Özel ödeme aracısı
        $payment_intermediary = isset($settings['payment_intermediary']) ? $settings['payment_intermediary'] : 'sanal';
        
        // Özel kargo şirketi adı
        $shipping_company = isset($settings['shipping_company_name']) ? $settings['shipping_company_name'] : 'Kargo';
        
        // Web adresi
        $web_address = isset($settings['company_website']) && !empty($settings['company_website']) ? 
                      $settings['company_website'] : site_url();
        
        // Collect exception codes from all items for PDF notes (for mixed invoices)
        $all_exception_codes = array();
        $exception_descriptions = EDM_EFatura_Premium_Exception_Codes::get_turkish_exception_descriptions();
        
        // Re-get order items to check for exception codes (since $processed_items might not have exception_code)
        if ($order) {
            $order_items_with_exceptions = $this->get_order_items_with_exception_codes($order, $invoice_type_code);
            if (isset($order_items_with_exceptions['items'])) {
                foreach ($order_items_with_exceptions['items'] as $item) {
                    if (isset($item['exception_code']) && !empty($item['exception_code']) && !in_array($item['exception_code'], $all_exception_codes)) {
                        $all_exception_codes[] = $item['exception_code'];
                    }
                }
            }
        }
        
        // Add global exception code if invoice type is ISTISNA and code exists
        if ($invoice_type_code === 'ISTISNA' && !empty($exception_code) && !in_array($exception_code, $all_exception_codes)) {
            $all_exception_codes[] = $exception_code;
        }
        
        
// Create HTML content for PDF
$html = '<!DOCTYPE html>
<html>
<head>
    <meta http-equiv="Content-Type" content="text/html; charset=utf-8"/>
    <title>' . __('Invoice', 'edm-efatura-premium') . ' #' . $invoice_id . '</title>
    <style>
        body {
            font-family: "DejaVu Sans", sans-serif;
            margin: 0;
            padding: 10px;
            font-size: 7pt;
            color: #000;
            line-height: 1.2;
        }
        table {
            border-collapse: collapse;
            width: 100%;
        }
        .header-section {
            width: 100%;
            margin-bottom: 15px;
            display: table;
        }
        .header-left {
            width: 30%;
            display: table-cell;
            vertical-align: top;
        }
        .header-center {
            width: 40%;
            display: table-cell;
            text-align: center;
            vertical-align: top;
        }
        .header-right {
            width: 30%;
            display: table-cell;
            text-align: right;
            vertical-align: top;
        }
        .logo {
            max-width: 80px;
            max-height: 80px;
            margin-bottom: 5px;
        }
        .company-logo {
            max-width: 100px;
            max-height: 80px;
        }
        .e-arsiv-title {
            text-align: center;
            font-weight: bold;
            font-size: 10pt;
            margin-top: 5px;
        }
        .signature img {
            max-height: 30px;
            margin-top: 5px;
        }
        .info-section {
            width: 100%;
            margin-bottom: 10px;
			margin-top: 100px;
        }
        .info-section-row {
            display: table;
            width: 100%;
        }
        .info-section-col {
            display: table-cell;
            vertical-align: top;
            width: 33.33%;
            padding-right: 10px;
        }
        .info-section-col:last-child {
            padding-right: 0;
        }
        .info-label {
            font-weight: bold;
            display: block;
            margin-bottom: 2px;
        }
        .dotted-line {
            border-bottom: 1px dashed #cccccc;
            height: 1px;
            width: 100%;
            margin: 10px 0;
        }
        .ettn {
            margin: 5px 0;
            font-size: 7pt;
        }
        .items-table {
            margin-top: 10px;
            margin-bottom: 10px;
            width: 100%;
            border-radius: 5px;
            overflow: hidden;
        }
        .items-table th {
            border: 1px solid #000;
            padding: 4px;
            background-color: #f0f0f0;
            font-weight: bold;
            text-align: center;
            font-size: 7pt;
        }
        .items-table td {
            border: 1px solid #000;
            padding: 4px;
            text-align: left;
            font-size: 7pt;
        }
        .items-table tr:first-child th:first-child {
            border-top-left-radius: 5px;
        }
        .items-table tr:first-child th:last-child {
            border-top-right-radius: 5px;
        }
        .items-table tr:last-child td:first-child {
            border-bottom-left-radius: 5px;
        }
        .items-table tr:last-child td:last-child {
            border-bottom-right-radius: 5px;
        }
        .items-table .number-col {
            text-align: center;
            width: 5%;
        }
        .items-table .desc-col {
            width: 35%;
        }
        .items-table .quantity-col {
            text-align: center;
            width: 10%;
        }
        .items-table .price-col, .items-table .tax-col, .items-table .tax-amount-col, .items-table .total-col {
            text-align: right;
            width: 10%;
        }
        .totals-table {
            width: 250px;
            float: right;
            margin-top: 10px;
            border-radius: 5px;
            overflow: hidden;
        }
        .totals-table td {
            border: 1px solid #000;
            padding: 4px;
            font-size: 7pt;
        }
        .totals-label {
            text-align: left;
        }
        .totals-value {
            text-align: right;
        }
        .notes-section {
            border: 1px solid #000;
            padding: 5px;
            margin-top: 10px;
            margin-bottom: 10px;
            font-size: 7pt;
            border-radius: 5px;
        }
        .return-section {
            border: 1px solid #000;
            padding: 5px;
            margin-top: 20px;
            margin-bottom: 10px;
            border-radius: 5px;
            font-size: 7pt;
        }
        .return-left {
            width: 50%;
            float: left;
        }
        .return-right {
            width: 50%;
            float: right;
        }
        .footer {
            margin-top: 20px;
            font-size: 7pt;
            text-align: center;
        }
        .clear {
            clear: both;
        }
    </style>
</head>
<body>
    <!-- Header Section with Company Info, GIB Logo, and Company Logo -->
    <div class="header-section">
        <div class="header-left">
            <strong>' . strtoupper($company_name) . '</strong><br/>
            <strong>ADRES:</strong> ' . $company_address . '<br/>
            <strong>E-POSTA:</strong> ' . (isset($settings['company_email']) ? $settings['company_email'] : '') . '<br/>
            <strong>VD:</strong> ' . $company_tax_office . '<br/>
            <strong>' . (strlen($company_vkn) == 11 ? 'TCKN' : 'VKN') . ':</strong> ' . $company_vkn . '<br/>
            <strong>TICARETSICILNO:</strong> ' . (isset($settings['company_trade_registry']) ? $settings['company_trade_registry'] : '') . '
        </div>
        
        <div class="header-center">
            <img src="' . site_url() . '/wp-content/plugins/edm-efatura-premium/assets/images/gib.png" class="logo" alt="GIB Logo" /><br/>
            <div class="e-arsiv-title">e-Arşiv Fatura</div>';

if (!empty($signature_image)) {
    $html .= '<div class="signature">
                <img src="data:image/png;base64,' . $signature_image . '" alt="Signature" />
            </div>';
}

$html .= '
        </div>
        
        <div class="header-right">';

if (!empty($logo_url)) {
    $html .= '<img src="' . $logo_url . '" class="company-logo" alt="Company Logo" />';
}

$html .= '
        </div>
    </div>
    
    <!-- Customer and Invoice Information Section (3 columns) -->
    <div class="info-section">
        <div class="info-section-row">
            <div class="info-section-col">
                <strong>SAYIN</strong> ' . $customer_name . '<br/>
                <strong>ADRES:</strong> ' . $customer_address . '<br/>
                <strong>E-POSTA:</strong> ' . ($order ? $order->get_billing_email() : '') . '<br/>
                <strong>' . (strlen($customer_vkn) == 11 ? 'TCKN' : 'VKN') . ':</strong> ' . $customer_vkn . '
            </div>
            
            <div class="info-section-col">
                <div class="info-row">
                    <strong>ÖZELLİŞTİRME NO:</strong> TR1.2<br/>
                    <strong>SENARYO:</strong> ' . $invoice_profile . '<br/>
                    <strong>FATURA TİPİ:</strong> ' . $invoice_type_code . '<br/>
                    <strong>FATURA NO:</strong> ' . $invoice_id . '<br/>
                    <strong>FATURA TARİHİ:</strong> ' . $date . '<br/>
                    <strong>FATURA ZAMANI:</strong> ' . $time . '<br/>';
                    
if ($order && !empty($order_number)) {
    $html .= '<strong>SİPARİŞ NO:</strong> ' . $order_number . '<br/>';
}

if ($order && $order->get_date_created()) {
    $html .= '<strong>SİPARİŞ TARİHİ:</strong> ' . date_i18n('d-m-Y', $order->get_date_created()->getTimestamp()) . '<br/>';
}

$html .= '
                </div>
            </div>
            
            <div class="info-section-col" style="text-align: right;">
                <div class="info-row">';
                
if (!empty($web_address)) {
    $html .= '<strong>WEB ADRESİ:</strong> ' . $web_address . '<br/>';
}

if (!empty($payment_method)) {
    $html .= '<strong>ÖDEME ŞEKLİ:</strong> ' . $payment_method . '<br/>';
}

if (!empty($payment_intermediary)) {
    $html .= '<strong>ÖDEME ARACISI:</strong> ' . $payment_intermediary . '<br/>';
}


if (!empty($payment_date)) {
    $html .= '<strong>ÖDEME TARİHİ:</strong> ' . $payment_date . '<br/>';
}

if (!empty($shipping_date)) {
    $html .= '<strong>GÖNDERİM TARİHİ:</strong> ' . $shipping_date . '<br/>';
}

if (!empty($delivery_person)) {
    $html .= '<strong>GÖNDERİ TAŞIYAN VKN:</strong> ' . $delivery_person . '<br/>';
}

if (!empty($shipping_company)) {
    $html .= '<strong>GÖNDERİ TAŞIYAN UNVAN:</strong> ' . $shipping_company . '<br/>';
}

$html .= '
                </div>
            </div>
        </div>
    </div>
    
    <div class="dotted-line"></div>
    
    <div class="ettn"><strong>ETTN:</strong> ' . $invoice_uuid . '</div>
    
    <!-- Invoice Items Table -->
    <table class="items-table">
        <thead>
            <tr>
                <th class="number-col">SIRA</th>
                <th class="desc-col">AÇIKLAMA</th>
                <th class="quantity-col">MİKTAR</th>
                <th class="price-col">BİRİM FİYAT</th>
                <th class="tax-col">KDV ORANI</th>
                <th class="tax-amount-col">KDV TUTARI</th>
                <th class="total-col">TUTAR</th>
            </tr>
        </thead>
        <tbody>';

// FIX: Öğe bilgilerini doğru şekilde görüntüle
if (is_array($items) && !empty($items)) {
    $item_count = 1;
    foreach ($items as $item) {
        $item_name = isset($item['name']) ? $item['name'] : '';
        $quantity = isset($item['quantity']) ? $item['quantity'] : 0;
        $unit_price = isset($item['unit_price']) ? $item['unit_price'] : 0;
        $tax_rate = isset($item['tax_rate']) ? $item['tax_rate'] : 0;
        $tax_amount = isset($item['tax_amount']) ? $item['tax_amount'] : 0;
        $line_total = isset($item['total']) ? $item['total'] : ($unit_price * $quantity);
        $is_shipping = isset($item['is_shipping']) && $item['is_shipping'];
        
        // Kargo için her zaman %20 KDV göster (ISTISNA olsa bile)
        if ($is_shipping) {
            $tax_rate = $cargo_tax_rate; // Always 20% for cargo
        }
        
        // Her kalem için (kargo dahil) tabloya satır ekle
        $html .= '
            <tr>
                <td class="number-col">' . $item_count . '</td>
                <td class="desc-col">' . $item_name . '</td>
                <td class="quantity-col">' . number_format($quantity, 0) . ' Adet</td>
                <td class="price-col">' . number_format($unit_price, 2, ',', '.') . ' ' . $currency . '</td>
                <td class="tax-col">%' . number_format($tax_rate, 2, ',', '.') . '</td>
                <td class="tax-amount-col">' . number_format($tax_amount, 2, ',', '.') . ' ' . $currency . '</td>
                <td class="total-col">' . number_format($line_total, 2, ',', '.') . ' ' . $currency . '</td>
            </tr>';
        
        $item_count++;
    }
}

$html .= '
        </tbody>
    </table>
    
    <!-- Totals Section -->
    <table class="totals-table">
        <tr>
            <td class="totals-label">ÜRÜN TOPLAMI</td>
            <td class="totals-value">' . number_format($grand_subtotal, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';

// FIX: Kargo tutarını ayrı bir satır olarak göster (eğer varsa)
if ($shipping_total > 0) {
    $html .= '
        <tr>
            <td class="totals-label">KARGO ÜCRETİ</td>
            <td class="totals-value">' . number_format($shipping_total, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';
}

// FIX: İndirim varsa göster
if ($discount_total > 0) {
    $html .= '
        <tr>
            <td class="totals-label">TOPLAM İSKONTO</td>
            <td class="totals-value">' . number_format($discount_total, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';
} else {
    $html .= '
        <tr>
            <td class="totals-label">TOPLAM İSKONTO</td>
            <td class="totals-value">0,00 ' . $currency . '</td>
        </tr>';
}

// FIX: KDV gösterimi - ISTISNA fatura tipinde bile kargo KDV'sini göster
if ($invoice_type_code === 'ISTISNA') {
    // ISTISNA tipinde ürün KDV'si 0
    $html .= '
        <tr>
            <td class="totals-label">ÜRÜN KDV (%0)</td>
            <td class="totals-value">0,00 ' . $currency . '</td>
        </tr>';
    
    // Ancak, kargo KDV'si her zaman %20 olarak hesaplanır
    if ($shipping_tax > 0) {
        $html .= '
        <tr>
            <td class="totals-label">KARGO KDV (%' . $default_tax_rate . ')</td>
            <td class="totals-value">' . number_format($shipping_tax, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';
    }
    
    // Toplam KDV (sadece kargo KDV'si olacak)
    $html .= '
        <tr>
            <td class="totals-label">TOPLAM KDV</td>
            <td class="totals-value">' . number_format($shipping_tax, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';
} else {
    // Normal (SATIS) fatura tipinde
    // Ürün KDV'sini göster
    $html .= '
        <tr>
            <td class="totals-label">ÜRÜN KDV</td>
            <td class="totals-value">' . number_format($tax_total, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';
    
    // Kargo KDV'sini göster (eğer varsa)
    if ($shipping_tax > 0) {
        $html .= '
        <tr>
            <td class="totals-label">KARGO KDV</td>
            <td class="totals-value">' . number_format($shipping_tax, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';
    }
    
    // Toplam KDV'yi göster
    $html .= '
        <tr>
            <td class="totals-label">TOPLAM KDV</td>
            <td class="totals-value">' . number_format($grand_tax_total, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>';
}

// FIX: Toplam tutar ve ödenecek tutar
$html .= '
        <tr>
            <td class="totals-label">TOPLAM TUTAR</td>';

// FIX: ISTISNA tipinde toplam tutar farklı hesaplanır            
if ($invoice_type_code === 'ISTISNA') {
    // ISTISNA tipinde: Ürün tutarı + Kargo tutarı (KDV hariç) = TOPLAM TUTAR
    $total_amount_without_shipping_tax = $grand_subtotal + $shipping_total - $discount_total;
    $html .= '<td class="totals-value">' . number_format($total_amount_without_shipping_tax, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>
        <tr>
            <td class="totals-label">ÖDENECEK TUTAR</td>';
    // ÖDENECEK TUTAR'a kargo KDV'si eklenir
    $payable_amount = $total_amount_without_shipping_tax + $shipping_tax;
    $html .= '<td class="totals-value">' . number_format($payable_amount, 2, ',', '.') . ' ' . $currency . '</td>';
} else {
    // Normal fatura tiplerinde toplam tutar = ödenecek tutar
    $html .= '<td class="totals-value">' . number_format($grand_total, 2, ',', '.') . ' ' . $currency . '</td>
        </tr>
        <tr>
            <td class="totals-label">ÖDENECEK TUTAR</td>
            <td class="totals-value">' . number_format($grand_total, 2, ',', '.') . ' ' . $currency . '</td>';
}

$html .= '
        </tr>
    </table>
    
    <div class="clear"></div>
    
    <!-- Notes Section -->
    <div class="notes-section">';

// Show exception codes for any invoice that contains exception items (both ISTISNA and mixed SATIS invoices)
if (!empty($all_exception_codes)) {
    $html .= '<strong>VERGİ İSTİSNA MUAFİYET SEBEBLERİ:</strong><br/>';
    foreach ($all_exception_codes as $exc_code) {
        $exc_description = isset($exception_descriptions[$exc_code]) ? $exception_descriptions[$exc_code] : 'Vergi İstisnası';
        $html .= '• ' . $exc_code . ' - ' . $exc_description . '<br/>';
    }
}

$html .= 'Bu satış internet üzerinden yapılmıştır.<br/>
        E-Arşiv İzni Kapsamında Elektronik Olarak İletilmiştir.<br/>
        İrsaliye yerine geçer.
    </div>
    
    <!-- Footer Section -->
    <div class="footer">
        <strong>WCPlugin.com EDM E-Fatura Premium WooCommerce</strong> Eklentisi ile
        E-Dönüşüm Merkezi EDM Teknolojileri tarafından üretilmiştir.
    </div>
</body>
</html>';
        
        // Create DOMPDF object
        if (class_exists('Dompdf\Dompdf')) {
            // For newer DOMPDF versions (0.8.0+)
            $dompdf = new Dompdf\Dompdf([
                'enable_remote' => true, // Enable remote images
                'enable_css_float' => true,
                'enable_html5_parser' => true,
                'default_paper_size' => 'A4',
                'default_font' => 'DejaVu Sans'
            ]);
        } else if (class_exists('DOMPDF')) {
            // For older DOMPDF versions
            $dompdf = new DOMPDF();
            $dompdf->set_option('enable_remote', true);
            $dompdf->set_option('enable_css_float', true);
            $dompdf->set_option('default_paper_size', 'A4');
            $dompdf->set_option('default_font', 'DejaVu Sans');
        } else {
            throw new Exception('DOMPDF class not found after including required files.');
        }
        
        // Set PDF size and orientation from settings
        $settings = get_option('edm_efatura_premium_settings', array());
        $paper_size = isset($settings['pdf_size']) ? $settings['pdf_size'] : 'A4';
        $orientation = isset($settings['pdf_orientation']) ? ($settings['pdf_orientation'] === 'L' ? 'landscape' : 'portrait') : 'portrait';
        
        // Load HTML
        $dompdf->loadHtml($html);
        
        // Set paper size
        $dompdf->setPaper($paper_size, $orientation);
        
        // Render PDF
        $dompdf->render();
        
        // Save to file
        file_put_contents($filename, $dompdf->output());
        
        // Update invoice with PDF path
        $invoice['pdf_path'] = $filename;
        $db->save_invoice($invoice);
        
        return $filename;
        
    } catch (Exception $e) {
        // Log the error
        $logger->error('PDF Generation Error: ' . $e->getMessage());
        
        // Try fallback method for servers without proper PDF libraries
        try {
            $fallback_filename = $pdf_dir . '/fatura-text-' . $invoice_id . '-' . $invoice['uuid'] . '.html';
            file_put_contents($fallback_filename, $html);
            return $fallback_filename;
        } catch (Exception $fallback_e) {
            $logger->error('PDF Fallback Error: ' . $fallback_e->getMessage());
            return false;
        }
    }
}

// Removed: Duplicate AJAX handler, now handled by admin class

    /**
 * Display invoice information in admin order page
 *
 * @param WC_Order $order Order object
 */
public function display_invoice_info_in_order($order) {
    $order_id = $order->get_id();
    
    // Get invoice
    $invoice = $this->db->get_invoice_by_order_id($order_id);
    
    echo '<div class="edm-invoice-info">';
    echo '<h3>' . esc_html__('E-Invoice Information', 'edm-efatura-premium') . '</h3>';
    
    if ($invoice) {
        // HPOS compatibility
        $hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
            \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
        
        // Get invoice type
        if ($hpos_enabled) {
            $invoice_type = $order->get_meta('edm_invoice_type', true);
            $invoice_email_sent = $order->get_meta('edm_invoice_email_sent', true);
        } else {
            $invoice_type = get_post_meta($order_id, 'edm_invoice_type', true);
            $invoice_email_sent = get_post_meta($order_id, 'edm_invoice_email_sent', true);
        }
        
        // Check if invoice is cancelled
        $is_cancelled = false;
        if (isset($invoice['status'])) {
            $status = strtoupper($invoice['status']);
            $is_cancelled = ($status === 'CANCELLED' || strpos($status, 'CANCELLED') !== false);
            
            // Delete PDF file if invoice is cancelled
            if ($is_cancelled && !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path'])) {
                // Delete the file
                @unlink($invoice['pdf_path']);
                
                // Update the invoice record to remove the PDF path
                $invoice['pdf_path'] = '';
                $this->db->save_invoice($invoice);
            }
        }
        
        if ($invoice_type === 'corporate') {
            if ($hpos_enabled) {
                $vkn = $order->get_meta('edm_tax_number', true);
                $tax_office = $order->get_meta('edm_tax_office', true);
                $company_name = $order->get_meta('edm_company_name', true);
            } else {
                $vkn = get_post_meta($order_id, 'edm_tax_number', true);
                $tax_office = get_post_meta($order_id, 'edm_tax_office', true);
                $company_name = get_post_meta($order_id, 'edm_company_name', true);
            }
            
            echo '<p><strong>' . esc_html__('Invoice Type:', 'edm-efatura-premium') . '</strong> ' . esc_html__('Corporate', 'edm-efatura-premium') . '</p>';
            echo '<p><strong>' . esc_html__('Company Name:', 'edm-efatura-premium') . '</strong> ' . esc_html($company_name) . '</p>';
            echo '<p><strong>' . esc_html__('Tax Number:', 'edm-efatura-premium') . '</strong> ' . esc_html($vkn) . '</p>';
            echo '<p><strong>' . esc_html__('Tax Office:', 'edm-efatura-premium') . '</strong> ' . esc_html($tax_office) . '</p>';
        } else {
            // First check if customer TC meta key is specified in settings
            $settings = get_option('edm_efatura_premium_settings', array());
            $customer_tc_meta_key = isset($settings['customer_tc_meta_key']) && !empty($settings['customer_tc_meta_key']) ? 
                                    $settings['customer_tc_meta_key'] : '';
            
            if (!empty($customer_tc_meta_key)) {
                // Try to get TC number from the specified order meta key
                if ($hpos_enabled) {
                    $vkn = $order->get_meta($customer_tc_meta_key, true);
                } else {
                    $vkn = get_post_meta($order_id, $customer_tc_meta_key, true);
                }
            } else {
                // Otherwise, check the default edm_tc_number from order meta
                if ($hpos_enabled) {
                    $vkn = $order->get_meta('edm_tc_number', true);
                } else {
                    $vkn = get_post_meta($order_id, 'edm_tc_number', true);
                }
            }
            
            echo '<p><strong>' . esc_html__('Invoice Type:', 'edm-efatura-premium') . '</strong> ' . esc_html__('Individual', 'edm-efatura-premium') . '</p>';
            /* echo '<p><strong>' . esc_html__('TC Number:', 'edm-efatura-premium') . '</strong> ' . esc_html($vkn) . '</p>'; */
        }
        
        // Display invoice profile
        $profile_id = isset($invoice['extra_data']['profile_id']) ? $invoice['extra_data']['profile_id'] : '';
        if ($profile_id) {
            $profile_name = '';
            switch ($profile_id) {
                case 'TICARIFATURA':
                    $profile_name = __('Commercial Invoice', 'edm-efatura-premium');
                    break;
                case 'EARSIVFATURA':
                    $profile_name = __('E-Archive Invoice', 'edm-efatura-premium');
                    break;
                case 'INTERNETSATIS':
                    $profile_name = __('Internet Sales', 'edm-efatura-premium');
                    break;
                case 'TEMELFATURA':
                    $profile_name = __('Basic Invoice', 'edm-efatura-premium');
                    break;
                default:
                    $profile_name = $profile_id;
            }
            echo '<p><strong>' . esc_html__('Invoice Profile:', 'edm-efatura-premium') . '</strong> ' . esc_html($profile_name) . '</p>';
        }
        
        // Display invoice type code
        $invoice_type_code = isset($invoice['extra_data']['invoice_type']) ? $invoice['extra_data']['invoice_type'] : '';
        if ($invoice_type_code) {
            $type_name = '';
            switch ($invoice_type_code) {
                case 'SATIS':
                    $type_name = __('Sales', 'edm-efatura-premium');
                    break;
                
                case 'ISTISNA':
                    $type_name = __('Exemption', 'edm-efatura-premium');
                    break;
                default:
                    $type_name = $invoice_type_code;
            }
            echo '<p><strong>' . esc_html__('Invoice Type Code:', 'edm-efatura-premium') . '</strong> ' . esc_html($type_name) . '</p>';
        }
        
        // Display exception code if applicable
        $exception_code = isset($invoice['extra_data']['exception_code']) ? $invoice['extra_data']['exception_code'] : '';
        if ($exception_code) {
            echo '<p><strong>' . esc_html__('Exception Code:', 'edm-efatura-premium') . '</strong> ' . esc_html($exception_code) . '</p>';
        }
        
        echo '<hr>';
        
        echo '<p><strong>' . esc_html__('Invoice ID:', 'edm-efatura-premium') . '</strong> ' . esc_html($invoice['invoice_id']) . '</p>';
        echo '<p><strong>' . esc_html__('UUID:', 'edm-efatura-premium') . '</strong> ' . esc_html($invoice['uuid']) . '</p>';
        echo '<p><strong>' . esc_html__('Status:', 'edm-efatura-premium') . '</strong> <span class="edm-status edm-status-' . esc_attr(strtolower($invoice['status'])) . '">' . esc_html($this->translate_invoice_status($invoice['status'])) . '</span></p>';
        echo '<p><strong>' . esc_html__('Created:', 'edm-efatura-premium') . '</strong> ' . esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($invoice['created_at']))) . '</p>';
        
        // Show PDF link or generate button if not cancelled
        if (!$is_cancelled) {
            // Show PDF link if available
            if (!empty($invoice['pdf_path']) && file_exists($invoice['pdf_path'])) {
                $pdf_url = add_query_arg(array(
                    'edm_invoice_download' => $invoice['uuid'],
                    'nonce' => wp_create_nonce('edm_invoice_download_' . $invoice['uuid'])
                ), admin_url('admin.php?page=edm-efatura-premium-list'));
                
                echo '<p><a href="' . esc_url($pdf_url) . '" target="_blank" class="button">' . esc_html__('Download PDF', 'edm-efatura-premium') . '</a></p>';
            } else {
                echo '<p><button type="button" class="button edm-generate-pdf" data-uuid="' . esc_attr($invoice['uuid']) . '">' . esc_html__('Generate PDF', 'edm-efatura-premium') . '</button></p>';
            }
        }
        
        // Check status button
        echo '<p><button type="button" class="button edm-check-status" data-uuid="' . esc_attr($invoice['uuid']) . '">' . esc_html__('Check Status', 'edm-efatura-premium') . '</button></p>';
        
        // YENİ: E-posta gönderme butonu (yalnızca iptal edilmemiş faturalar için)
        if (!$is_cancelled) {
            if ($invoice_email_sent !== 'yes') {
                echo '<p><button type="button" class="button button-primary edm-send-invoice-email" data-order-id="' . esc_attr($order_id) . '" data-uuid="' . esc_attr($invoice['uuid']) . '">' . esc_html__('Faturayı E-posta ile gönder', 'edm-efatura-premium') . '</button></p>';
            } else {
                echo '<p><span class="dashicons dashicons-email-alt"></span> ' . esc_html__('Fatura maili zaten gönderildi.', 'edm-efatura-premium') . '</p>';
                // Opsiyonel: Yeniden gönderme butonu
                echo '<p><button type="button" class="button edm-send-invoice-email" data-order-id="' . esc_attr($order_id) . '" data-uuid="' . esc_attr($invoice['uuid']) . '">' . esc_html__('Faturayı E-posta ile tekrar gönder', 'edm-efatura-premium') . '</button></p>';
            }
        }
        
        // Cancel button for all invoices except already cancelled ones
        if (!$is_cancelled) {
            echo '<p><button type="button" class="button button-secondary edm-cancel-invoice" data-uuid="' . esc_attr($invoice['uuid']) . '">' . esc_html__('Faturayı İptal Et', 'edm-efatura-premium') . '</button></p>';
        }
        
        // If invoice is cancelled, show form to create new invoice with different settings
        if ($is_cancelled) {
            echo '<hr>';
            echo '<h4>' . esc_html__('Create New Invoice', 'edm-efatura-premium') . '</h4>';
            
            // Get current invoice settings to prefill the form
            $current_profile = isset($invoice['extra_data']['profile_id']) ? $invoice['extra_data']['profile_id'] : '';
            $current_type_code = isset($invoice['extra_data']['invoice_type']) ? $invoice['extra_data']['invoice_type'] : '';
            $current_exception_code = isset($invoice['extra_data']['exception_code']) ? $invoice['extra_data']['exception_code'] : '';
            
            $this->display_invoice_settings_form($order, $current_profile, $current_type_code, $current_exception_code);
        }
        
        // Add status message placeholder
        echo '<div class="edm-result-box" style="display:none;"></div>';
        echo '<div id="edm-invoice-status-message" class="updated" style="display:none;"><p></p></div>';
    } else {
        // Show simple invoice creation buttons (no invoice exists)
        $this->display_invoice_settings_form($order);
    }
    
    echo '</div>';
}

    /**
     * Display invoice settings form for cancelled invoices
     *
     * @param WC_Order $order Order object
     * @param string $current_profile Current invoice profile
     * @param string $current_type_code Current invoice type code
     * @param string $current_exception_code Current exception code
     */
    private function display_invoice_settings_form($order, $current_profile = '', $current_type_code = '', $current_exception_code = '') {
        $order_id = $order->get_id();
        
        // HPOS compatibility
        $hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
            \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
        
        // Global settings
        $settings = get_option('edm_efatura_premium_settings', array());
        
        if ($hpos_enabled) {
            $tax_number = $order->get_meta('edm_tax_number', true);
            $tax_office = $order->get_meta('edm_tax_office', true);
            $company_name = $order->get_meta('edm_company_name', true);
        } else {
            $tax_number = get_post_meta($order_id, 'edm_tax_number', true);
            $tax_office = get_post_meta($order_id, 'edm_tax_office', true);
            $company_name = get_post_meta($order_id, 'edm_company_name', true);
        }
        
        echo '<div class="edm-invoice-form">';
        
        // Simple button-based selection
        echo '<p style="margin-bottom: 20px;">';
        echo '<button type="button" class="button button-primary button-large edm-generate-invoice-list" data-order-id="' . esc_attr($order_id) . '" style="margin-right: 10px;">' . esc_html__('Kişiye Fatura Oluştur', 'edm-efatura-premium') . '</button>';
        echo '<button type="button" class="button button-secondary button-large edm-show-corporate-form" data-order-id="' . esc_attr($order_id) . '">' . esc_html__('Firmaya Fatura Oluştur', 'edm-efatura-premium') . '</button>';
        echo '</p>';
        
        // Add result container for individual invoice button
        echo '<div class="edm-generate-result" data-order="' . esc_attr($order_id) . '" style="display:none;margin-top:15px;"></div>';
        
        // Corporate form (hidden by default)
        echo '<div id="edm-corporate-form" style="display: none; border: 1px solid #ccc; padding: 15px; background: #f9f9f9; margin-top: 10px;">';
        echo '<h4>' . esc_html__('Firma Bilgileri', 'edm-efatura-premium') . '</h4>';
        echo '<p><strong>' . esc_html__('Firma Adı:', 'edm-efatura-premium') . '</strong><br>';
        echo '<input type="text" id="edm_company_name" name="edm_company_name" value="' . esc_attr($company_name) . '" style="width: 100%; max-width: 400px;" /></p>';
        echo '<p><strong>' . esc_html__('Vergi Numarası:', 'edm-efatura-premium') . '</strong><br>';
        echo '<input type="text" id="edm_tax_number" name="edm_tax_number" value="' . esc_attr($tax_number) . '" pattern="[0-9]{10,11}" maxlength="11" style="width: 100%; max-width: 400px;" /></p>';
        echo '<p><strong>' . esc_html__('Vergi Dairesi (Zorunlu):', 'edm-efatura-premium') . '</strong><br>';
        echo '<input type="text" id="edm_tax_office" name="edm_tax_office" value="' . esc_attr($tax_office) . '" style="width: 100%; max-width: 400px;" /></p>';
        
        // GIB Email field
        echo '<p><strong>' . esc_html__('GİB E-posta (Zorunlu):', 'edm-efatura-premium') . '</strong><br>';
        echo '<input type="email" id="edm_gib_email" name="edm_gib_email" value="" placeholder="defaultpk@gmail.com" style="width: 100%; max-width: 400px;" />';
        echo '<small style="color: #666; display: block; margin-top: 5px;">Sadece e-posta adresini girin. &quot;urn:mail:&quot; ön eki otomatik eklenecektir.</small></p>';
        
        // Company address field
        $company_address = get_post_meta($order_id, 'edm_company_address', true);
        echo '<p><strong>' . esc_html__('Şirket Adresi (Zorunlu):', 'edm-efatura-premium') . '</strong><br>';
        echo '<textarea id="edm_company_address" name="edm_company_address" placeholder="Şirket adresini giriniz" style="width: 100%; max-width: 400px; height: 60px;">' . esc_textarea($company_address) . '</textarea></p>';
        
        // Country selection
        $company_country = get_post_meta($order_id, 'edm_company_country', true);
        if (empty($company_country)) {
            $company_country = 'TR'; // Default to Turkey
        }
        echo '<p><strong>' . esc_html__('Ülke:', 'edm-efatura-premium') . '</strong><br>';
        echo '<select id="edm_company_country" name="edm_company_country" style="width: 100%; max-width: 400px;">';
        echo '<option value="TR"' . selected($company_country, 'TR', false) . '>Türkiye</option>';
        echo '<option value="US"' . selected($company_country, 'US', false) . '>Amerika Birleşik Devletleri</option>';
        echo '<option value="DE"' . selected($company_country, 'DE', false) . '>Almanya</option>';
        echo '<option value="FR"' . selected($company_country, 'FR', false) . '>Fransa</option>';
        echo '<option value="GB"' . selected($company_country, 'GB', false) . '>İngiltere</option>';
        echo '<option value="IT"' . selected($company_country, 'IT', false) . '>İtalya</option>';
        echo '<option value="NL"' . selected($company_country, 'NL', false) . '>Hollanda</option>';
        echo '</select></p>';
        echo '<p>';
        echo '<button type="button" class="button button-primary edm-generate-corporate-invoice" data-order-id="' . esc_attr($order_id) . '" onclick="testCorporateInvoice(' . esc_attr($order_id) . ')">' . esc_html__('Firma Faturası Oluştur', 'edm-efatura-premium') . '</button> ';
        echo '<button type="button" class="button edm-cancel-corporate-form">' . esc_html__('İptal', 'edm-efatura-premium') . '</button>';
        echo '</p>';
        ?>
        <script>
        function testCorporateInvoice(orderId) {
            if (typeof edmEfaturaParams !== "undefined") {
                
                var companyName = jQuery("#edm_company_name").val();
                var taxNumber = jQuery("#edm_tax_number").val();
                var taxOffice = jQuery("#edm_tax_office").val();
                var gibEmail = jQuery("#edm_gib_email").val();
                var companyAddress = jQuery("#edm_company_address").val();
                var companyCountry = jQuery("#edm_company_country").val();
                
                if (!companyName || !taxNumber || !taxOffice || !gibEmail || !companyAddress) {
                    alert("Lütfen tüm zorunlu alanları doldurun");
                    return;
                }
                
                gibEmail = gibEmail.replace(/^urn:mail:/, '');
                
                var gibEmailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!gibEmailRegex.test(gibEmail)) {
                    alert("Geçerli e-posta adresi girin");
                    return;
                }
                
                if (!confirm("Emin misiniz?")) {
                    return;
                }
                
                jQuery.ajax({
                    url: edmEfaturaParams.ajaxUrl,
                    type: "POST",
                    data: {
                        action: "edm_generate_invoice_for_order",
                        nonce: edmEfaturaParams.nonce,
                        order_id: orderId,
                        invoice_type: "corporate",
                        invoice_profile: "TEMELFATURA",
                        company_name: companyName,
                        tax_number: taxNumber,
                        tax_office: taxOffice,
                        gib_email: gibEmail,
                        company_address: companyAddress,
                        company_country: companyCountry
                    },
                    success: function(response) {
                        if (response.success) {
                            alert("Fatura oluşturuldu");
                            location.reload();
                        } else {
                            alert("Hata: " + response.data);
                        }
                    },
                    error: function(xhr, status, error) {
                        alert("AJAX Hatası: " + error);
                    }
                });
            } else {
                alert("Script yüklenemedi");
            }
        }
        </script>
        <?php
        echo '</div>';
        
        echo '</div>';
        
        // JavaScript for form interactions
        echo '<script>
jQuery(document).ready(function($) {
    // Initialize edmEfaturaParams if not defined
    if (typeof window.edmEfaturaParams === "undefined") {
        window.edmEfaturaParams = {
            ajaxUrl: "' . admin_url('admin-ajax.php') . '",
            nonce: "' . wp_create_nonce('edm_efatura_premium_nonce') . '"
        };
    }
    
    // Show corporate form
    $(".edm-show-corporate-form").on("click", function() {
        $("#edm-corporate-form").slideDown();
        $(this).prop("disabled", true);
    });
    
    // Hide corporate form
    $(".edm-cancel-corporate-form").on("click", function() {
        $("#edm-corporate-form").slideUp();
        $(".edm-show-corporate-form").prop("disabled", false);
    });
});
</script>';
    }
    
    /**
     * Add generate invoice action to order actions
     *
     * @param WC_Order $order Order object
     */
    public function add_generate_invoice_action($order) {
        $order_id = $order->get_id();
        
        // Check if invoice already exists
        $invoice = $this->db->get_invoice_by_order_id($order_id);
        
        if ($invoice) {
            echo '<a href="' . esc_url(admin_url('admin.php?page=edm-efatura-premium-list&action=view&invoice_id=' . $invoice['id'])) . '" class="button tips" data-tip="' . esc_attr__('View Invoice', 'edm-efatura-premium') . '"><span class="dashicons dashicons-media-text"></span></a>';
        } else {
            echo '<button type="button" class="button tips edm-generate-invoice" data-tip="' . esc_attr__('Generate E-Invoice', 'edm-efatura-premium') . '" data-order-id="' . esc_attr($order_id) . '"><span class="dashicons dashicons-media-text"></span></button>';
        }
    }
    
    /**
     * AJAX handler for generating invoice for order
     */
    public function ajax_generate_invoice_for_order() {
    // Initialize logger
    $logger = new EDM_Efatura_Premium_Logger();
    
    // Check nonce
    check_ajax_referer('edm_efatura_premium_nonce', 'nonce');
    
    // Check if user has permission
    if (!current_user_can('edit_shop_orders')) {
        $logger->warning('Unauthorized invoice generation attempt', array(
            'user_id' => get_current_user_id(),
            'action' => 'ajax_generate_invoice_for_order'
        ));
        wp_send_json_error(__('You do not have permission to perform this action.', 'edm-efatura-premium'));
        return;
    }
    
    // Get order ID
    $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
    
    if (!$order_id) {
        $logger->error('Invalid order ID provided for invoice generation', array(
            'provided_order_id' => $_POST['order_id'] ?? 'not_set',
            'user_id' => get_current_user_id()
        ));
        wp_send_json_error(__('Invalid order ID.', 'edm-efatura-premium'));
        return;
    }
    
    // Get order
    $order = wc_get_order($order_id);
    
    if (!$order) {
        $logger->error('Order not found for invoice generation', array(
            'order_id' => $order_id,
            'user_id' => get_current_user_id()
        ));
        wp_send_json_error(__('Order not found.', 'edm-efatura-premium'));
        return;
    }
    
    // Check if order total is zero or the order has no items with value
    if ($order->get_total() <= 0) {
        $logger->warning('Attempted to generate invoice for zero-value order', array(
            'order_id' => $order_id,
            'order_total' => $order->get_total(),
            'order_status' => $order->get_status(),
            'user_id' => get_current_user_id()
        ));
        wp_send_json_error(__('Cannot generate invoice for zero-value order.', 'edm-efatura-premium'));
        return;
    }
    
    // Check if invoice already exists
    $db = new EDM_Efatura_Premium_DB();
    $existing_invoice = $db->get_invoice_by_order_id($order_id);
    $is_cancelled_invoice = false;
    
    if ($existing_invoice) {
        // Check if the existing invoice is cancelled
        if (isset($existing_invoice['status'])) {
            $status = strtoupper($existing_invoice['status']);
            $is_cancelled_invoice = ($status === 'CANCELLED' || stripos($status, 'CANCELLED') !== false);
        }
        
        // If invoice exists but is not cancelled, return error
        if (!$is_cancelled_invoice) {
            wp_send_json_error(__('Invoice already exists for this order.', 'edm-efatura-premium'));
            return;
        }
        
        // If invoice is cancelled, we'll continue with generating a new one
        // Clear old invoice meta data to start fresh
        if ($hpos_enabled) {
            $order->delete_meta_data('edm_invoice_profile');
            $order->delete_meta_data('edm_invoice_type_code');
            $order->delete_meta_data('edm_exception_code');
            $order->delete_meta_data('edm_invoice_type');
            $order->delete_meta_data('edm_company_name');
            $order->delete_meta_data('edm_tax_number');
            $order->delete_meta_data('edm_tax_office');
            $order->delete_meta_data('edm_tc_number');
            $order->save();
        } else {
            delete_post_meta($order_id, 'edm_invoice_profile');
            delete_post_meta($order_id, 'edm_invoice_type_code');
            delete_post_meta($order_id, 'edm_exception_code');
            delete_post_meta($order_id, 'edm_invoice_type');
            delete_post_meta($order_id, 'edm_company_name');
            delete_post_meta($order_id, 'edm_tax_number');
            delete_post_meta($order_id, 'edm_tax_office');
            delete_post_meta($order_id, 'edm_tc_number');
        }
    }
    
    // Get order meta data for invoice parameters
    $hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
        \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
    
    if ($hpos_enabled) {
        $invoice_profile = $order->get_meta('edm_invoice_profile', true);
        $invoice_type_code = $order->get_meta('edm_invoice_type_code', true);
        $exception_code = $order->get_meta('edm_exception_code', true);
    } else {
        $invoice_profile = get_post_meta($order_id, 'edm_invoice_profile', true);
        $invoice_type_code = get_post_meta($order_id, 'edm_invoice_type_code', true);
        $exception_code = get_post_meta($order_id, 'edm_exception_code', true);
    }
    
    // Get global settings
    $settings = get_option('edm_efatura_premium_settings', array());
    
    // Check if form data is provided
    $has_form_data = (isset($_POST['invoice_profile']) && !empty($_POST['invoice_profile'])) ||
                     (isset($_POST['invoice_type_code']) && !empty($_POST['invoice_type_code'])) ||
                     (isset($_POST['exception_code']) && !empty($_POST['exception_code'])) ||
                     (isset($_POST['tax_rate']) && $_POST['tax_rate'] !== '');
    
    // For ANY invoice with form data, prioritize form data over everything
    if ($has_form_data) {
        $logger->info('Using form data for cancelled invoice', array(
            'order_id' => $order_id,
            'post_data' => array(
                'invoice_profile' => $_POST['invoice_profile'] ?? 'not_set',
                'invoice_type_code' => $_POST['invoice_type_code'] ?? 'not_set',
                'exception_code' => $_POST['exception_code'] ?? 'not_set'
            )
        ));
        
        // Force use form data for cancelled invoices
        $invoice_profile = isset($_POST['invoice_profile']) && !empty($_POST['invoice_profile']) 
            ? sanitize_text_field($_POST['invoice_profile'])
            : (isset($settings['invoice_profile']) ? $settings['invoice_profile'] : 'EARSIVFATURA');
            
        $invoice_type_code = isset($_POST['invoice_type_code']) && !empty($_POST['invoice_type_code'])
            ? sanitize_text_field($_POST['invoice_type_code'])
            : (isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS');
            
        $exception_code = isset($_POST['exception_code']) && !empty($_POST['exception_code'])
            ? sanitize_text_field($_POST['exception_code'])
            : '';
            
        // Handle tax rate for SATIS invoices in form data branch too
        if (isset($_POST['tax_rate']) && $_POST['tax_rate'] !== '') {
            $custom_tax_rate = floatval($_POST['tax_rate']);
            
            // Clear any previous custom tax rate first
            $order->delete_meta_data('_edm_custom_tax_rate');
            
            // Save new custom tax rate to order meta
            $order->update_meta_data('_edm_custom_tax_rate', $custom_tax_rate);
            $order->save_meta_data();
        }
    } else {
        // Normal flow - use order meta first, then global settings
        if (empty($invoice_profile)) {
            $invoice_profile = isset($settings['invoice_profile']) ? $settings['invoice_profile'] : 'EARSIVFATURA';
        }
        
        if (empty($invoice_type_code)) {
            $invoice_type_code = isset($settings['invoice_type']) ? $settings['invoice_type'] : 'SATIS';
        }
        
        if (empty($exception_code) && $invoice_type_code === 'ISTISNA') {
            $exception_code = isset($settings['invoice_exception_code']) ? $settings['invoice_exception_code'] : '';
        }
        
        // If parameters are passed via POST, they override the order meta and defaults
        if (isset($_POST['invoice_profile']) && !empty($_POST['invoice_profile'])) {
            $invoice_profile = sanitize_text_field($_POST['invoice_profile']);
        }
        
        if (isset($_POST['invoice_type_code']) && !empty($_POST['invoice_type_code'])) {
            $invoice_type_code = sanitize_text_field($_POST['invoice_type_code']);
        }
        
        if (isset($_POST['exception_code']) && !empty($_POST['exception_code'])) {
            $exception_code = sanitize_text_field($_POST['exception_code']);
        }
        
        // Handle tax rate for SATIS invoices
        if (isset($_POST['tax_rate']) && $_POST['tax_rate'] !== '') {
            $custom_tax_rate = floatval($_POST['tax_rate']);
            
            // Clear any previous custom tax rate first
            $order->delete_meta_data('_edm_custom_tax_rate');
            
            // Save new custom tax rate to order meta
            $order->update_meta_data('_edm_custom_tax_rate', $custom_tax_rate);
            $order->save_meta_data();
        }
    }
    
    // Handle additional form parameters
    $invoice_type = '';
    $company_name = '';
    $tax_number = '';
    $tax_office = '';
    
    if (isset($_POST['invoice_type']) && !empty($_POST['invoice_type'])) {
        $invoice_type = sanitize_text_field($_POST['invoice_type']);
    }
    
    if (isset($_POST['company_name']) && !empty($_POST['company_name'])) {
        $company_name = sanitize_text_field($_POST['company_name']);
    }
    
    if (isset($_POST['tax_number']) && !empty($_POST['tax_number'])) {
        $tax_number = sanitize_text_field($_POST['tax_number']);
    }
    
    if (isset($_POST['tax_office']) && !empty($_POST['tax_office'])) {
        $tax_office = sanitize_text_field($_POST['tax_office']);
    }
    
    // Handle new TEMELFATURA fields
    $gib_email = '';
    $company_address = '';
    $company_country = 'TR'; // Default to Turkey
    
    if (isset($_POST['gib_email']) && !empty($_POST['gib_email'])) {
        $gib_email = sanitize_email($_POST['gib_email']);
    }
    
    if (isset($_POST['company_address']) && !empty($_POST['company_address'])) {
        $company_address = sanitize_textarea_field($_POST['company_address']);
    }
    
    if (isset($_POST['company_country']) && !empty($_POST['company_country'])) {
        $company_country = sanitize_text_field($_POST['company_country']);
    }
    
    // If corporate data is provided, force TEMELFATURA profile and corporate type
    // Tax office, GIB email and address are required for TEMELFATURA
    if (!empty($company_name) && !empty($tax_number) && !empty($tax_office) && !empty($gib_email)) {
        $invoice_profile = 'TEMELFATURA';
        $invoice_type = 'corporate';
        
        $logger->info('Corporate invoice data provided, forcing TEMELFATURA profile', array(
            'order_id' => $order_id,
            'company_name' => $company_name,
            'tax_number' => $tax_number,
            'tax_office' => $tax_office
        ));
    }
    
    // Save all parameters to order meta for future reference
    if ($hpos_enabled) {
        $order->update_meta_data('edm_invoice_profile', $invoice_profile);
        $order->update_meta_data('edm_invoice_type_code', $invoice_type_code);
        
        if ($exception_code) {
            $order->update_meta_data('edm_exception_code', $exception_code);
        }
        
        if ($invoice_type) {
            $order->update_meta_data('edm_invoice_type', $invoice_type);
        }
        
        if ($company_name) {
            $order->update_meta_data('edm_company_name', $company_name);
        }
        
        if ($tax_number) {
            $order->update_meta_data('edm_tax_number', $tax_number);
        }
        
        if ($tax_office) {
            $order->update_meta_data('edm_tax_office', $tax_office);
        }
        
        if ($gib_email) {
            $order->update_meta_data('edm_gib_email', $gib_email);
        }
        
        if ($company_address) {
            $order->update_meta_data('edm_company_address', $company_address);
        }
        
        if ($company_country) {
            $order->update_meta_data('edm_company_country', $company_country);
        }
        
        // Save meta data
        $order->save();
    } else {
        update_post_meta($order_id, 'edm_invoice_profile', $invoice_profile);
        update_post_meta($order_id, 'edm_invoice_type_code', $invoice_type_code);
        
        if ($exception_code) {
            update_post_meta($order_id, 'edm_exception_code', $exception_code);
        }
        
        if ($invoice_type) {
            update_post_meta($order_id, 'edm_invoice_type', $invoice_type);
        }
        
        if ($company_name) {
            update_post_meta($order_id, 'edm_company_name', $company_name);
        }
        
        if ($tax_number) {
            update_post_meta($order_id, 'edm_tax_number', $tax_number);
        }
        
        if ($tax_office) {
            update_post_meta($order_id, 'edm_tax_office', $tax_office);
        }
        
        if ($gib_email) {
            update_post_meta($order_id, 'edm_gib_email', $gib_email);
        }
        
        if ($company_address) {
            update_post_meta($order_id, 'edm_company_address', $company_address);
        }
        
        if ($company_country) {
            update_post_meta($order_id, 'edm_company_country', $company_country);
        }
    }
    
    // API connection check and if needed recreate
    if (!$this->api) {
        $settings = get_option('edm_efatura_premium_settings', array());
        if (!empty($settings['api_url']) && !empty($settings['username']) && !empty($settings['password'])) {
            $edm_email = !empty($settings['edm_email']) ? $settings['edm_email'] : '';
            
            $this->api = new EDM_Efatura_Premium_API(
                $settings['api_url'],
                $settings['username'],
                $settings['password'],
                isset($settings['debug_mode']) && $settings['debug_mode'] === 'yes',
                $edm_email
            );
        }
        
        if (!$this->api) {
            wp_send_json_error(__('API not available. Please check your API settings.', 'edm-efatura-premium'));
            return;
        }
    }
    
    // Corporate invoices use TEMELFATURA profile - no GIB email needed
    
    // Generate invoice with the additional parameters
    $order_handler = new EDM_Efatura_Premium_Order_Handler($this->api);
    $result = $order_handler->generate_invoice_for_order_with_params($order, $invoice_profile, $invoice_type_code, $exception_code);
    
    if ($result) {
        // Log successful response
        $logger = new EDM_Efatura_Premium_Logger();
        $logger->info('Invoice generated successfully via AJAX', array(
            'order_id' => $order_id,
            'invoice_id' => isset($result['id']) ? $result['id'] : '',
            'uuid' => isset($result['uuid']) ? $result['uuid'] : ''
        ));
        
        // Add private order note (not sent to customer)
        $order->add_order_note(
            sprintf(
                __('EDM E-invoice generated successfully. Invoice ID: %s', 'edm-efatura-premium'),
                isset($result['id']) ? $result['id'] : 'N/A'
            ),
            0 // Private note - not sent to customer
        );
        
        wp_send_json_success(__('Invoice generated successfully.', 'edm-efatura-premium'));
    } else {
        wp_send_json_error(__('Failed to generate invoice. Please check the logs for more information.', 'edm-efatura-premium'));
    }
}

	/**
	 * Generate invoice for order with additional parameters
	 *
	 * @param WC_Order $order Order object
	 * @param string $invoice_profile Invoice profile
	 * @param string $invoice_type_code Invoice type code
	 * @param string $exception_code Exception code
	 * @return array|bool Invoice data on success, false on failure
	 */
	public function generate_invoice_for_order_with_params($order, $invoice_profile = '', $invoice_type_code = '', $exception_code = '') {
		$order_id = $order->get_id();
		
		// Debug logs
		$this->logger->info('Starting invoice generation with parameters', array(
			'order_id' => $order_id,
			'invoice_profile' => $invoice_profile,
			'invoice_type_code' => $invoice_type_code,
			'exception_code' => $exception_code
		));
		
		// Check if order total is zero or the order has no items with value
		if ($order->get_total() <= 0) {
			$this->logger->info('Skipping invoice generation for zero-value order', array(
				'order_id' => $order_id,
				'total' => $order->get_total()
			));
			
			// Add private order note
			$order->add_order_note(
				__('EDM E-invoice generation skipped. Order has zero value.', 'edm-efatura-premium'),
				0 // Private note
			);
			
			return false;
		}
		
		// Check if invoice already exists
		$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
		if ($existing_invoice) {
			// Check if invoice is cancelled
			$is_cancelled = false;
			if (isset($existing_invoice['status'])) {
				$status = strtoupper($existing_invoice['status']);
				$is_cancelled = ($status === 'CANCELLED' || stripos($status, 'CANCELLED') !== false);
			}
			
			if (!$is_cancelled) {
				$this->logger->info('Invoice already exists for order', array(
					'order_id' => $order_id,
					'invoice_id' => $existing_invoice['id'],
					'uuid' => $existing_invoice['uuid']
				));
				return $existing_invoice;
			}
			
			// If the invoice is cancelled, delete it from database before creating a new one
			$this->logger->info('Deleting cancelled invoice to create a new one', array(
				'order_id' => $order_id,
				'invoice_id' => isset($existing_invoice['id']) ? $existing_invoice['id'] : 'unknown',
				'uuid' => isset($existing_invoice['uuid']) ? $existing_invoice['uuid'] : 'unknown',
				'status' => isset($existing_invoice['status']) ? $existing_invoice['status'] : 'unknown'
			));
			
			// Delete the cancelled invoice
			if (isset($existing_invoice['uuid'])) {
				// If there's a delete_invoice_by_uuid method
				if (method_exists($this->db, 'delete_invoice_by_uuid')) {
					$this->db->delete_invoice_by_uuid($existing_invoice['uuid']);
				}
				// Or if there's a delete_invoice method that accepts uuid
				else if (method_exists($this->db, 'delete_invoice')) {
					$this->db->delete_invoice($existing_invoice['uuid']);
				}
				// As a fallback, use the invoice ID
				else if (isset($existing_invoice['id']) && method_exists($this->db, 'delete_invoice')) {
					$this->db->delete_invoice($existing_invoice['id']);
				}
				else {
					$this->logger->warning('Unable to delete cancelled invoice - no suitable delete method found', array(
						'order_id' => $order_id,
						'invoice_id' => isset($existing_invoice['id']) ? $existing_invoice['id'] : 'unknown'
					));
				}
			}
		}
		
		// Check if API is available
		if (!$this->api) {
			// Try to create a new API connection if not available
			$settings = get_option('edm_efatura_premium_settings', array());
			if (!empty($settings['api_url']) && !empty($settings['username']) && !empty($settings['password'])) {
				$edm_email = !empty($settings['edm_email']) ? $settings['edm_email'] : '';
				
				$this->api = new EDM_Efatura_Premium_API(
					$settings['api_url'],
					$settings['username'],
					$settings['password'],
					isset($settings['debug_mode']) && $settings['debug_mode'] === 'yes',
					$edm_email
				);
				
				// Check the new API connection
				if (!$this->api) {
					$this->logger->error('API not available, cannot generate invoice', array('order_id' => $order_id));
					return false;
				}
			} else {
				$this->logger->error('API credentials not set, cannot generate invoice', array('order_id' => $order_id));
				return false;
			}
		}
		
		try {
			// Get order data for invoice - pass invoice type code for custom tax rate handling
			$invoice_data = $this->prepare_invoice_data_from_order($order, $invoice_type_code);
			
			// DEBUG: Check what prepare_invoice_data_from_order returned
			
			// Override with the parameters if provided
			if (!empty($invoice_profile)) {
				$invoice_data['profile_id'] = $invoice_profile;
			}
			
			// IMPORTANT FIX: Don't override invoice_type if prepare_invoice_data_from_order already set it correctly
			// Only use parameter if no category override was applied (i.e., still global default)
			if (!empty($invoice_type_code)) {
				
				// Only override if current type is still the global default and matches the parameter
				// This prevents category settings from being overridden
				if ($invoice_data['invoice_type'] === $invoice_type_code) {
				} else {
					// Keep the category-determined type instead of parameter override
				}
			}
			
			if (!empty($exception_code)) {
				$invoice_data['exception_code'] = $exception_code;
			}
			
			// Generate UUID if not provided
			$invoice_uuid = wp_generate_uuid4();
			
			// Generate invoice number
			$invoice_id = $this->generate_invoice_number();
			
			// Generate XML with the proper ID and UUID
			$invoice_xml = $this->generate_invoice_xml_with_id_uuid($invoice_data, $invoice_id, $invoice_uuid);
			
			// Send invoice
			$result = $this->api->send_invoice(
				$invoice_data['receiver_vkn'],
				$invoice_data['receiver_alias'],
				$invoice_xml
			);
			
			if ($result['success']) {
				// Save invoice to database
				$invoice = array(
					'id' => $invoice_id,
					'uuid' => $invoice_uuid,
					'order_id' => $order_id,
					'type' => 'EINVOICE',
					'direction' => 'OUT',
					'status' => $result['status'],
					'customer_vkn' => $invoice_data['receiver_vkn'],
					'customer_name' => $invoice_data['customer_name'],
					'amount' => $order->get_total(),
					'currency' => get_woocommerce_currency(),
					'content' => $invoice_xml,
					'extra_data' => array(
						'status_description' => $result['status_description'],
						'order_number' => $order->get_order_number(),
						'profile_id' => $invoice_data['profile_id'],
						'invoice_type' => $invoice_data['invoice_type'],
						'exception_code' => isset($invoice_data['exception_code']) ? $invoice_data['exception_code'] : ''
					)
				);
				
				$invoice_db_id = $this->db->save_invoice($invoice);
				
				if (!$invoice_db_id) {
					$this->logger->error('Failed to save invoice to database', array(
						'order_id' => $order_id,
						'invoice_id' => $invoice_id,
						'uuid' => $invoice_uuid
					));
					return false;
				}
				
				// Save items
				$items = array();
				foreach ($order->get_items() as $item) {
					$product = $item->get_product();
					$tax_rate = 0;
					$tax_amount = 0;
					
					// Calculate tax rate and amount
					if (wc_tax_enabled()) {
						$tax_data = $item->get_taxes();
						if (!empty($tax_data['total'])) {
							$tax_amount = array_sum($tax_data['total']);
							$tax_rate = round(($tax_amount / $item->get_subtotal()) * 100, 2);
						}
					}
					
					$items[] = array(
						'product_id' => $product ? $product->get_id() : null,
						'name' => $item->get_name(),
						'quantity' => $item->get_quantity(),
						'unit_price' => $item->get_subtotal() / $item->get_quantity(),
						'tax_rate' => $tax_rate,
						'tax_amount' => $tax_amount,
						'total_amount' => $item->get_total() + $tax_amount
					);
				}
				
				// Add shipping as a line item if applicable
				if ($order->get_shipping_total() > 0) {
					$shipping_tax = $order->get_shipping_tax();
					$shipping_method = $order->get_shipping_method();
					
					$shipping_name = __('Shipping', 'edm-efatura-premium');
					if (!empty($shipping_method)) {
						$shipping_name .= ': ' . $shipping_method;
					}
					
					// Use settings for shipping tax rate and unit code
					$settings = get_option('edm_efatura_premium_settings', array());
					$default_tax_rate = 20; // Kargo KDV oranı her zaman %20 sabit
					$shipping_unit_code = isset($settings['shipping_unit_code']) ? $settings['shipping_unit_code'] : 
										 (isset($settings['default_unit_code']) ? $settings['default_unit_code'] : 'C62');
					
					$items[] = array(
						'product_id' => null,
						'name' => $shipping_name,
						'quantity' => 1,
						'unit_price' => $order->get_shipping_total(),
						'unit_code' => $shipping_unit_code,
						'tax_rate' => $default_tax_rate,
						'tax_amount' => $shipping_tax,
						'total_amount' => $order->get_shipping_total() + $shipping_tax,
						'is_shipping' => true
					);
				}
				
				$this->db->save_invoice_items($invoice_db_id, $items);
				
				// Log success
				$this->logger->info('Invoice generated successfully for order', array(
					'order_id' => $order_id,
					'invoice_id' => $invoice_id,
					'uuid' => $invoice_uuid
				));
				
				// Update order meta to indicate invoice was created
				$this->update_order_invoice_meta($order, $invoice_id, $invoice_uuid);
				
				// Generate PDF
				$this->generate_invoice_pdf(array_merge($invoice, array('db_id' => $invoice_db_id)));
				
				// Trigger usage tracking hook
				do_action('edm_efatura_invoice_created', 
					isset($invoice['uuid']) ? $invoice['uuid'] : '', 
					$order_id, 
					$invoice
				);
				
				// Return invoice data
				return array_merge($invoice, array('db_id' => $invoice_db_id));
			} else {
				// Check if this is a duplicate invoice error from GIB
				$is_duplicate_error = false;
				$error_message = isset($result['message']) ? $result['message'] : 'Unknown error';
				if (strpos($error_message, 'Daha önce gönderdiğiniz bir faturayı tekrar göndermeye çalıştınız') !== false ||
				    strpos($error_message, 'duplicate') !== false ||
				    (isset($result['error_code']) && $result['error_code'] == '11048')) {
					$is_duplicate_error = true;
				}

				if ($is_duplicate_error) {
					// Check if we have this invoice in our database
					$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
					if ($existing_invoice && (!isset($existing_invoice['status']) || $existing_invoice['status'] !== 'CANCELLED')) {
						// We have a valid invoice, return it instead of failing
						$this->logger->info('Duplicate invoice error from GIB, but invoice exists in database', array(
							'order_id' => $order_id,
							'invoice_id' => $existing_invoice['id'],
							'uuid' => $existing_invoice['uuid']
						));
						return $existing_invoice;
					}
				}

				// Log error
				$this->logger->error('Failed to generate invoice for order', array(
					'order_id' => $order_id,
					'error' => $error_message
				));

				return false;
			}
		} catch (Exception $e) {
			$this->logger->error('Exception while generating invoice', array(
				'order_id' => $order_id,
				'error' => $e->getMessage()
			));
			
			return false;
		}
	}

	/**
	 * Translate invoice status from English to Turkish
	 *
	 * @param string $status Invoice status
	 * @return string Translated status
	 */
	private function translate_invoice_status($status) {
		$translations = array(
			'SEND - SUCCEED - SUCCEED' => __('Gönderildi - Başarılı - Başarılı', 'edm-efatura-premium'),
			'SEND - SUCCEED' => __('Gönderildi - Başarılı', 'edm-efatura-premium'),
			'PACKAGE - PROCESSING' => __('Paket - İşleniyor', 'edm-efatura-premium'),
			'CANCELLED' => __('İptal Edildi', 'edm-efatura-premium'),
			'DRAFT' => __('Taslak', 'edm-efatura-premium'),
			'PENDING' => __('Beklemede', 'edm-efatura-premium'),
			'FAILED' => __('Başarısız', 'edm-efatura-premium'),
			'PROCESSING' => __('İşleniyor', 'edm-efatura-premium'),
			'SUCCEED' => __('Başarılı', 'edm-efatura-premium'),
		);

		return isset($translations[$status]) ? $translations[$status] : $status;
	}

	/**
	 * Display invoice download button on order details page
	 *
	 * @param WC_Order $order Order object
	 */
	public function display_invoice_download_button($order) {
		$order_id = $order->get_id();

		// Get invoice
		$invoice = $this->db->get_invoice_by_order_id($order_id);

		// If no invoice or cancelled, don't show button
		if (!$invoice || (isset($invoice['status']) && $invoice['status'] === 'CANCELLED')) {
			return;
		}

		// Check if PDF exists
		$pdf_exists = !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path']);

		if (!$pdf_exists) {
			return;
		}

		// Generate nonce for security
		$nonce = wp_create_nonce('edm_download_invoice_' . $order_id);

		echo '<section class="woocommerce-order-edm-invoice" style="margin-top: 30px; padding: 20px; background: #f8f9fa; border: 1px solid #e3e8ee; border-radius: 8px;">';
		echo '<h2 style="margin-top: 0; color: #2c3e50; font-size: 20px; border-bottom: 2px solid #3498db; padding-bottom: 10px;">' . esc_html__('E-Invoice', 'edm-efatura-premium') . '</h2>';
		echo '<div style="display: flex; align-items: center; justify-content: space-between; flex-wrap: wrap; gap: 15px;">';
		echo '<div style="flex: 1; min-width: 200px;">';
		echo '<p style="margin: 5px 0; color: #555;"><strong>' . esc_html__('Invoice ID:', 'edm-efatura-premium') . '</strong> ' . esc_html($invoice['id']) . '</p>';

		if (!empty($invoice['created_at'])) {
			echo '<p style="margin: 5px 0; color: #555;"><strong>' . esc_html__('Date:', 'edm-efatura-premium') . '</strong> ' . esc_html(date_i18n(get_option('date_format'), strtotime($invoice['created_at']))) . '</p>';
		}

		echo '</div>';
		echo '<div>';
		echo '<a href="' . esc_url(add_query_arg(array(
			'action' => 'edm_download_invoice_pdf',
			'order_id' => $order_id,
			'nonce' => $nonce
		), admin_url('admin-ajax.php'))) . '" class="button edm-download-invoice-btn" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 12px 30px; border-radius: 6px; text-decoration: none; display: inline-flex; align-items: center; gap: 8px; font-weight: 600; box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4); transition: all 0.3s ease; border: none;">';
		echo '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="7 10 12 15 17 10"></polyline><line x1="12" y1="15" x2="12" y2="3"></line></svg>';
		echo esc_html__('Download PDF Invoice', 'edm-efatura-premium');
		echo '</a>';
		echo '</div>';
		echo '</div>';
		echo '</section>';

		// Add hover effect with inline style
		echo '<style>
		.edm-download-invoice-btn:hover {
			transform: translateY(-2px);
			box-shadow: 0 6px 20px rgba(102, 126, 234, 0.6) !important;
		}
		</style>';
	}

	/**
	 * Add invoice download action to my orders page
	 *
	 * @param array $actions Existing actions
	 * @param WC_Order $order Order object
	 * @return array Modified actions
	 */
	public function add_invoice_download_action($actions, $order) {
		$order_id = $order->get_id();

		// Get invoice
		$invoice = $this->db->get_invoice_by_order_id($order_id);

		// If no invoice or cancelled, don't add action
		if (!$invoice || (isset($invoice['status']) && $invoice['status'] === 'CANCELLED')) {
			return $actions;
		}

		// Check if PDF exists
		$pdf_exists = !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path']);

		if (!$pdf_exists) {
			return $actions;
		}

		// Generate nonce for security
		$nonce = wp_create_nonce('edm_download_invoice_' . $order_id);

		// Add download action
		$actions['edm_download_invoice'] = array(
			'url' => add_query_arg(array(
				'action' => 'edm_download_invoice_pdf',
				'order_id' => $order_id,
				'nonce' => $nonce
			), admin_url('admin-ajax.php')),
			'name' => __('Download Invoice', 'edm-efatura-premium')
		);

		return $actions;
	}

	/**
	 * Handle AJAX request to download invoice PDF
	 */
	public function ajax_download_invoice_pdf() {
		// Get order ID and nonce
		$order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;
		$nonce = isset($_GET['nonce']) ? sanitize_text_field($_GET['nonce']) : '';

		// Verify nonce
		if (!wp_verify_nonce($nonce, 'edm_download_invoice_' . $order_id)) {
			wp_die(__('Security check failed.', 'edm-efatura-premium'));
		}

		// Get order
		$order = wc_get_order($order_id);
		if (!$order) {
			wp_die(__('Order not found.', 'edm-efatura-premium'));
		}

		// Check if user has permission to view this order
		if (!current_user_can('manage_woocommerce') && get_current_user_id() !== $order->get_customer_id()) {
			wp_die(__('You do not have permission to download this invoice.', 'edm-efatura-premium'));
		}

		// Get invoice
		$invoice = $this->db->get_invoice_by_order_id($order_id);

		if (!$invoice) {
			wp_die(__('Invoice not found.', 'edm-efatura-premium'));
		}

		// Check if cancelled
		if (isset($invoice['status']) && $invoice['status'] === 'CANCELLED') {
			wp_die(__('This invoice has been cancelled.', 'edm-efatura-premium'));
		}

		// Check if PDF exists
		if (empty($invoice['pdf_path']) || !file_exists($invoice['pdf_path'])) {
			// Try to generate PDF
			$this->generate_invoice_pdf($invoice);

			// Reload invoice from database
			$invoice = $this->db->get_invoice_by_order_id($order_id);

			if (empty($invoice['pdf_path']) || !file_exists($invoice['pdf_path'])) {
				wp_die(__('Invoice PDF not found. Please contact support.', 'edm-efatura-premium'));
			}
		}

		// Set headers for PDF download
		header('Content-Type: application/pdf');
		header('Content-Disposition: attachment; filename="fatura-' . $invoice['id'] . '.pdf"');
		header('Content-Length: ' . filesize($invoice['pdf_path']));
		header('Cache-Control: private, max-age=0, must-revalidate');
		header('Pragma: public');

		// Output PDF
		readfile($invoice['pdf_path']);
		exit;
	}
}