<?php
/**
 * EDM E-Fatura Premium WooCommerce Class
 *
 * @package EDM_Efatura_Premium
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class for WooCommerce integration
 */
/**
 * Class for WooCommerce integration
 */
class EDM_Efatura_Premium_WooCommerce {
    /**
     * API instance
     *
     * @var EDM_Efatura_Premium_API
     */
    private $api;
    
    /**
     * Logger instance
     *
     * @var EDM_Efatura_Premium_Logger
     */
    private $logger;
    
    /**
     * DB instance
     *
     * @var EDM_Efatura_Premium_DB
     */
    private $db;
    
    /**
     * Settings
     *
     * @var array
     */
    private $settings;
    
    /**
     * Order handler instance
     *
     * @var EDM_Efatura_Premium_Order_Handler
     */
    private $order_handler;
    
    
    
    /**
     * Email instance
     *
     * @var EDM_Efatura_Premium_Email
     */
    private $email;
    
    /**
     * Constructor
     *
     * @param EDM_Efatura_Premium_API $api API instance
     */
    public function __construct($api) {
		$this->api = $api;
		$this->logger = new EDM_Efatura_Premium_Logger();
		$this->db = new EDM_Efatura_Premium_DB();
		$this->settings = get_option('edm_efatura_premium_settings', array());
		
		// Initialize sub-modules
		$this->order_handler = new EDM_Efatura_Premium_Order_Handler($api);
		
		// Email handler moved to main plugin only to prevent duplicates
		
		// Register AJAX handlers
		add_action('wp_ajax_edm_save_invoice_info', array($this, 'ajax_save_invoice_info'));
		
		// Add JavaScript for cancel functionality
		// JavaScript for cancel functionality is handled by admin.js
		
		// Add product meta box for unit code
		add_action('add_meta_boxes', array($this, 'add_product_unit_code_field'));
		add_action('save_post_product', array($this, 'save_product_unit_code'));
		
		// Add exception code field to product categories
		add_action('product_cat_add_form_fields', array($this, 'add_product_category_exception_field'), 10, 1);
		add_action('product_cat_edit_form_fields', array($this, 'edit_product_category_exception_field'), 10, 1);
		add_action('created_product_cat', array($this, 'save_product_category_exception_field'), 10, 2);
		add_action('edited_product_cat', array($this, 'save_product_category_exception_field'), 10, 1);
		
		// Add column to orders list
		add_filter('manage_edit-shop_order_columns', array($this, 'add_invoice_column'), 20);
		add_action('manage_shop_order_posts_custom_column', array($this, 'display_invoice_column'), 20, 2);
		add_action('wp_ajax_edm_generate_invoice_for_order', array($this, 'ajax_generate_invoice_for_order'));
		// Add column to HPOS orders list if available
		if (class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled()) {
			add_filter('woocommerce_shop_order_list_table_columns', array($this, 'add_invoice_column'), 20);
			add_action('woocommerce_shop_order_list_table_custom_column', array($this, 'display_invoice_column_hpos'), 20, 2);
		}
	}
    /**
	 * Get order handler instance
	 * 
	 * @return EDM_Efatura_Premium_Order_Handler
	 */
	public function get_order_handler() {
		return $this->order_handler;
	}
	/**
	 * Add invoice column to orders table
	 *
	 * @param array $columns Existing columns
	 * @return array Modified columns
	 */
	public function add_invoice_column($columns) {
		$new_columns = array();
		
		// Insert the column after the order status column
		foreach ($columns as $key => $value) {
			$new_columns[$key] = $value;
			
			if ($key === 'order_status') {
				$new_columns['edm_invoice'] = __('E-Invoice', 'edm-efatura-premium');
			}
		}
		
		// If order_status column doesn't exist, add it at the end
		if (!isset($new_columns['edm_invoice'])) {
			$new_columns['edm_invoice'] = __('E-Invoice', 'edm-efatura-premium');
		}
		
		return $new_columns;
	}
    
    /**
	 * Display invoice column for classic post table
	 *
	 * @param string $column Column name
	 * @param int $post_id Post ID
	 */
	public function display_invoice_column($column, $post_id) {
    if ($column !== 'edm_invoice') {
        return;
    }
    
    // Get invoice for this order
    $invoice = $this->db->get_invoice_by_order_id($post_id);
    
    if ($invoice) {
        // Check if invoice is cancelled
        $is_cancelled = false;
        if (isset($invoice['status'])) {
            $status = strtoupper($invoice['status']);
            $is_cancelled = ($status === 'CANCELLED' || strpos($status, 'CANCELLED') !== false);
            
            // Delete PDF file if invoice is cancelled
            if ($is_cancelled && !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path'])) {
                // Delete the file
                @unlink($invoice['pdf_path']);
                
                // Update the invoice record to remove the PDF path
                $invoice['pdf_path'] = '';
                $this->db->save_invoice($invoice);
            }
        }
        
        // Get invoice status and determine if it's a successful status
        $status = isset($invoice['status']) ? strtoupper($invoice['status']) : '';
        $is_success_status = (strpos($status, 'PACKAGE - PROCESSING') !== false || strpos($status, 'SEND - SUCCEED') !== false);
        
        // Display invoice status with appropriate styling
        echo '<span class="edm-status ' . ($is_success_status ? 'edm-status-success' : 'edm-status-' . esc_attr(strtolower($status))) . '">' . esc_html($status) . '</span><br>';
        
        // Display appropriate button based on status
        if ($is_success_status && !$is_cancelled) {
            // Display PDF link or generate button for successful invoices
            if (!empty($invoice['pdf_path']) && file_exists($invoice['pdf_path'])) {
                // Create PDF download URL
                $pdf_url = add_query_arg(array(
                    'edm_invoice_download' => $invoice['uuid'],
                    'nonce' => wp_create_nonce('edm_invoice_download_' . $invoice['uuid'])
                ), admin_url('admin.php?page=edm-efatura-premium-list'));
                
                echo '<a href="' . esc_url($pdf_url) . '" class="button button-small button-primary" target="_blank">' . esc_html__('Download PDF', 'edm-efatura-premium') . '</a>';
                echo '<button type="button" class="button button-small button-secondary edm-cancel-invoice" data-uuid="' . esc_attr($invoice['uuid']) . '" style="margin-top: 2px; display: block;">' . esc_html__('Cancel Invoice', 'edm-efatura-premium') . '</button>';
            } else {
                echo '<button type="button" class="button button-small button-primary edm-generate-pdf" data-uuid="' . esc_attr($invoice['uuid']) . '">' . esc_html__('Generate PDF', 'edm-efatura-premium') . '</button>';
                echo '<button type="button" class="button button-small button-secondary edm-cancel-invoice" data-uuid="' . esc_attr($invoice['uuid']) . '" style="margin-top: 2px; display: block;">' . esc_html__('Cancel Invoice', 'edm-efatura-premium') . '</button>';
            }
        } else {
            // For non-successful statuses (including cancelled), show both logs and generate invoice buttons
            echo '<a href="' . esc_url(admin_url('admin.php?page=edm-efatura-premium-logs')) . '" class="button button-small">' . esc_html__('View Logs', 'edm-efatura-premium') . '</a><br>';
            echo '<button type="button" class="button button-small edm-generate-invoice-list" data-order-id="' . esc_attr($post_id) . '" style="margin-top: 2px;">' . esc_html__('Generate Invoice', 'edm-efatura-premium') . '</button>';
            
            // Add results container
            echo '<div class="edm-generate-result" data-order="' . esc_attr($post_id) . '" style="display:none;margin-top:5px;"></div>';
        }
    } else {
        // Display generate invoice button
        echo '<button type="button" class="button button-small edm-generate-invoice-list" data-order-id="' . esc_attr($post_id) . '">' . esc_html__('Generate Invoice', 'edm-efatura-premium') . '</button>';
        
        // Add results container
        echo '<div class="edm-generate-result" data-order="' . esc_attr($post_id) . '" style="display:none;margin-top:5px;"></div>';
    }
}
	
	/**
	 * Direct invoice generation without using cron
	 * This will be triggered by our custom cron checker
	 *
	 * @param int $order_id Order ID
	 * @param string $schedule_key Scheduled key for verification
	 * @return bool Success status
	 */
	public function direct_generate_invoice_for_order($order_id, $schedule_key) {
		// Get order
		$order = wc_get_order($order_id);
		if (!$order) {
			$this->logger->error('Order not found for invoice generation', array(
				'order_id' => $order_id
			));
			return false;
		}
		
		// HPOS compatibility check
		$hpos_enabled = class_exists('\Automattic\WooCommerce\Utilities\OrderUtil') && 
			\Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
		
		// Verify the schedule key matches to prevent unauthorized execution
		$stored_key = '';
		if ($hpos_enabled) {
			$stored_key = $order->get_meta('edm_invoice_schedule_key', true);
		} else {
			$stored_key = get_post_meta($order_id, 'edm_invoice_schedule_key', true);
		}
		
		$this->logger->info('Checking schedule key', array(
			'order_id' => $order_id,
			'provided_key' => $schedule_key,
			'stored_key' => $stored_key
		));
		
		if ($stored_key !== $schedule_key) {
			// Keys don't match - check if key is empty (might have been cleaned up)
			if (empty($stored_key)) {
				$this->logger->info('Schedule key is empty, checking if invoice exists', array(
					'order_id' => $order_id
				));
				// Check if invoice already exists in database
				$order_handler = $this->get_order_handler();
				if ($order_handler && isset($order_handler->db)) {
					$existing_invoice = $order_handler->db->get_invoice_by_order_id($order_id);
					if ($existing_invoice && (!isset($existing_invoice['status']) || $existing_invoice['status'] !== 'CANCELLED')) {
						$this->logger->info('Invoice already exists, marking as sent', array(
							'order_id' => $order_id,
							'invoice_id' => $existing_invoice['id']
						));
						// Mark as sent
						if ($hpos_enabled) {
							$order->update_meta_data('edm_invoice_email_sent', 'yes');
							$order->save();
						} else {
							update_post_meta($order_id, 'edm_invoice_email_sent', 'yes');
						}
						return true;
					}
				}
			}
			// Keys don't match, abort
			$this->logger->error('Invalid schedule key for invoice generation', array(
				'order_id' => $order_id
			));
			return false;
		}

		// Check if already processed
		$invoice_sent = '';
		if ($hpos_enabled) {
			$invoice_sent = $order->get_meta('edm_invoice_email_sent', true);
		} else {
			$invoice_sent = get_post_meta($order_id, 'edm_invoice_email_sent', true);
		}

		if ($invoice_sent === 'yes') {
			// Already processed
			$this->logger->info('Invoice already sent for order', array(
				'order_id' => $order_id
			));
			return true;
		}
		
		// Get order handler
		$order_handler = $this->get_order_handler();
		if (!$order_handler) {
			$this->logger->error('Order handler not available for invoice generation', array(
				'order_id' => $order_id
			));
			return false;
		}
		
		// Generate invoice
		$this->logger->info('Generating invoice for order', array(
			'order_id' => $order_id
		));
		
		$invoice = $order_handler->generate_invoice_for_order($order);
		
		if ($invoice) {
			// Generate PDF if needed
			if (empty($invoice['pdf_path']) || !file_exists($invoice['pdf_path'])) {
				$this->logger->info('Generating PDF for invoice', array(
					'order_id' => $order_id,
					'invoice_id' => isset($invoice['id']) ? $invoice['id'] : 'unknown'
				));
				$order_handler->generate_invoice_pdf($invoice);
			}
			
			// Send email directly
			$this->logger->info('Sending invoice email', array(
				'order_id' => $order_id,
				'invoice_id' => isset($invoice['id']) ? $invoice['id'] : 'unknown'
			));
			
			$email_sent = $order_handler->send_direct_invoice_email($order, $invoice);
			
			if ($email_sent) {
				// Mark as sent
				if ($hpos_enabled) {
					$order->update_meta_data('edm_invoice_email_sent', 'yes');
					$order->update_meta_data('edm_invoice_email_sent_date', current_time('mysql'));
					$order->delete_meta_data('edm_invoice_scheduled');
					$order->delete_meta_data('edm_invoice_schedule_key');
					$order->save();
				} else {
					update_post_meta($order_id, 'edm_invoice_email_sent', 'yes');
					update_post_meta($order_id, 'edm_invoice_email_sent_date', current_time('mysql'));
					delete_post_meta($order_id, 'edm_invoice_scheduled');
					delete_post_meta($order_id, 'edm_invoice_schedule_key');
				}
				
				// Add order note
				$order->add_order_note(
					sprintf(
						__('EDM E-invoice generated successfully and sent to customer. Invoice ID: %s', 'edm-efatura-premium'),
						isset($invoice['id']) ? $invoice['id'] : 'N/A'
					)
				);
				
				$this->logger->info('Invoice successfully generated and sent', array(
					'order_id' => $order_id,
					'invoice_id' => isset($invoice['id']) ? $invoice['id'] : 'unknown'
				));
				
				return true;
			} else {
				$this->logger->error('Failed to send invoice email', array(
					'order_id' => $order_id
				));
			}
		} else {
			$this->logger->error('Failed to generate invoice', array(
				'order_id' => $order_id
			));
		}
		
		// Failed to generate or send
		$order->add_order_note(
			__('EDM E-invoice generation failed. Please check logs for details.', 'edm-efatura-premium')
		);
		
		return false;
	}
	
	
	/**
	 * Add exception code field to product categories
	 */
	public function add_product_category_exception_field() {
		// Get all exception codes
		$exception_codes = $this->get_exception_codes();
		
		?>
		<div class="form-field">
			<label for="edm_exception_code"><?php _e('E-Invoice Exception Code', 'edm-efatura-premium'); ?></label>
			<select name="edm_exception_code" id="edm_exception_code">
				<option value=""><?php _e('None', 'edm-efatura-premium'); ?></option>
				<?php foreach ($exception_codes as $code => $label) : ?>
					<option value="<?php echo esc_attr($code); ?>"><?php echo esc_html($label); ?></option>
				<?php endforeach; ?>
			</select>
			<p class="description"><?php _e('Select exception code for products in this category.', 'edm-efatura-premium'); ?></p>
		</div>
		<?php
	}
	
	
	
	/**
	 * Edit exception code field in product categories
	 * 
	 * @param WP_Term $term Term object
	 */
	public function edit_product_category_exception_field($term) {
		// Get all exception codes
		$exception_codes = $this->get_exception_codes();
		
		// Get saved values
		$edm_tax_type = get_term_meta($term->term_id, 'edm_tax_type', true);
		$edm_exception_code = get_term_meta($term->term_id, 'edm_exception_code', true);
		$edm_tax_rate = get_term_meta($term->term_id, 'edm_tax_rate', true);
		
		// Set defaults
		$edm_tax_type = $edm_tax_type ? $edm_tax_type : '';
		$edm_tax_rate = $edm_tax_rate ? $edm_tax_rate : '';
		
		?>
		<tr class="form-field">
			<th scope="row" valign="top">
				<label for="edm_tax_type"><?php _e('E-Invoice Tax Settings', 'edm-efatura-premium'); ?></label>
			</th>
			<td>
				<p><strong><?php _e('Tax Type', 'edm-efatura-premium'); ?></strong></p>
				<select name="edm_tax_type" id="edm_tax_type" style="margin-bottom: 10px;">
					<option value=""><?php _e('Use Global Settings', 'edm-efatura-premium'); ?></option>
					<option value="exception" <?php selected($edm_tax_type, 'exception'); ?>><?php _e('Tax Exception (İstisna)', 'edm-efatura-premium'); ?></option>
					<option value="taxed" <?php selected($edm_tax_type, 'taxed'); ?>><?php _e('Taxed Sales (KDV)', 'edm-efatura-premium'); ?></option>
				</select>
				
				<div id="edm_exception_section" style="<?php echo ($edm_tax_type === 'exception') ? 'display: block;' : 'display: none;'; ?>">
					<p><strong><?php _e('Exception Code', 'edm-efatura-premium'); ?></strong></p>
					<select name="edm_exception_code" id="edm_exception_code">
						<option value=""><?php _e('Select Exception Code', 'edm-efatura-premium'); ?></option>
						<?php foreach ($exception_codes as $code => $label) : ?>
							<option value="<?php echo esc_attr($code); ?>" <?php selected($edm_exception_code, $code); ?>><?php echo esc_html($label); ?></option>
						<?php endforeach; ?>
					</select>
				</div>
				
				<div id="edm_tax_section" style="<?php echo ($edm_tax_type === 'taxed') ? 'display: block;' : 'display: none;'; ?>">
					<p><strong><?php _e('Tax Rate (%)', 'edm-efatura-premium'); ?></strong></p>
					<input type="number" name="edm_tax_rate" id="edm_tax_rate" value="<?php echo esc_attr($edm_tax_rate); ?>" step="0.01" min="0" max="100" placeholder="18.00" />
				</div>
				
				<p class="description"><?php _e('Configure tax settings for products in this category. If not set, global plugin settings will be used.', 'edm-efatura-premium'); ?></p>
				
				<script type="text/javascript">
				jQuery(document).ready(function($) {
					$('#edm_tax_type').change(function() {
						var selected = $(this).val();
						if (selected === 'exception') {
							$('#edm_exception_section').show();
							$('#edm_tax_section').hide();
						} else if (selected === 'taxed') {
							$('#edm_exception_section').hide();
							$('#edm_tax_section').show();
						} else {
							$('#edm_exception_section').hide();
							$('#edm_tax_section').hide();
						}
					});
				});
				</script>
			</td>
		</tr>
		<?php
	}

	/**
	 * Save tax settings from product category form
	 * 
	 * @param int $term_id Term ID
	 * @param int $tt_id Term taxonomy ID
	 */
	public function save_product_category_exception_field($term_id, $tt_id = '') {
		// Save tax type
		if (isset($_POST['edm_tax_type'])) {
			update_term_meta($term_id, 'edm_tax_type', sanitize_text_field($_POST['edm_tax_type']));
		}
		
		// Save exception code
		if (isset($_POST['edm_exception_code'])) {
			update_term_meta($term_id, 'edm_exception_code', sanitize_text_field($_POST['edm_exception_code']));
		}
		
		// Save tax rate
		if (isset($_POST['edm_tax_rate'])) {
			$tax_rate = floatval($_POST['edm_tax_rate']);
			$tax_rate = max(0, min(100, $tax_rate)); // Ensure between 0 and 100
			update_term_meta($term_id, 'edm_tax_rate', $tax_rate);
		}
	}

	/**
	 * Get all available exception codes
	 * 
	 * @return array Exception codes with labels
	 */
	private function get_exception_codes() {
		return EDM_EFatura_Premium_Exception_Codes::get_formatted_exception_codes_for_forms();
	}
	
	/**
	 * Get category tax settings for a product
	 * 
	 * @param WC_Product $product Product object
	 * @return array|null Category tax settings or null if no category settings
	 */
	public function get_product_category_tax_settings($product) {
		$category_ids = $product->get_category_ids();
		
		// Debug log
		$logger = new EDM_Efatura_Premium_Logger();
		$logger->info('Getting category tax settings', array(
			'product_id' => $product->get_id(),
			'category_ids' => $category_ids
		));
		
		if (empty($category_ids)) {
			$logger->info('No category IDs found for product ' . $product->get_id());
			return null;
		}
		
		// Check each category for tax settings (first match wins)
		foreach ($category_ids as $category_id) {
			$tax_type = get_term_meta($category_id, 'edm_tax_type', true);
			$exception_code = get_term_meta($category_id, 'edm_exception_code', true);
			$tax_rate = get_term_meta($category_id, 'edm_tax_rate', true);
			
			// Debug log for each category
			$logger->info('Category meta check', array(
				'category_id' => $category_id,
				'tax_type' => $tax_type,
				'exception_code' => $exception_code,
				'tax_rate' => $tax_rate
			));
			
			if (!empty($tax_type)) {
				$settings = array(
					'tax_type' => $tax_type,
					'category_id' => $category_id,
					'category_name' => get_term($category_id)->name
				);
				
				if ($tax_type === 'exception') {
					$settings['exception_code'] = get_term_meta($category_id, 'edm_exception_code', true);
				} elseif ($tax_type === 'taxed') {
					$settings['tax_rate'] = get_term_meta($category_id, 'edm_tax_rate', true);
				}
				
				$logger->info('Category settings found', array(
					'category_id' => $category_id,
					'settings' => $settings
				));
				
				return $settings;
			}
		}
		
		$logger->info('No category tax settings found for product ' . $product->get_id());
		return null;
	}

	/**
	 * Display invoice column for HPOS table
	 *
	 * @param mixed $order WC_Order object for HPOS
	 * @param string $column Column name
	 */
	public function display_invoice_column_hpos($column, $order) {
    if ($column !== 'edm_invoice') {
        return;
    }
    
    $post_id = $order->get_id();
    
    // Get invoice for this order
    $invoice = $this->db->get_invoice_by_order_id($post_id);
    
    if ($invoice) {
        // Check if invoice is cancelled
        $is_cancelled = false;
        if (isset($invoice['status'])) {
            $status = strtoupper($invoice['status']);
            $is_cancelled = ($status === 'CANCELLED' || strpos($status, 'CANCELLED') !== false);
            
            // Delete PDF file if invoice is cancelled
            if ($is_cancelled && !empty($invoice['pdf_path']) && file_exists($invoice['pdf_path'])) {
                // Delete the file
                @unlink($invoice['pdf_path']);
                
                // Update the invoice record to remove the PDF path
                $invoice['pdf_path'] = '';
                $this->db->save_invoice($invoice);
            }
        }
        
        // Get invoice status and determine if it's a successful status
        $status = isset($invoice['status']) ? strtoupper($invoice['status']) : '';
        $is_success_status = (strpos($status, 'PACKAGE - PROCESSING') !== false || strpos($status, 'SEND - SUCCEED') !== false);
        
        // Display invoice status with appropriate styling
        echo '<span class="edm-status ' . ($is_success_status ? 'edm-status-success' : 'edm-status-' . esc_attr(strtolower($status))) . '">' . esc_html($status) . '</span><br>';
        
        // Display appropriate button based on status
        if ($is_success_status && !$is_cancelled) {
            // Display PDF link or generate button for successful invoices
            if (!empty($invoice['pdf_path']) && file_exists($invoice['pdf_path'])) {
                // Create PDF download URL
                $pdf_url = add_query_arg(array(
                    'edm_invoice_download' => $invoice['uuid'],
                    'nonce' => wp_create_nonce('edm_invoice_download_' . $invoice['uuid'])
                ), admin_url('admin.php?page=edm-efatura-premium-list'));
                
                echo '<a href="' . esc_url($pdf_url) . '" class="button button-small button-primary" target="_blank">' . esc_html__('Download PDF', 'edm-efatura-premium') . '</a>';
                echo '<button type="button" class="button button-small button-secondary edm-cancel-invoice" data-uuid="' . esc_attr($invoice['uuid']) . '" style="margin-top: 2px; display: block;">' . esc_html__('Cancel Invoice', 'edm-efatura-premium') . '</button>';
            } else {
                echo '<button type="button" class="button button-small button-primary edm-generate-pdf" data-uuid="' . esc_attr($invoice['uuid']) . '">' . esc_html__('Generate PDF', 'edm-efatura-premium') . '</button>';
                echo '<button type="button" class="button button-small button-secondary edm-cancel-invoice" data-uuid="' . esc_attr($invoice['uuid']) . '" style="margin-top: 2px; display: block;">' . esc_html__('Cancel Invoice', 'edm-efatura-premium') . '</button>';
            }
        } else {
            // For non-successful statuses (including cancelled), show both logs and generate invoice buttons
            echo '<a href="' . esc_url(admin_url('admin.php?page=edm-efatura-premium-logs')) . '" class="button button-small">' . esc_html__('View Logs', 'edm-efatura-premium') . '</a><br>';
            echo '<button type="button" class="button button-small edm-generate-invoice-list" data-order-id="' . esc_attr($post_id) . '" style="margin-top: 2px;">' . esc_html__('Generate Invoice', 'edm-efatura-premium') . '</button>';
            
            // Add results container
            echo '<div class="edm-generate-result" data-order="' . esc_attr($post_id) . '" style="display:none;margin-top:5px;"></div>';
        }
    } else {
        // Display generate invoice button
        echo '<button type="button" class="button button-small edm-generate-invoice-list" data-order-id="' . esc_attr($post_id) . '">' . esc_html__('Generate Invoice', 'edm-efatura-premium') . '</button>';
        
        // Add results container
        echo '<div class="edm-generate-result" data-order="' . esc_attr($post_id) . '" style="display:none;margin-top:5px;"></div>';
    }
}
	
	
	
    /**
     * Save invoice fields to order meta
     *
     * @param int $order_id Order ID
     */
    public function save_invoice_fields($order_id) {
    // Get order object (HPOS-compatible way)
    $order = wc_get_order($order_id);
    
    if (!$order) {
        return;
    }
    
    // Save invoice type
    if (!empty($_POST['edm_invoice_type'])) {
        $invoice_type = sanitize_text_field($_POST['edm_invoice_type']);
        $order->update_meta_data('edm_invoice_type', $invoice_type);
        
        // Save TC number for individuals
        if ($invoice_type === 'individual' && !empty($_POST['edm_tc_number'])) {
            $tc_number = sanitize_text_field($_POST['edm_tc_number']);
            
            // Validate TC number
            if (!EDM_Efatura_Premium_Security::validate_tc_number($tc_number)) {
                wc_add_notice(__('Invalid TC number format.', 'edm-efatura-premium'), 'error');
                return;
            }
            
            $order->update_meta_data('edm_tc_number', $tc_number);
        }
        
        // Save corporate fields
        if ($invoice_type === 'corporate') {
            if (!empty($_POST['edm_company_name'])) {
                $company_name = EDM_Efatura_Premium_Security::sanitize_turkish_text($_POST['edm_company_name']);
                $order->update_meta_data('edm_company_name', $company_name);
            }
            
            if (!empty($_POST['edm_tax_number'])) {
                $tax_number = sanitize_text_field($_POST['edm_tax_number']);
                
                // Validate VKN/Tax number
                if (!EDM_Efatura_Premium_Security::validate_vkn($tax_number)) {
                    wc_add_notice(__('Invalid tax number format.', 'edm-efatura-premium'), 'error');
                    return;
                }
                
                $order->update_meta_data('edm_tax_number', $tax_number);
            }
            
            if (!empty($_POST['edm_tax_office'])) {
                $tax_office = EDM_Efatura_Premium_Security::sanitize_turkish_text($_POST['edm_tax_office']);
                $order->update_meta_data('edm_tax_office', $tax_office);
            }
        }
        
        // Save changes
        $order->save();
    }
}
    
    /**
     * Maybe generate invoice when order status is completed
     *
     * @param int $order_id Order ID
     */
    public function maybe_generate_invoice($order_id) {
		// Düzeltme: Order ID yerine WC_Order nesnesi oluştur
		$order = wc_get_order($order_id);
		if (!$order) {
			$this->logger->error('Order not found', array('order_id' => $order_id));
			return;
		}
		
		// Check if invoice already exists to prevent duplicate processing
		$existing_invoice = $this->db->get_invoice_by_order_id($order_id);
		if ($existing_invoice) {
			$this->logger->info('Invoice already exists for this order, skipping duplicate generation', array(
				'order_id' => $order_id,
				'invoice_id' => $existing_invoice['invoice_id'],
				'hook' => 'woocommerce_order_status_completed'
			));
			return;
		}

		// Check if order total is zero or the order has no items with value
		if ($order->get_total() <= 0) {
			$this->logger->info('Skipping invoice generation for zero-value order', array(
				'order_id' => $order_id,
				'total' => $order->get_total()
			));
			
			// Add order note
			$order->add_order_note(
				__('EDM E-invoice generation skipped. Order has zero value.', 'edm-efatura-premium')
			);
			
			return;
		}
		
		// Check if auto-generate is enabled
		if (isset($this->settings['auto_generate_invoice']) && $this->settings['auto_generate_invoice'] === 'yes') {
			// ✅ Use ORDER-HANDLER's handle_order_status_change which includes delay logic
			$this->order_handler->handle_order_status_change($order_id, 'any', 'completed', $order);
		}
	}
    
   
  

	/**
	 * Add unit code meta box to product edit page
	 */
	public function add_product_unit_code_field() {
		add_meta_box(
			'edm_product_unit_code',
			__('E-Invoice Unit Code', 'edm-efatura-premium'),
			array($this, 'render_product_unit_code_field'),
			'product',
			'side',
			'default'
		);
	}
	/**
	 * Render unit code meta box content
	 * 
	 * @param WP_Post $post Post object
	 */
	public function render_product_unit_code_field($post) {
		// Get current value
		$unit_code = get_post_meta($post->ID, '_unit_code', true);
		
		// Get default unit code from settings
		$settings = get_option('edm_efatura_premium_settings', array());
		$default_unit_code = isset($settings['default_unit_code']) ? $settings['default_unit_code'] : 'C62';
		
		// If not set, use default
		if (empty($unit_code)) {
			$unit_code = $default_unit_code;
		}
		
		// Create field
		echo '<p>' . __('Select the unit code to be used in e-invoices for this product.', 'edm-efatura-premium') . '</p>';
		echo '<select name="edm_unit_code" id="edm_unit_code" class="select">';
		echo '<option value="C62"' . selected($unit_code, 'C62', false) . '>C62 - ' . __('Unit/Piece', 'edm-efatura-premium') . '</option>';
		echo '<option value="KGM"' . selected($unit_code, 'KGM', false) . '>KGM - ' . __('Kilogram', 'edm-efatura-premium') . '</option>';
		echo '<option value="LTR"' . selected($unit_code, 'LTR', false) . '>LTR - ' . __('Liter', 'edm-efatura-premium') . '</option>';
		echo '<option value="MTR"' . selected($unit_code, 'MTR', false) . '>MTR - ' . __('Meter', 'edm-efatura-premium') . '</option>';
		echo '<option value="MTK"' . selected($unit_code, 'MTK', false) . '>MTK - ' . __('Square Meter', 'edm-efatura-premium') . '</option>';
		echo '<option value="MTQ"' . selected($unit_code, 'MTQ', false) . '>MTQ - ' . __('Cubic Meter', 'edm-efatura-premium') . '</option>';
		echo '<option value="GRM"' . selected($unit_code, 'GRM', false) . '>GRM - ' . __('Gram', 'edm-efatura-premium') . '</option>';
		echo '<option value="KWH"' . selected($unit_code, 'KWH', false) . '>KWH - ' . __('Kilowatt Hour', 'edm-efatura-premium') . '</option>';
		echo '<option value="SET"' . selected($unit_code, 'SET', false) . '>SET - ' . __('Set', 'edm-efatura-premium') . '</option>';
		// Diğer birim kodları da eklenebilir
		echo '</select>';
		
		// Nonce field for security
		wp_nonce_field('edm_save_unit_code', 'edm_unit_code_nonce');
	}
	
	
	/**
	 * Save unit code meta when product is saved
	 * 
	 * @param int $post_id Post ID
	 */
	public function save_product_unit_code($post_id) {
		// Check if nonce is set
		if (!isset($_POST['edm_unit_code_nonce'])) {
			return;
		}
		
		// Verify nonce
		if (!wp_verify_nonce($_POST['edm_unit_code_nonce'], 'edm_save_unit_code')) {
			return;
		}
		
		// If this is an autosave, our form has not been submitted, so we don't want to do anything
		if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
			return;
		}
		
		// Check user permissions
		if (!current_user_can('edit_post', $post_id)) {
			return;
		}
		
		// Check if our custom field is set and save it
		if (isset($_POST['edm_unit_code'])) {
			update_post_meta($post_id, '_unit_code', sanitize_text_field($_POST['edm_unit_code']));
		}
	}
	
    


    /**
     * AJAX handler for generating invoice for order
     */
    public function ajax_generate_invoice_for_order() {
        // Delegate to order handler
        if ($this->order_handler) {
            $this->order_handler->ajax_generate_invoice_for_order();
        } else {
            wp_send_json_error(__('Order handler not available.', 'edm-efatura-premium'));
        }
    }
    
    
    
}