<?php
/**
 * EDM E-Fatura Premium - Performance Benchmark Script
 * 
 * This script runs performance benchmarks to compare before/after optimization results
 * 
 * @package EDM_Efatura_Premium
 * @author  EDM Team
 * @version 1.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    define('ABSPATH', dirname(__FILE__) . '/../../../');
}

// Load WordPress
require_once ABSPATH . 'wp-config.php';

// Include required classes
require_once dirname(__FILE__) . '/includes/class-edm-efatura-premium-performance.php';
require_once dirname(__FILE__) . '/includes/class-edm-efatura-premium-cache.php';
require_once dirname(__FILE__) . '/includes/class-edm-efatura-premium-db.php';

/**
 * Performance Benchmark Class
 */
class EDM_Efatura_Premium_Benchmark {
    
    private $benchmark_results = array();
    private $iterations = 100;
    
    public function __construct($iterations = 100) {
        $this->iterations = $iterations;
        
        echo "<h1>EDM E-Fatura Premium - Performance Benchmark</h1>\n";
        echo "<p>Running performance benchmarks with {$this->iterations} iterations...</p>\n";
        
        // Initialize performance tracking
        EDM_Efatura_Premium_Performance::init();
    }
    
    /**
     * Run all benchmarks
     */
    public function run_benchmarks() {
        $this->benchmark_caching();
        $this->benchmark_database_queries();
        $this->benchmark_memory_operations();
        $this->benchmark_data_processing();
        $this->benchmark_file_operations();
        
        $this->display_benchmark_results();
    }
    
    /**
     * Benchmark caching performance
     */
    public function benchmark_caching() {
        echo "<h2>Benchmarking Caching Performance</h2>\n";
        
        $cache = new EDM_Efatura_Premium_Cache();
        
        // Benchmark cache writes
        $start_time = microtime(true);
        for ($i = 0; $i < $this->iterations; $i++) {
            $cache->set("benchmark_key_{$i}", array('data' => str_repeat('test', 100), 'id' => $i), 300);
        }
        $write_time = microtime(true) - $start_time;
        
        // Benchmark cache reads
        $start_time = microtime(true);
        $hits = 0;
        for ($i = 0; $i < $this->iterations; $i++) {
            $result = $cache->get("benchmark_key_{$i}");
            if ($result !== null) {
                $hits++;
            }
        }
        $read_time = microtime(true) - $start_time;
        
        // Cleanup
        for ($i = 0; $i < $this->iterations; $i++) {
            $cache->delete("benchmark_key_{$i}");
        }
        
        $this->benchmark_results['caching'] = array(
            'write_time' => $write_time,
            'read_time' => $read_time,
            'write_ops_per_sec' => $this->iterations / $write_time,
            'read_ops_per_sec' => $this->iterations / $read_time,
            'cache_hit_rate' => ($hits / $this->iterations) * 100,
            'avg_write_time_ms' => ($write_time / $this->iterations) * 1000,
            'avg_read_time_ms' => ($read_time / $this->iterations) * 1000
        );
        
        echo "<p>✓ Cache benchmark completed: " . round($this->benchmark_results['caching']['write_ops_per_sec'], 2) . " writes/sec, " . round($this->benchmark_results['caching']['read_ops_per_sec'], 2) . " reads/sec</p>\n";
    }
    
    /**
     * Benchmark database query performance
     */
    public function benchmark_database_queries() {
        echo "<h2>Benchmarking Database Query Performance</h2>\n";
        
        global $wpdb;
        $db = new EDM_Efatura_Premium_DB();
        
        // Benchmark simple queries
        $start_time = microtime(true);
        for ($i = 0; $i < $this->iterations; $i++) {
            $wpdb->get_results("SELECT option_name FROM {$wpdb->options} LIMIT 1");
        }
        $simple_query_time = microtime(true) - $start_time;
        
        // Benchmark optimized queries
        $start_time = microtime(true);
        for ($i = 0; $i < $this->iterations; $i++) {
            $query = "SELECT option_name FROM {$wpdb->options}";
            $optimized_query = EDM_Efatura_Premium_Performance::optimize_db_query($query);
            $wpdb->get_results($optimized_query);
        }
        $optimized_query_time = microtime(true) - $start_time;
        
        // Test cursor pagination performance
        $start_time = microtime(true);
        $cursor = 0;
        for ($i = 0; $i < 10; $i++) { // Fewer iterations for pagination
            $results = EDM_Efatura_Premium_Performance::cursor_paginate(
                $wpdb->options,
                array(),
                20,
                $cursor,
                'option_id'
            );
            if (!empty($results['data'])) {
                $cursor = $results['next_cursor'];
            }
        }
        $pagination_time = microtime(true) - $start_time;
        
        $this->benchmark_results['database'] = array(
            'simple_query_time' => $simple_query_time,
            'optimized_query_time' => $optimized_query_time,
            'pagination_time' => $pagination_time,
            'simple_ops_per_sec' => $this->iterations / $simple_query_time,
            'optimized_ops_per_sec' => $this->iterations / $optimized_query_time,
            'improvement_ratio' => $simple_query_time / $optimized_query_time,
            'avg_simple_time_ms' => ($simple_query_time / $this->iterations) * 1000,
            'avg_optimized_time_ms' => ($optimized_query_time / $this->iterations) * 1000
        );
        
        echo "<p>✓ Database benchmark completed: " . round($this->benchmark_results['database']['improvement_ratio'], 2) . "x improvement with optimization</p>\n";
    }
    
    /**
     * Benchmark memory operations
     */
    public function benchmark_memory_operations() {
        echo "<h2>Benchmarking Memory Operations</h2>\n";
        
        // Benchmark memory allocation without tracking
        $start_time = microtime(true);
        $start_memory = memory_get_usage();
        
        $arrays = array();
        for ($i = 0; $i < $this->iterations; $i++) {
            $arrays[] = array_fill(0, 1000, str_repeat('test', 10));
        }
        unset($arrays);
        
        $untracked_time = microtime(true) - $start_time;
        $untracked_memory = memory_get_usage() - $start_memory;
        
        // Benchmark memory allocation with performance tracking
        EDM_Efatura_Premium_Performance::prepare_for_large_operation('memory_benchmark');
        
        $start_time = microtime(true);
        $start_memory = memory_get_usage();
        
        $arrays = array();
        for ($i = 0; $i < $this->iterations; $i++) {
            EDM_Efatura_Premium_Performance::mark_memory("iter_{$i}");
            $arrays[] = array_fill(0, 1000, str_repeat('test', 10));
        }
        unset($arrays);
        
        $tracked_time = microtime(true) - $start_time;
        $tracked_memory = memory_get_usage() - $start_memory;
        
        EDM_Efatura_Premium_Performance::cleanup_after_large_operation('memory_benchmark');
        
        $this->benchmark_results['memory'] = array(
            'untracked_time' => $untracked_time,
            'tracked_time' => $tracked_time,
            'tracking_overhead' => (($tracked_time - $untracked_time) / $untracked_time) * 100,
            'memory_overhead' => $tracked_memory - $untracked_memory,
            'allocations_per_sec' => $this->iterations / $tracked_time
        );
        
        echo "<p>✓ Memory benchmark completed: " . round($this->benchmark_results['memory']['tracking_overhead'], 2) . "% tracking overhead</p>\n";
    }
    
    /**
     * Benchmark data processing
     */
    public function benchmark_data_processing() {
        echo "<h2>Benchmarking Data Processing</h2>\n";
        
        // Create test dataset
        $large_dataset = array();
        for ($i = 0; $i < $this->iterations * 10; $i++) {
            $large_dataset[] = array(
                'id' => $i,
                'name' => 'Item ' . $i,
                'data' => str_repeat('content', 50)
            );
        }
        
        // Benchmark regular array processing
        $processor = function($data) {
            return array_map(function($item) {
                return array(
                    'processed_id' => $item['id'] * 2,
                    'length' => strlen($item['data']),
                    'hash' => md5($item['name'])
                );
            }, $data);
        };
        
        $start_time = microtime(true);
        $regular_results = $processor($large_dataset);
        $regular_time = microtime(true) - $start_time;
        
        // Benchmark batch processing
        $start_time = microtime(true);
        $batch_results = EDM_Efatura_Premium_Performance::batch_process_array($large_dataset, $processor, 100);
        $batch_time = microtime(true) - $start_time;
        
        $this->benchmark_results['data_processing'] = array(
            'regular_time' => $regular_time,
            'batch_time' => $batch_time,
            'dataset_size' => count($large_dataset),
            'regular_items_per_sec' => count($large_dataset) / $regular_time,
            'batch_items_per_sec' => count($large_dataset) / $batch_time,
            'batch_vs_regular_ratio' => $regular_time / $batch_time,
            'memory_efficiency' => count($batch_results) === count($regular_results)
        );
        
        echo "<p>✓ Data processing benchmark completed: " . count($large_dataset) . " items processed</p>\n";
    }
    
    /**
     * Benchmark file operations
     */
    public function benchmark_file_operations() {
        echo "<h2>Benchmarking File Operations</h2>\n";
        
        // Create test data
        $test_data_small = str_repeat('test data ', 1000); // ~9KB
        $test_data_large = str_repeat('large test data ', 10000); // ~150KB
        
        // Benchmark small data streaming
        $start_time = microtime(true);
        ob_start();
        for ($i = 0; $i < 10; $i++) {
            EDM_Efatura_Premium_Performance::stream_large_data($test_data_small, 1024);
        }
        ob_end_clean();
        $small_stream_time = microtime(true) - $start_time;
        
        // Benchmark large data streaming
        $start_time = microtime(true);
        ob_start();
        EDM_Efatura_Premium_Performance::stream_large_data($test_data_large, 8192);
        ob_end_clean();
        $large_stream_time = microtime(true) - $start_time;
        
        // Benchmark CSV export
        $csv_test_data = array();
        for ($i = 0; $i < 1000; $i++) {
            $csv_test_data[] = array($i, 'Item ' . $i, rand(1000, 9999));
        }
        
        $start_time = microtime(true);
        $csv_result = EDM_Efatura_Premium_Performance::export_large_csv(
            $csv_test_data,
            'benchmark_test_' . time() . '.csv',
            array('ID', 'Name', 'Value')
        );
        $csv_time = microtime(true) - $start_time;
        
        // Cleanup
        if ($csv_result && file_exists($csv_result)) {
            unlink($csv_result);
        }
        
        $this->benchmark_results['file_operations'] = array(
            'small_stream_time' => $small_stream_time,
            'large_stream_time' => $large_stream_time,
            'csv_export_time' => $csv_time,
            'small_data_throughput' => (strlen($test_data_small) * 10) / $small_stream_time, // bytes per second
            'large_data_throughput' => strlen($test_data_large) / $large_stream_time,
            'csv_records_per_sec' => count($csv_test_data) / $csv_time
        );
        
        echo "<p>✓ File operations benchmark completed</p>\n";
    }
    
    /**
     * Display benchmark results
     */
    public function display_benchmark_results() {
        echo "<h2>Performance Benchmark Results</h2>\n";
        echo "<div style='background: #f9f9f9; padding: 20px; border: 1px solid #ddd; font-family: monospace;'>\n";
        
        // Caching Results
        if (isset($this->benchmark_results['caching'])) {
            $cache = $this->benchmark_results['caching'];
            echo "<h3>📦 Caching Performance</h3>\n";
            echo "<table style='border-collapse: collapse; width: 100%;'>\n";
            echo "<tr><th style='text-align: left; padding: 8px; border: 1px solid #ddd;'>Metric</th><th style='text-align: right; padding: 8px; border: 1px solid #ddd;'>Value</th></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Write Operations/sec</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($cache['write_ops_per_sec'], 2) . "</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Read Operations/sec</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($cache['read_ops_per_sec'], 2) . "</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Cache Hit Rate</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($cache['cache_hit_rate'], 2) . "%</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Avg Write Time</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($cache['avg_write_time_ms'], 3) . "ms</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Avg Read Time</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($cache['avg_read_time_ms'], 3) . "ms</td></tr>\n";
            echo "</table>\n";
        }
        
        // Database Results
        if (isset($this->benchmark_results['database'])) {
            $db = $this->benchmark_results['database'];
            echo "<h3>🗄️ Database Performance</h3>\n";
            echo "<table style='border-collapse: collapse; width: 100%;'>\n";
            echo "<tr><th style='text-align: left; padding: 8px; border: 1px solid #ddd;'>Metric</th><th style='text-align: right; padding: 8px; border: 1px solid #ddd;'>Value</th></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Simple Queries/sec</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($db['simple_ops_per_sec'], 2) . "</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Optimized Queries/sec</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($db['optimized_ops_per_sec'], 2) . "</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Performance Improvement</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($db['improvement_ratio'], 2) . "x</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Avg Simple Query Time</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($db['avg_simple_time_ms'], 3) . "ms</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Avg Optimized Query Time</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($db['avg_optimized_time_ms'], 3) . "ms</td></tr>\n";
            echo "</table>\n";
        }
        
        // Memory Results
        if (isset($this->benchmark_results['memory'])) {
            $mem = $this->benchmark_results['memory'];
            echo "<h3>💾 Memory Operations</h3>\n";
            echo "<table style='border-collapse: collapse; width: 100%;'>\n";
            echo "<tr><th style='text-align: left; padding: 8px; border: 1px solid #ddd;'>Metric</th><th style='text-align: right; padding: 8px; border: 1px solid #ddd;'>Value</th></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Allocations/sec</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($mem['allocations_per_sec'], 2) . "</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Tracking Overhead</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($mem['tracking_overhead'], 2) . "%</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Memory Overhead</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . size_format(abs($mem['memory_overhead'])) . "</td></tr>\n";
            echo "</table>\n";
        }
        
        // Data Processing Results
        if (isset($this->benchmark_results['data_processing'])) {
            $proc = $this->benchmark_results['data_processing'];
            echo "<h3>⚙️ Data Processing</h3>\n";
            echo "<table style='border-collapse: collapse; width: 100%;'>\n";
            echo "<tr><th style='text-align: left; padding: 8px; border: 1px solid #ddd;'>Metric</th><th style='text-align: right; padding: 8px; border: 1px solid #ddd;'>Value</th></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Dataset Size</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($proc['dataset_size']) . " items</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Regular Processing</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($proc['regular_items_per_sec'], 2) . " items/sec</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Batch Processing</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($proc['batch_items_per_sec'], 2) . " items/sec</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Batch Performance Ratio</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($proc['batch_vs_regular_ratio'], 2) . "x</td></tr>\n";
            echo "</table>\n";
        }
        
        // File Operations Results
        if (isset($this->benchmark_results['file_operations'])) {
            $file = $this->benchmark_results['file_operations'];
            echo "<h3>📁 File Operations</h3>\n";
            echo "<table style='border-collapse: collapse; width: 100%;'>\n";
            echo "<tr><th style='text-align: left; padding: 8px; border: 1px solid #ddd;'>Metric</th><th style='text-align: right; padding: 8px; border: 1px solid #ddd;'>Value</th></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Small Data Throughput</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . size_format($file['small_data_throughput']) . "/sec</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>Large Data Throughput</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . size_format($file['large_data_throughput']) . "/sec</td></tr>\n";
            echo "<tr><td style='padding: 8px; border: 1px solid #ddd;'>CSV Export Speed</td><td style='text-align: right; padding: 8px; border: 1px solid #ddd;'>" . number_format($file['csv_records_per_sec'], 2) . " records/sec</td></tr>\n";
            echo "</table>\n";
        }
        
        echo "</div>\n";
        
        echo "<h3 style='color: green;'>🚀 Performance Optimization Summary</h3>\n";
        echo "<ul>\n";
        echo "<li><strong>Caching System:</strong> Implemented with high-performance read/write operations</li>\n";
        echo "<li><strong>Database Queries:</strong> Optimized with automatic LIMIT injection and cursor pagination</li>\n";
        echo "<li><strong>Memory Management:</strong> Advanced tracking with minimal overhead</li>\n";
        echo "<li><strong>Data Processing:</strong> Batch processing for large datasets with memory management</li>\n";
        echo "<li><strong>File Operations:</strong> Streaming support for large files and optimized CSV exports</li>\n";
        echo "</ul>\n";
        
        // Performance recommendations
        echo "<h3>📊 Performance Recommendations</h3>\n";
        echo "<ul>\n";
        
        if (isset($this->benchmark_results['caching']['cache_hit_rate']) && $this->benchmark_results['caching']['cache_hit_rate'] == 100) {
            echo "<li><strong style='color: green;'>✓ Cache Performance:</strong> Excellent hit rate achieved</li>\n";
        }
        
        if (isset($this->benchmark_results['database']['improvement_ratio']) && $this->benchmark_results['database']['improvement_ratio'] > 1) {
            echo "<li><strong style='color: green;'>✓ Database Performance:</strong> Query optimization providing " . round($this->benchmark_results['database']['improvement_ratio'], 1) . "x improvement</li>\n";
        }
        
        if (isset($this->benchmark_results['memory']['tracking_overhead']) && $this->benchmark_results['memory']['tracking_overhead'] < 10) {
            echo "<li><strong style='color: green;'>✓ Memory Tracking:</strong> Low overhead (" . round($this->benchmark_results['memory']['tracking_overhead'], 1) . "%) memory tracking</li>\n";
        }
        
        echo "</ul>\n";
    }
}

// Only run if accessed directly
if (basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    // Set content type for browser output
    if (!headers_sent()) {
        header('Content-Type: text/html; charset=utf-8');
    }
    
    echo "<!DOCTYPE html>\n";
    echo "<html><head><title>EDM E-Fatura Performance Benchmark</title></head><body>\n";
    
    try {
        $benchmark = new EDM_Efatura_Premium_Benchmark(50); // Run with 50 iterations for faster testing
        $benchmark->run_benchmarks();
    } catch (Exception $e) {
        echo "<h2 style='color: red;'>Fatal Error</h2>\n";
        echo "<p>Benchmark failed to run: " . htmlspecialchars($e->getMessage()) . "</p>\n";
    }
    
    echo "</body></html>\n";
} else {
    return 'EDM_Efatura_Premium_Benchmark';
}